package executing;


import java.io.Serializable;

/**
 * This abstract class defines the services that should be implemented in a
 * class that will be part of the execution layer on the Mapper side.
 * 
 * @author Majd Kokaly
 */
public abstract class MapperSideExecuter implements Serializable {

	/**
	 * The serialVersionUID is used to universally identify this version of this class. 
	 */
	private static final long serialVersionUID = 395486745983649674L;
	
	/**
	 * This constant is returned when a method is not implemented
	 */
	public final static int NOT_IMPLEMENTED = -10;
	
	/**
	 * In this method the executer is set up.
	 */
	public abstract void startExecuter();
	
	/**
	 * This method is used by the Mapper to submit loop jobs
	 * 
	 * @param hostName
	 * @param serverID
	 * @param jobID
	 * @param executionTime
	 * @param ratio
	 * @return a number. Depends on the implementation.
	 */
	public abstract long submitLoopJob(String hostName, int serverID, long jobID, long executionTime,
			double ratio);
	
	
	/**
	 * This method stops the job with id "id" on hostname "hostname"
	 * 
	 * @param id is job id
	 * @param hostname Is the hostname of the server executing that job
	 * @return NOT_IMPLEMENTED if not implemented
	 */
	public abstract long stopJob(long id, String hostname); 
	
	/**
	 * This method deletes the job with id "id" on host name "hostname".
	 * 
	 * @param id Is job id
	 * @param hostname Is the hostname of the server executing that job
	 * @return NOT_IMPLEMENTED if not implemented
	 */
	public abstract long deleteJob(long id, String hostname);
	
	/**
	 * 
	 * @param id Is job id
	 * @param hostname Is the hostname of the server executing that job
	 * @return the percentage done from job with job id "id"
	 */
	public abstract double getPercentageDone(long id, String hostname);
	

	/**
	 * @param id Is job id
	 * @param hostname Is the hostname of the server executing that job.
	 * @return the date when the job with id "id" has started.
	 */
	public abstract String getDateStarted(long id, String hostname);

	/**
	 * @param id Is job id
	 * @param hostname Is the hostname of the server executing that job
	 * @return the date when the job with id "id" has stopped.
	 */
	public abstract String getDateStopped(long id, String hostname);
	
	
	/**
	 * @param id Is job id
	 * @param hostname Is the hostname of the server executing that job
	 * @return the status of the job with id "id" (e.g. Running, Done)
	 */
	public abstract String getJobStatus(long id, String hostname);
	

}
