package interfacing;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import javax.swing.*;

import mapping.Mapper;

/**
 * This JPanel with an object of ServersProcessingRatesTableModelListener are
 * the controller and the view of the Processing Rates objects of Server objects
 */
public class ServersProcessingRatesJPanel extends JPanel {

	/**
	 * The serialVersionUID is used to universally identify this version of this
	 * class.
	 */
	private static final long serialVersionUID = -1314942526341209428L;

	/** Names of the columns in jTable (Attributes edited) */
	private String[] columnNames = { "Job Class ID", "Real Rate", "Assumed Rate" };

	/** The MainFrame object */
	private MainFrame frame;

	/** The server id */
	private int serverID;

	/** The view component of the table. */
	private JTable jTable;

	/** The tableModel object of JTable */
	private ProcessingRatesTableModel tableModel;

	/** The listener of the tableModel */
	private ServersProcessingRatesTableModelListener serversProcessingRatesTableModelListener;

	/** The scroll pane that the table is put in. */
	private JScrollPane jTableScrollPane;

	public Mapper getMapper() {
		return frame.getMapper();
	}

	public int getServerID() {
		return serverID;
	}

	public void setServerId(int serverID) {
		this.serverID = serverID;
	}

	public ServersProcessingRatesJPanel() {
		super();
	}

	/**
	 * Default Constructor
	 * 
	 * @param frame
	 *            the MainFrame containing the JPanel
	 * @param id
	 *            the id of the Server object that this JPanel will show the
	 *            processing rates of.
	 */
	public ServersProcessingRatesJPanel(MainFrame frame, int id) {
		// Setting the job classes table
		this.frame = frame;
		this.setServerId(id);

		// Init the GUI components
		
		// left side of of splitPane
		jTable = new JTable(); // jTable component that shows the table
		jTableScrollPane = new JScrollPane(); // The scroll pane where the
												// jTable lives.
		this.setLayout(new BorderLayout());

		jTableScrollPane.setViewportView(jTable);
		// scrollPane.setViewportView(jTableScrollPane);

		this.add(jTableScrollPane);

		// Registering the Listener which responsible for validating data and
		// send it the view
		serversProcessingRatesTableModelListener = new ServersProcessingRatesTableModelListener(this.getMapper(), id);

		jTable.setGridColor(Color.BLACK);

		this.fillJTable();

		int h = (1 + jTable.getModel().getRowCount()) * jTable.getRowHeight();
		this.jTable.setPreferredSize(new Dimension(200, h));
		this.jTableScrollPane.setPreferredSize(new Dimension(200, h));
		this.setPreferredSize(new Dimension(200, h));
	}

	/**
	 * This method fills the table model (tableModel) of the corresponding data
	 * from the Server object that this JPanel is the view for. The jTable
	 * object is changed correspondingly.
	 */
	public void fillJTable() {
		tableModel = new ProcessingRatesTableModel(columnNames, 0);
		jTable.setModel(tableModel);
		String realRateString = null;
		String assumedRateString = null;

		for (int i = 0; i < this.getMapper().getClassesTable().size(); i++) {

			try {
				realRateString = this.getMapper().getRealRate(this.getServerID(), i + 1) + "";
				assumedRateString = this.getMapper().getAssumedRate(this.getServerID(), i + 1) + "";
			} catch (NullPointerException ex) {
				realRateString = "NaN";
				assumedRateString = "NaN";
			}

			String[] s = { (i + 1) + "", realRateString, assumedRateString, };
			tableModel.addRow(s);
		}
		tableModel.addTableModelListener(serversProcessingRatesTableModelListener);
	}

	/**
	 * This method disables input from user.
	 */
	public void disableInput() {
		tableModel.setEditable(false);
	}

	/**
	 * This method enables input from user.
	 */
	public void enableInput() {
		tableModel.setEditable(true);
	}

	public void printModel(OurDefaultTableModel model) {
		for (int i = 0; i < model.getRowCount(); i++) {
			for (int j = 0; j < model.getColumnCount(); j++) {
				System.out.print(model.getValueAt(i, j) + " ");
			}
			System.out.println();
		}
	}

}
