package interfacing;

import java.awt.Color;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.DateFormat;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;

//import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * This JPanel contains a table of statistics about general information about
 * the System and the current running test. The information include "Start
 * Time", "Time Units Passed", "System Time Units in Minutes", "Waiting Time",
 * "Waiting Time-T/O", "Communication Delay", "Communication Delay-T/O",
 * "Response Time" and "Response Time-T/O".
 */
public class SystemLevelStatisticsJPanel extends JPanel {
	/**
	 * The serialVersionUID is used to universally identify this version of this
	 * class.
	 */
	private static final long serialVersionUID = 3704614389093084232L;

	/**
	 * The lables of the stats showed by this JPanel
	 */
	private String[] headers = { "Start Time", "Time Units Passed", "System Time Units in Minutes", "Waiting Time",
			"Waiting Time-T/O", "Communication Delay", "Communication Delay-T/O", "Response Time", "Response Time-T/O" };

	/**
	 * An array of tips about statistics showed.
	 */
	private String[] tooltips = {
			"The time when the current test has started.",
			"The time passed since the start time of the test expressed in Time Units.",
			"One Time Unit is worth this value.",
			"Mean of all the differences between Time Submitted and Time Done for all jobs execluding timed out jobs.",
			"Mean of all the differences between Time Submitted and Time Done for all jobs including timed out jobs.",
			"Mean of all the differences between Time Submitted and Time Sent for all jobs execluding timed out jobs..",
			"Mean of all the differences between Time Submitted and Time Sent for all jobs including timed out jobs.",
			"Mean of all the differences between Time Sent and Time Started for all jobs execluding timed out jobs..",
			"Mean of all the differences between Time Sent and Time Started for all jobs including timed out jobs..", };

	/**
	 * The MainFrame object containing this JPanel
	 */
	private MainFrame frame;

	/**
	 * Any array of labels labeling the statistics
	 */
	JLabel[] labels;

	/**
	 * This button updates the statistics showed
	 */
	private JButton buttonUpdate = new JButton("Update");

	/**
	 * The default constructor is initialize and lay out the GUI objects
	 * 
	 * @param frame to set the MainFrame field.
	 */
	public SystemLevelStatisticsJPanel(MainFrame frame) {
		this.frame = frame;
		this.setLayout(new GridBagLayout());
		GridBagConstraints gbConstraints = new GridBagConstraints();
		gbConstraints.anchor = GridBagConstraints.WEST;

		labels = new JLabel[headers.length];

		int i;
		for (i = 0; i < headers.length; i++) {
			gbConstraints.gridy = i;
			gbConstraints.gridx = 0;
			JLabel label = new JLabel(headers[i] + ": ");
			label.setForeground(Color.GRAY);
			label.setToolTipText(tooltips[i]);
			label.setAlignmentX(JLabel.LEFT);
			label.setHorizontalTextPosition(JLabel.LEFT);
			label.setHorizontalAlignment(JLabel.LEFT);
			// label.setBorder(BorderFactory.createBevelBorder(0 ) );
			this.add(label, gbConstraints);

			gbConstraints.gridx = 1;
			labels[i] = new JLabel("Can't get it now");
			labels[i].setToolTipText(tooltips[i]);
			this.add(labels[i], gbConstraints);
		}

		gbConstraints.gridy = i;
		gbConstraints.gridx = 0;
		gbConstraints.gridwidth = 2;
		this.add(this.buttonUpdate, gbConstraints);

		gbConstraints.weighty = 1.0;
		gbConstraints.fill = GridBagConstraints.BOTH;
		gbConstraints.gridy = i + 1;
		this.add(javax.swing.Box.createVerticalGlue(), gbConstraints);

		this.buttonUpdate.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				refresh();
			}
		});
	}
	
	
	/**
	 * This method is invoked when the update button is pushed.
	 * It recalculate the stats and show them.
	 */
	public void refresh() {
		if (frame.getMapper().getStartTime() == null)
			return;

		DecimalFormat decimalFomatter = new DecimalFormat();
		decimalFomatter.setMaximumFractionDigits(3);

		DateFormat dateFormatter = new SimpleDateFormat("MMM, dd HH:mm:ss");

		labels[0].setText(dateFormatter.format(frame.getMapper().getStartTime().getTime()));
		labels[1].setText(decimalFomatter.format(frame.getMapper().getTimeElapsedInTimeUnits()) + " Time Unit(s)");
		labels[2].setText(decimalFomatter.format(frame.getMapper().getTimeUnitInMinutes()) + " minute(s)");
		labels[3].setText(decimalFomatter.format(this.frame.getMapper().getJobsTable()
				.getSchedulingDelayInMilliSeconds() / 1000.0)
				+ " seconds");
		labels[4].setText(decimalFomatter.format(this.frame.getMapper().getJobsTable()
				.getSchedulingDelayInMilliSecondsIncludingTimedOut() / 1000.0)
				+ " seconds");
		labels[5].setText(decimalFomatter
				.format(this.frame.getMapper().getJobsTable().getCommunicationDelayInSeconds())
				+ " seconds");
		labels[6].setText(decimalFomatter.format(this.frame.getMapper().getJobsTable()
				.getCommunicationDelayInSecondsIncludingTimedOut())
				+ " seconds");
		labels[7].setText(decimalFomatter.format(this.frame.getMapper().getJobsTable().getResponseTimeInSeconds())
				+ " seconds");
		labels[8].setText(decimalFomatter.format(this.frame.getMapper().getJobsTable()
				.getResponseTimeInSecondsIncludingTimedOut())
				+ " seconds");
	}
}
