package pulling.executing;

import java.io.Serializable;

/**
 * This abstract class defines the services that should be implemented in a
 * class that will be part of the execution layer on the Server side.
 * 
 * @author Majd Kokaly
 */
public abstract class ServerSideExecuter implements Serializable {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;
	/**
	 * This constant is returned when a method is not implemented
	 */
	public final static int NOT_IMPLEMENTED = -10;

	/**
	 * In this method the executer is set up.
	 */
	public abstract void startExecuter();
	
	/**
	 * In this method the executer is stopped and streams are closed
	 */
	public abstract void stopExecuter();
	
	/**
	 * This method sends the availability to the executer
	 * @param avail The availability of the server
	 */
	public abstract void setAvailability(double avail);
	
	/**
	 * This method is used by the Mapper to submit loop jobs
	 * 
	 * @param hostName The hostname of the Server object to send the job to.
	 * @param password The password of the Server object to send the job to.
	 * @param workingDirectory
	 * @param iter The executionTime of the job.
	 * @param ratio The ratio of the jobs.
	 * @return a number. It depends on the implementation.
	 */
	public abstract long submitLoopJob(String hostName, String password,
			String workingDirectory, long iter, double ratio);

	/**
	 * This method stops the job with id "id" on hostname "hostname"\
	 * 
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job
	 * @param password
	 * @return NOT_IMPLEMENTED if not implemented
	 */
	public abstract long stopJob(long id, String hostname, String password);
	
	/**
	 * This method deletes the job with id "id" on hostname "hostname"\
	 * 
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job
	 * @param password
	 * @return NOT_IMPLEMENTED if not implemented
	 */
	public abstract long deleteJob(long id, String hostname, String password);
	
	/**
	 * 
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job
	 * @param password
	 * @return the percentage done from job with job id "id"
	 */
	public abstract double getPercentageDone(long id, String hostname,
			String password);

	
	
	
	/**
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job.
	 * @param password
	 * @return the date when the job with id "id" has submitted to the execution layer.
	 */
	public abstract double getDateSubmitted(long id, String hostname,
			String password);

	/**
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job.
	 * @param password
	 * @return the date when the job with id "id" has started.
	 */
	public abstract String getDateStarted(long id, String hostname, String password);

	/**
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job
	 * @param password
	 * @return the date when the job with id "id" has stopped.
	 */
	public abstract String getDateStopped(long id, String hostname, String password);
	
	
	/**
	 * @param id is job id
	 * @param hostname is the hostname of the server executing that job
	 * @param password
	 * @return the status of the job with id "id" (e.g. Runnung, Done)
	 */
	public abstract String getJobStatus(long id, String hostname, String password);


}
