/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			   *
 *	  Minimum priority queue (binary heap implementation) Abstract Data Type.  *
 *																			   *
 *								  Source file								   *
 *																			   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include "owe_header.h"


unsigned char incidence_sorted_orbit_priority_queue_initalize(incidence_sorted_orbit_priority_queue_ptr queue,unsigned long int max_size) { 

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_initalize
	 *  -----------------------------------------------
	 *  Parameters:
	 *		- queue: pointer to the priority queue to initialize.
	 *		- max_size: maximum number of elements the priority queue will have.
	 *
	 *  Return values:
	 *		- If there is memory available for the priority queue:
	 *           The corresponding memory is allocated, the max_size of the priority queue is set to
	 *			 its corresponding value (parameter max_size), the size (number of elements in the
	 *			 queue) is set to 0 and the returned value is 1.
	 *
	 *      - If there is not enough available memory for the priority queue:
	 *			 No memory is allocated and the value returned is 0.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Allocate memory for the elements of the queue. If no memory is available, return 0. */
	if ((queue->elements = (orbit_ptr*)malloc(sizeof(orbit_tree_node_ptr)*(max_size+1)))==0) return 0;
	
	/* The maximum size of the queue is initialized according to the max_size parameter */
	queue->max_elems = max_size;

	/* The size (number of elements in the queue) is set to 0. */
	queue->size = 0; 

	/* The value returned is 1. */
	return(1);
}



void incidence_sorted_orbit_priority_queue_finalize(incidence_sorted_orbit_priority_queue_ptr queue) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_finalize
	 *  ----------------------------------------------
	 *  Parameters:
	 *		- Pointer to the priority queue to finalize.
	 *
	 *  The memory allocated by the queue is freed.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* The maximum number of elements of the queue is set to 0. */
	queue->max_elems = 0;

	/* The size of the queue is set to 0. */
	queue->size = 0;

	/* The memory taken by the elements of the queue is freed. */
	free(queue->elements);
}



orbit_ptr incidence_sorted_orbit_priority_queue_insert(incidence_sorted_orbit_priority_queue_ptr queue, orbit_ptr orbit) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_insert
	 *  --------------------------------------------
	 *  Parameters:
	 *		- queue: pointer to the priority queue on which the operation will be performed.
	 *		- orbit_ptr: pointer to the orbit to be inserted into the priority queue.
	 *
	 *  Return values:
	 *		- If there is space available in the priority queue for the new orbit:
	 *           The orbit is inserted into the priority queue and the value returned is a pointer
	 *			 to the newly inserted orbit (same as the input parameter).
	 *
	 *      - If there is not enough space available in the priority queue for the new orbit:
	 *			 the function terminates the program.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned long i;	/* This variable */

	/* If the heap already has the maximum number of elements the function returns 0. */
	if (queue->size >= queue->max_elems) owe_error_handler_raise_error("incidence sorted orbit priority queue overflow", 1);

	/* Increase the heap size to acomodate the new element, and set its inital position to be the last one in the heap */
	i = ++(queue->size);

	/* Traverse the path from the leaf to the root to find the a proper place for the new element */
	while (i > 1 && (orbit->incidence < queue->elements[incidence_sorted_orbit_priority_queue_PARENT(i)]->incidence)) {
		queue->elements[i] = queue->elements[incidence_sorted_orbit_priority_queue_PARENT(i)];
		i = incidence_sorted_orbit_priority_queue_PARENT(i);
	}

	/* insert the element at the position that was determined and return it*/
	return(queue->elements[i] = orbit);
}



orbit_ptr incidence_sorted_orbit_priority_queue_extract_min(incidence_sorted_orbit_priority_queue_ptr queue) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_extract_min
	 *  -------------------------------------------------
	 *  Parameters:
	 *		- queue: ointer to the priority queue on which the operation will be performed.
	 *
	 *  Return values:
	 *		- If there is at least one element in the priority queue:
	 *           The value returned is a pointer to the orbit whose incidence is the smalles in the
	 *           priority queue (note there could be more than one orbit with the same incidence
	 *           and no alternative sorting rule is applied in such case).
	 *
	 *      - If the priority queue is empty
	 *			  The value returned is 0.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_ptr min;	/* This variable will contain the orbit in the queue with the smallest incidence. */
	if (queue->size >= 1) {
		/* The element in the first position of the queue is retrieved */
		min = queue->elements[1];
		
		/* The element in the first position of the queue is then swapped. */
		queue->elements[1] = queue->elements[(queue->size)--];

		/* The incidence_sorted_orbit_priority_queue_heapify function is called to maintain the heap. */
		incidence_sorted_orbit_priority_queue_heapify(queue);
		}
	else return 0;
	return min;
}



void incidence_sorted_orbit_priority_queue_print_heap(incidence_sorted_orbit_priority_queue_ptr queue) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_print_heap
	 *  ------------------------------------------------
	 *  Parameters:
	 *		- Pointer to the priority queue to print.
	 *
	 *	This function prints the contents of the incidence soerted orbit priority queue.
	 *	The purpose of this function is for debugging.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned long int i;								/* Auxiliary variable used as a counter for extracting the elements of the queue. */
	incidence_sorted_orbit_priority_queue temp_queue;	/* Auxiliary queue used for printing. It is necesary to use an auxiliary queue in
														order to avoid modifying the contents (emptying) of the queue to me printed. */
	orbit_ptr temp_orbit;								/* Auxiliary pointer used for printing orbits. */
	
	/* The auxiliary queue is initialized */
	incidence_sorted_orbit_priority_queue_initalize(&temp_queue,queue->size);
	
	/* The contents of the queue to be printed is copied into the auxiliary queue. */
	memcpy(temp_queue.elements,queue->elements,sizeof(orbit_tree_node_ptr)*(queue->size+1));
	
	/* The size of the auxiliary queue is set accordinly. */
	temp_queue.size=queue->size;

	/* Print contents header. */
	printf("\n Incidence  Orbit canonical representative\n -----------------------------------------\n");
	
	/* Extracy all elements from the queue using the incidence_sorted_orbit_priority_queue_extract_min and print them iteratively. */
	for (i=1;i<=queue->size;i++)
	{
		/* A pointer to the orbit in the first position of the queue is obtained. */
		temp_orbit = incidence_sorted_orbit_priority_queue_extract_min(&temp_queue);
		
		/* The incidence of the orbit is printed. */
		printf(" %9lu  ",temp_orbit->incidence);

		/* The coordinate of the orbit is printed. */
		orbit_print_coordinate(temp_orbit->canonical_representative); printf("\n");
	}

	/* A break line is printed after listing the contents of the queue. */
	printf("\n");

	/* The auxiliary queue is finalized. */
	incidence_sorted_orbit_priority_queue_finalize(&temp_queue);

}



static void incidence_sorted_orbit_priority_queue_heapify(incidence_sorted_orbit_priority_queue_ptr queue) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  incidence_sorted_orbit_priority_queue_heapify
	 *  ---------------------------------------------
	 *  Parameters:
	 *		- incidence_sorted_orbit_priority_queue_ptr: pointer to the priority queue to be heapify.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned long int l, r, smallest, i = 1; /* These variables are used to store the positions of the elements in the heap
											being 'i' the current element, 'l' the element on its left, 'r' the element on
											its	right, and 'smallest' the smallest element found. */
	
	orbit_ptr tmp;	/* This variable is used to perform swapping of elements in the queue. */
	
	unsigned char procede = 1; /* This variable is used to control the iteration over the heap structure. */
  
	while (procede) {

		l = incidence_sorted_orbit_priority_queue_LEFT(i);	/* Obtain in variable 'l' the position of the element on the left. */
		r = incidence_sorted_orbit_priority_queue_RIGHT(i);	/* Obtain in variable 'l' the position of the element on the right. */

		/* check the left child */
		smallest = ((l <= queue->size && (queue->elements[l]->incidence < queue->elements[i]->incidence )) ? l : i);

		/* check the right child */
		if (r <= queue->size && (queue->elements[r]->incidence < queue->elements[smallest]->incidence)) smallest = r;

		if (smallest != i) { 
			/* swap nodes largest and i */
			tmp = queue->elements[i] ; queue->elements[i] = queue->elements[smallest] ; queue->elements[smallest] = tmp;
			i = smallest;
		}
		else
			procede=0;
	}
}
