/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *							  OWE Master process								   *
 *																				   *
 *								  Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "owe_header.h"

static int slave_processor_count;
static unsigned long int owe_master_process_orbits_computed = 0;
static unsigned long int owe_master_process_orbits_skipped = 0;
static orbit_ptr *processor_orbit_assignment;		/* This variable is used (as an array) to track what orbit is being computed by each processor. */

void owe_master_process_initialize(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_initialize
	*  -----------------------------
	*  
	*	This function prints a header for the computation (into the output stream of the master process)
	*	and it loads the input parameters from the input parameter file by using the input_parameter_reader_read_input.
	*	function. It then makes sure there are at least two processors to perform the computation. In case
	*	only one processor is detected, a fatal error is raised.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Obtain the total number of processors and initialize the processor queue. */
	MPI_Comm_size(MPI_COMM_WORLD,&slave_processor_count);

	if(slave_processor_count == 1)
		owe_error_handler_raise_error("at least two processors are required for computation.", 1);
	else
		processor_queue_initialize(--slave_processor_count);

	/* Allocate memory for the processor/orbit assignment. */
	processor_orbit_assignment = (orbit_ptr*) malloc(sizeof(coordinate)*(slave_processor_count+1));

	/* Print the header for the computation. */
	printf("Orbitwise Enumeration\n=====================\n\nProcessors used in computation = %d\n\n", slave_processor_count + 1);

	/* Obtain input parameters from input parameter file and initialize owe_setting variables, orbit databank and canon. */
	input_parameter_reader_read_input();

}



void owe_master_process_finalize(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_finalize
	*  ---------------------------
	*  
	*	This function finalizes the orbit databank and the metric polytope.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Finalize orbit databank. */
	orbit_databank_finalize();
	
	/* Finalize metric polytope. */
	metric_polytope_finalize();

}



void owe_master_process_execute(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_execute
	*  --------------------------
	*  
	*	This function performs the set of tasks to be executed on the master process, including:
	*		- Program initialization.
	*		- Computation time monitoring.
	*		- Broadcast computation parameters to slave processes.
	*		- Process orbits within computation range (dispatch them to slave processes)
	*		- Print process report and orbit databank contents.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Initialize the master process. */
	owe_master_process_initialize();

	/* Start the timer to keep track of computation time. */
	owe_master_process_computation_timer();

	/* Boradcast to all slaves the parameters for the computation. */
	owe_master_process_broadcast_parameters();

	/* Process orbits. */
	owe_master_process_process_orbits();

	/* Finalize slave processes. */
	owe_master_process_finalize_slave_processes();

	/* Putput computed result. */
	orbit_databank_print_orbits();

	/* Stop the timer and display the computation time. */
	owe_master_process_computation_timer();

	/* Finalize the master process. */
	owe_master_process_finalize();

}



void owe_master_process_process_orbits(void) {
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_process_orbits
	*  ---------------------------------
	*  
	*	This function dispatches orbits to the slave processes to be computed. At the same time, it
	*	retreives newly found orbits from the slave processes updating the contents of the orbit
	*	databank.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	coordinate orbit_canonical_representative;	/* This variable is used to send and recieve the canonical
													representatives of the orbits to be computed */

	MPI_Status status;							/* This variable is used to retrieve the status values returned
													by the MPI function calls. */
	time_t timer;								/* This variable is used to calculate the time lapse between each status report. */

	unsigned long int initial_orbit_set_count = orbit_databank_get_number_of_orbits_known(); /* This variable is used to calculate the
																								number of new orbits found in the computation. */
	/* Verify there is at least one orbit in the orbit databank queue. */
	if(!orbit_databank_get_number_of_orbits_to_compute()) owe_error_handler_raise_error("initial orbit set within incidence range is empty", 1);

	/* Allocate memory for the orbit canonical representative. */
	if(!(orbit_canonical_representative = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))) owe_error_handler_raise_error("memory allocation error (coordinate variable : owe_master_process_process_orbits function : owe_master_process ADT)", 1);

	/* While there are processors available and orbits to be processed in the queue, process them. */
	while(processor_queue_get_number_of_processors_in_queue() && orbit_databank_get_number_of_orbits_to_compute())
		owe_master_process_dispatch_orbit(0);
	
	/* Print computation start header. */
	 printf("Computation start.\n\n Orbits:\n\n     Known/found          Queued       Computing        Computed         Skipped    elapsed time\n --------------- --------------- --------------- --------------- --------------- ---------------\n");

	 /* Start the timer. */
	time(&timer);

	/* After all initial orbits have been sent to be computed, inform the status. */
	timer = owe_master_process_inform_status(timer, 1);

	do {
		
		/* Recieve a message from one of the slave, indicating that a new orbit has been found or that the slave process has finished computing the orbit. */
		MPI_Recv(orbit_canonical_representative, owe_setting.metric_polytope.dimension + 1, MPI_UNSIGNED, MPI_ANY_SOURCE, MPI_ANY_TAG, MPI_COMM_WORLD, &status);

		switch(status.MPI_TAG) {
			/* If the tag recieved indicates a "potentially" new orbit has been found, insert it
			into the orbit databank. */
			case OWE_MPI_TAG_NEW_ORBIT_FOUND:
				if(orbit_databank_insert_orbit(orbit_canonical_representative, 0, 0)) {
					/* If the orbit was endeed new and there are processors available, dispatch it. */
					if(processor_queue_get_number_of_processors_in_queue() && orbit_databank_get_number_of_orbits_to_compute())
						owe_master_process_dispatch_orbit(0);
					timer = owe_master_process_inform_status(timer, 0);
				}
				break;


			/* If the tag recieved indicates a slave processor finished computing an orbit. */
			case OWE_MPI_TAG_SLAVE_FINISHED_COMPUTING_ORBIT:
				/* If there is an orbit to be computed in the orbit databank, dispatch it (since
				now we know there is at least one processor available). */
				owe_master_process_orbits_computed++;
				if(orbit_databank_get_number_of_orbits_to_compute())
					owe_master_process_dispatch_orbit(status.MPI_SOURCE);
				else
					/* If there aren't orbits to be computed, queue the processor. */
					processor_queue_insert_processor(status.MPI_SOURCE);
				timer = owe_master_process_inform_status(timer, 0);
				break;


			/* If the tag received indicates a slave processor skipped an orbit (this happens when the orbit adjacency is outside range). */
			case OWE_MPI_TAG_SLAVE_SKIPPED_ORBIT:
				processor_orbit_assignment[status.MPI_SOURCE]->skipped = 1;
				owe_master_process_orbits_skipped++;
				/* If there is an orbit to be computed in the orbit databank, dispatch it (since
				now we know there is at least one processor available). */
				if(orbit_databank_get_number_of_orbits_to_compute())
					owe_master_process_dispatch_orbit(status.MPI_SOURCE);
				else
					/* If there aren't orbits to be computed, queue the processor. */
					processor_queue_insert_processor(status.MPI_SOURCE);
				timer = owe_master_process_inform_status(timer, 0);
				break;

		}


	} while(orbit_databank_get_number_of_orbits_to_compute() || processor_queue_get_number_of_processors_in_queue() < (unsigned) slave_processor_count);

	/* After computing all orbits, show the status one last time. */
	timer = owe_master_process_inform_status(timer, 1);

	/* Free the memory taken by the orbit canonical representative. */
	free(orbit_canonical_representative);

	/* Print computation complete header. */
	printf("\nComputation complete.\n\nOrbits skipped = %lu\nNew orbits found = %lu\nTotal orbits known = %lu\n\n", owe_master_process_orbits_skipped, orbit_databank_get_number_of_orbits_known() - initial_orbit_set_count, orbit_databank_get_number_of_orbits_known());
}




void owe_master_process_broadcast_parameters(void) {
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_broadcast_parameters
	*  ---------------------------------------
	*  
	*	This function broadcasts the computation parameters to the slave processes:
	*		- metric polytope.
	*		- Adjacency range (upper and lower bounds).
	*		- Double description time limit.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned int parameters[4];
	
	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_MASTER
	/*DEBUG*/ printf("\n\nSending broadcast for initialization");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

	/* Obtain the set of parameters to be sent to the slaves. */
	parameters[0] = owe_setting.metric_polytope.n;
	parameters[1] = owe_setting.adjacency_lower_bound;
	parameters[2] = owe_setting.adjacency_upper_bound;
	parameters[3] = owe_setting.double_description_time_limit;

	/* Broadcast to all slave processes parameters. */
	MPI_Bcast(&parameters, 4, MPI_UNSIGNED, 0, MPI_COMM_WORLD);

	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_MASTER
	/*DEBUG*/ printf("\n\nbroadcast sent");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

}



void owe_master_process_finalize_slave_processes(void) {
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_finalize_slave_processes
	*  -------------------------------------------
	*  
	*	This function finalizes the computation by sending a message to all slaves indicating
	*	no further orbits will be computed.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	int i = slave_processor_count;
	
	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_MASTER
	/*DEBUG*/ printf("\n\nFinalizing slave processes...");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

	do 
		MPI_Send(&i, 0, MPI_UNSIGNED, i, OWE_MPI_TAG_FINALIZE, MPI_COMM_WORLD);
	while(--i);

	/*DEBUG*/ #ifdef DEBUG_MPI_MESSAGES_MASTER
	/*DEBUG*/ printf("\n\nFinished finalizing slave processes");
	/*DEBUG*/ fflush(stdout);
	/*DEBUG*/ #endif

}



time_t owe_master_process_inform_status(time_t timer, unsigned char force_update) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_inform_status
	*  --------------------------------
	*  
	*	Parameters:
	*		- timer:	the number of seconds ellapsed since the last time the function was called.
	*		- force_update: 1 indicates the output will be updated no matter how much time has been
	*						ellapsed since the last update.
	*
	*
	*	This function checks if the number of seconds since the last time it printed the status is
	*	greater or equal than 600 (10 minutes). In such case, the status is updated and the output
	*	buffer flushed. If the input parameter "force_update"is 1, the status is updated even if the time ellapsed
	*	since the last status update is less than 600 seconds.
	*
	*	Return value:
	*		The new starting time for the timer.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	time_t timer_stop;
	time(&timer_stop);
	if(force_update || timer_stop - timer >= 600) {
		printf("      %10lu      %10lu      %10u      %10lu      %10lu        ", orbit_databank_get_number_of_orbits_known(), orbit_databank_get_number_of_orbits_to_compute(), slave_processor_count - processor_queue_get_number_of_processors_in_queue(), owe_master_process_orbits_computed, owe_master_process_orbits_skipped);
		owe_lib_print_time(timer_stop - timer); printf("\n");
		fflush(stdout);
		return(timer_stop);
	}

	return(timer);

}



void owe_master_process_computation_timer(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_computation_timer
	*  ------------------------------------
	*  
	*	This function keeps track of the time lapse between the first and second time it is called.
	*	This is, on the first time it is called, it starts the timer, and on the second time it is
	*	called, it stops the timer, calculates the ellapsed time and prints it on screen.
	*	The time is shown in seconds.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
	
	static time_t start_time = 0;
	time_t finish_time;
	
	if(!start_time)
		time(&start_time);
	else
	{
		time(&finish_time);
		printf("\n\nTotal computation time: ");
		owe_lib_print_time(finish_time - start_time); printf("\n");
	}

}

void owe_master_process_dispatch_orbit(int predefined_processor) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_master_process_dispatch_orbit
	*  ---------------------------------
	*  
	*	Parameters:
	*		- predefined processor (optional): integer number that identifies the slave processor to which
	*		  the orbit to be processed is dispatched. If the parameter is 0, the slave process will be
	*		  determined by the processor queue.
    *
	*	This function dispatches the next orbit to be computed to the next slave process available for computation.
	*	If the orbit to be computed is to be sent to a specific slave, then the optional input parameter
	*	(predefined_processor) has to indicate it.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_ptr orbit;
	int processor = (predefined_processor)?predefined_processor:processor_queue_get_next_available_processor();

	/* Obtain the next orbit to compute. */
	orbit = orbit_databank_get_next_orbit_to_compute();

	/* Keep track of the orbit to be computed and the processor. */
	processor_orbit_assignment[processor] = orbit;

	/* Send the mpi message to the slave to compute the orbit. */
	MPI_Send(orbit->canonical_representative, owe_setting.metric_polytope.dimension + 1, MPI_UNSIGNED, processor, OWE_MPI_TAG_PROCESS_ORBIT, MPI_COMM_WORLD);

}
