/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *								Orbit DataBank									   *
 *																				   *
 *								 Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "owe_header.h"

static orbit_tree orbit_databank_tree;				/* tree data structure where orbits will be stored. */
static incidence_sorted_orbit_priority_queue orbit_databank_queue;		/* priority queue data structure to get the next orbit to compute. */
static unsigned long int orbit_databank_precomputed_orbit_count;
static unsigned long int smallest_incidence_outside_range;

void orbit_databank_initialize(long unsigned int heap_size) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_initialize
	*  ---------------------
	*  Parameters:
	*		- heapsize: maximum number of elements the incidence sorted orbit priority queue will hold.
	*					This number should be at least as big as an estimate of the number of orbits to be found.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_tree_initialize(&orbit_databank_tree);
	incidence_sorted_orbit_priority_queue_initalize(&orbit_databank_queue,heap_size);
	smallest_incidence_outside_range = (owe_setting.metric_polytope.n*(owe_setting.metric_polytope.n-1)*(owe_setting.metric_polytope.n-2))/2;
	
}



void orbit_databank_finalize(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_finalize
	*  -----------------------
	*
	*	Description:
	*		Frees the memory taken by the orbit databank.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_tree_delete(&orbit_databank_tree);
	incidence_sorted_orbit_priority_queue_finalize(&orbit_databank_queue);
	orbit_databank_precomputed_orbit_count = 0;
}



unsigned char orbit_databank_insert_orbit(coordinate canonical_representative, unsigned long int orbit_incidence, unsigned char orbit_precomputed) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_insert_orbit
	*  ---------------------------
	*  Parameters:
	*		- canonical_representative: canonical representative of the orbit to be inserted into the orbit databank.
	*		- orbit_incidence: (optional) incidence of the orbit to be inserted. If the incidence of the orbit to be
	*							inserted is known, then it should be supplied in this parameter to avoid calculating
	*							it again. If the incidence value of the orbit is not yet known, then this parameter
	*							should be 0 and the orbit databank will calculate its incidence on insertion.
	*		- precomputed:	Indicates whether the orbit to be inserted is precomputed (value 1) or not (value 0). The
	*						effect of this parameter is that precomputed orbits are not queued to then be processed.
	*
	*	Return value:
	*		If the orbit was not previously in the orbit databank, the value returned is 1. If the orbit was already
	*		in the databank, the value returned is 0.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_ptr orbit;
	
	/* Change in the size of the tree is traced to determine whether an orbit was inserted. */
	const unsigned int current_orbit_tree_size = orbit_databank_tree.size;

	/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
	/*DEBUG*/ printf("Orbit databank attempting to insert (%s) orbit:\n\n ",(orbit_precomputed)?"precomputed":"not yet computed"); orbit_print_coordinate(canonical_representative);
	/*DEBUG*/ #endif

	/* Insert the orbit into the orbit tree. */
	orbit = orbit_tree_insert(&orbit_databank_tree, canonical_representative);

	if (current_orbit_tree_size < orbit_databank_tree.size) {
		
		/* If the orbit_incidence is 0, it means the incidence has not yet been calculated. */
		if(!orbit_incidence)
			orbit_incidence = orbit_databank_calculate_orbit_incidence(orbit);
		else
			orbit->incidence = orbit_incidence;

		/* Assign the corresponding value depending on whether the orbit is precomputed or not. */
		orbit->pre_computed = orbit_precomputed;

		/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
		/*DEBUG*/ printf("\n\n Orbit inserted succesfully (was not previously in orbit databank).\n Current orbits in orbit databank: %lu.\n",orbit_databank_tree.size);
		/*DEBUG*/ orbit_tree_print_canonical_representatives(&orbit_databank_tree);
		/*DEBUG*/ #endif

		/* Check if preloaded orbit is to be computed or precomputed. */
		if(!orbit_precomputed) {
			if(owe_setting.incidence_lower_bound <= orbit_incidence && orbit_incidence <= owe_setting.incidence_upper_bound) {
				incidence_sorted_orbit_priority_queue_insert(&orbit_databank_queue,orbit);
			
				/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
				/*DEBUG*/ printf("\n Inserted orbit incidence is in allowed range, queueing.\n\n Queue contents (orbits to be processed sorted by incidence)");
				/*DEBUG*/ incidence_sorted_orbit_priority_queue_print_heap(&orbit_databank_queue);
				/*DEBUG*/ #endif
			}
			else
			{
				/* Keep track of the smallest incidence outside range. */
				if(orbit->incidence < smallest_incidence_outside_range)
					smallest_incidence_outside_range = orbit->incidence;

				/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
				/*DEBUG*/ printf("\n Inserted orbit incidence outside allowed range, wont be queued.\n\n");
				/*DEBUG*/ #endif
			}
		}
		else
		{
			orbit_databank_precomputed_orbit_count++;
			/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
			/*DEBUG*/ printf("\n Inserted orbit is precomputed, wont be queued.\n\n");
			/*DEBUG*/ #endif
		}

		return(1);
	}
	else
	{
	/*DEBUG*/ #ifdef DEBUG_ORBIT_DATABANK
	/*DEBUG*/ printf("\n\n Orbit was already in databank.\n\n");
	/*DEBUG*/ #endif
	}
	return(0);
}



orbit_ptr orbit_databank_get_next_orbit_to_compute(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_next_orbit_to_compute
	*  ----------------------------------------
	*
	*	This function retrieves and returns the orbit in the databank that has been queued in the
	*	incidence sorted priority queue and has the smallest incidence (of the orbits currently queued).
	*	Note that this function works as the "pop" method of a stack, this is, it returns the next orbit in the
	*	queue and at the same time it eliminates it from the queue.
	*
	*	Return value:
	*		A pointer to the canonical representative of the next orbit to compute.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/*orbit_ptr next_orbit_to_process = incidence_sorted_orbit_priority_queue_extract_min(&orbit_databank_queue);
	return((next_orbit_to_process)?next_orbit_to_process->canonical_representative:0);
	*/

	return(incidence_sorted_orbit_priority_queue_extract_min(&orbit_databank_queue));
}



unsigned long int orbit_databank_get_number_of_orbits_to_compute(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_number_of_orbits_to_compute
	*  ----------------------------------------------
	*
	*	This function returns the number of orbits currently present in the incidence sorted orbit
	*	priority queue.
	*
	*	Return value:
	*		Number of orbits currently queued for computation.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	return (orbit_databank_queue.size);
}

void orbit_databank_print_orbits(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_print_orbits
	*  ---------------------------
	*
	*	This function prints the contents of the orbit databank, specifying the incidence of each
	*	orbit in it, and whether it is computed (incidence within range), precomputed (preloaded
	*	from the input parameter file), or not computed(incidence outside range).
	*	It also generates the next iteration input file generation through the next_iteration_input_generator_initialize
	*	and next_iteration_input_generator_finalize function calls. In between them (and for efficiency reasons)
	*	The orbits are printed in both, the standard output and the next iteration input file.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	FILE* next_iteration_input_file_ptr;	/* This file pointer will be used to write the orbits into the next iteration input file. */

	/* Initialize the next iteration input file. */
	next_iteration_input_file_ptr = next_iteration_input_generator_initialize();

	/* Print the orbit set on both, the standard output and the next iteration input file. */
	orbit_tree_print_orbits_lexicographically_ordered(&orbit_databank_tree, next_iteration_input_file_ptr);

	/* Finalize the next iteration input file. */
	next_iteration_input_generator_finalize();
}



long unsigned int orbit_databank_calculate_orbit_incidence(orbit_ptr orbit) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_calculate_orbit_incidence
	*  ---------------------------------
	*  Parameters:
	*		- orbit: pointer to the orbit whose incidence is to be calculated
	*
	*	This function stores the orbit incidence in the "incidence" attribute of the orbit structure.
	*
	*	Return value:
	*		- The incidence of the orbit (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	metric_polytope_inequality inequality_value = owe_setting.metric_polytope.inequalities;
	unsigned short int inequality_count = owe_setting.metric_polytope.inequality_count;
	unsigned short int dimension;
	signed int equality_value;
	coordinate canonical_representative_value;

	orbit->incidence = 0;

	/* For each inequality of the metric polytope*/
	while(inequality_count--) {

		/* Set the pointer value of the canonical representative. */
		canonical_representative_value = orbit->canonical_representative;
		
		/* Set the value of the equality to the product of the metric polytope inequality constant and the
		canonical representative denominator. */
		equality_value = *(inequality_value++) * *(canonical_representative_value+owe_setting.metric_polytope.dimension);
		
		/* Add to the equality value the product of each element of the canonical representative
		coordinate times the corresponding coheficient of the metric polytope inequality. */
		dimension = owe_setting.metric_polytope.dimension;
		while(dimension--) {
			equality_value += *(inequality_value++) * *(canonical_representative_value++);
		}

		/* If the equality is 0, increment incidence and update incident inequalities/non incident inequalities. */
		if(!equality_value) orbit->incidence++;
	}
	return(orbit->incidence);
}



unsigned long int orbit_databank_get_number_of_orbits_known(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_number_of_orbits_known
	*  -----------------------------------------
	*
	*	This function retrieves the total number of orbits known (present in the orbit databank).
	*
	*	Return value:
	*		- Number of orbits in orbit databank (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	return orbit_databank_tree.size;

}



unsigned long int orbit_databank_get_orbit_queue_size(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_orbit_queue_size
	*  -----------------------------------
	*
	*	This function retrieves the total number of orbits queued.
	*
	*	Return value:
	*		- Number of orbits queued for computation (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	return orbit_databank_queue.max_elems;

}



unsigned long int orbit_databank_get_precomputed_orbit_count(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_precomputed_orbit_count
	*  ------------------------------------------
	*
	*	This function retrieves the number of orbits precomputed.
	*
	*	Return value:
	*		- Number of orbits precomputed (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

return orbit_databank_precomputed_orbit_count;

}



unsigned long int orbit_databank_get_smallest_incidence_outside_range(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_databank_get_smallest_incidence_outside_range
	*  ---------------------------------------------------
	*
	*	This function retrieves, from the set of orbits present in the databank, the one
	*	not computed with the smallest incidence.
	*
	*	Return value:
	*		- Smallest incidence of orbits not computed (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	return(smallest_incidence_outside_range);
}
