/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *								Orbit Processor									   *
 *																				   *
 *								  Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "owe_header.h"

unsigned char orbit_processor_process_orbit(coordinate orbit_canonical_representative, slave_process_orbit_tree_ptr slave_process_slave_process_tree_ptr) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_processor_process_orbit
	*  -----------------------------
	*  Parameters:
	*		- orbit_canonical_representative:	canonical representative of the orbit to be computed.
  	*
	*  Return values:
	*		- 1 if the orbit is within adjacency range (in this case the orbit is computed).
	*		- 0 if the orbit is outside the adjacency range (in this case the orbit is not computed).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* CDDLib variables requires for the double description computation. */
	dd_MatrixPtr dd_inputMatrix, dd_outputMatrix;
	dd_PolyhedraPtr poly;
	dd_ErrorType err;

	unsigned long orbit_incidence;

	unsigned long orbit_adjacency;					/* adjacency of the orbit. */

	unsigned long orbit_computation_time;		/* time that the entire orbit computation takes. */

	unsigned long cdd_computation_time;			/* time that CDDLib took to process the orbit. */

	orbitwise_adjacency_tree orbit_orbitwise_adjacenct_tree;	/* red-black orbitwise adjacency tree that stores the adjacent orbits */

	metric_polytope_inequality *incidence_inequalities;		/* Set of pointers
															to the inequalities that contain the
															canonical representative of the orbit,
															followed by the inequalities that do not
															contain it. */

	metric_polytope_inequality *incidence_inequality;		/* Auxiliary pointer used to parse through the pointers of metric_polytope_inequality *incidence_inequalities. */
	
	metric_polytope_inequality inequality_value;	/* Pointer to a single value of an inequality. */
	
	coordinate vertex_coordinate;			/* Coordinate used to store the neighbour vertices of the orbit. These values
														will be obtained from calls to the  orbit_processor_get_neighbor_vertex
														function. Then it will be used to obtain the canonical representative
														of such vertices through the canon function. Finally it will be used
														for inserting the newly found canonical representatives of the neighbour
														vertices of the orbit into the orbitwise adjacency tree through the 
														orbitwise_adjacency_tree_insert function. */
														
	int* integer_ray;								/* Rays are being turned into integers and stored into a coordinate
													to then be passed as a parameter to orbit_processor_get_neighbor_orbit.*/

	int* integer_ray_denominators;				/* When turning rays into integer rays, its coordinate denominators have to be computed.
													These are calculated in the orbit_processor_enlarge_ray function. For efficiency, instead
													of having such function allocate and deallocate memory for the integer ray denominators
													each time it is called, these are allocated in orbit_processor_process_orbit and passed as
													a parameter to the function. Note  that these denominators are only used while enlarging
													a ray, once the ray enlargement is complete (and its elements are stored in the integer_ray
													variable, then integer_ray_denominators contains garbage (integer rays do NOT have denominators). */

	unsigned long int i, j;	/* Counters used for parsing the CDDLib input and output structures. */

	unsigned char vertex_present_in_output = 0;	/* Variable that checks whether the vertex is present in the output cone
													generated by CDDLib. */
	time_t orbit_timer_start, orbit_timer_stop, cdd_timer_start, cdd_timer_stop;	/* Variables used to calculate the time consumed by the entire function and CDDLib. */

	/* Track time of orbit computation (begin). */
	time(&orbit_timer_start);

	/* Allocate memory for the set of incident/non inident metric polytope inequalities. */
	if (!(incidence_inequalities = (metric_polytope_inequality*) malloc(sizeof(metric_polytope_inequality)*(owe_setting.metric_polytope.inequality_count)))) owe_error_handler_raise_error("memory allocation error (metric_polytope_inequality variable : incidence_inequalities variable : orbit_processor ADT)", 1);

	/* Compute the incidence of the orbit. */
	orbit_incidence = orbit_processor_calculate_orbit_incidence(orbit_canonical_representative, incidence_inequalities);

	/* Allocate memory for the canonical representative coordinate and the integer ray coordinate and denominator variables. */
	if(!(vertex_coordinate = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))) owe_error_handler_raise_error("memory allocation error (coordinate variable : orbit_processor_process_orbit function : orbit_processor ADT)", 1);
	if(!(integer_ray = (int*) malloc(sizeof(int)*owe_setting.metric_polytope.dimension))) owe_error_handler_raise_error("memory allocation error (coordinate variable : orbit_processor_process_orbit function : orbit_processor ADT)", 1);
	if(!(integer_ray_denominators = (int*) malloc(sizeof(int)*owe_setting.metric_polytope.dimension))) owe_error_handler_raise_error("memory allocation error (coordinate variable : orbit_processor_process_orbit function : orbit_processor ADT)", 1);

	/* Get the pointer to the set if inequalities incident to the orbit. */
	incidence_inequality = incidence_inequalities;

	/* Create the input matrix for CDDLib. */
	dd_inputMatrix = dd_CreateMatrix(orbit_incidence, owe_setting.metric_polytope.dimension+1);
	for(i=0; i < orbit_incidence; i++){
		inequality_value = *incidence_inequality++;
		/* The inequalities of the metric polytope are passed to CDD. */
		dd_set_si(dd_inputMatrix->matrix[i][0], *inequality_value++);
		
		for(j=1; j<=owe_setting.metric_polytope.dimension; j++){
			dd_set_si(dd_inputMatrix->matrix[i][j], *(inequality_value++));
		}
	}

	dd_inputMatrix->representation = Inequality;
   
	/*DEBUG*/ #ifdef DEBUG_CDDLIB
	/*DEBUG*/ printf("CDDLib call for computing orbit: ");
	/*DEBUG*/ orbit_print_coordinate(orbit_canonical_representative);
	/*DEBUG*/ printf("\n\nINPUT\n-----\n");
	/*DEBUG*/ dd_WriteMatrix(stdout,dd_inputMatrix);
	/*DEBUG*/ #endif

	/* Time track of CDDLib (begin). */
	time(&cdd_timer_start);

	/* CDDLib processing. */
	poly = timed_dd_DDMatrix2Poly(dd_inputMatrix, &err, owe_setting.double_description_time_limit, owe_setting.adjacency_upper_bound); if (err != NoError) owe_error_handler_raise_error("CDDLib error on dd_DDMatrix2Poly2 : orbit_processor_process_orbit : orbit_processor ADT", 1);
	
	/* Verify if the orbit was computed or skipped. */
	if(poly->skipped || (!poly->skipped && owe_setting.adjacency_lower_bound > (unsigned long int) poly->n)) {

		/* If the orbit was skipped, stop the orbit computation. All memory taken so far is freed and the value returned is 0. */
		dd_FreeMatrix(dd_inputMatrix);
		dd_FreePolyhedra(poly); 
		free(vertex_coordinate);
		free(integer_ray);
		free(incidence_inequalities);
		free(integer_ray_denominators);
		return(0);

  }




	/* Time track of CDDLib (end). */
	time(&cdd_timer_stop);

	/* Store the CDDLib time. */
	cdd_computation_time = cdd_timer_stop - cdd_timer_start;

	/* Output of CDDLib processing is passed to the output matrix. */
	dd_outputMatrix = dd_CopyGenerators(poly);
	

	/* Input matrix and intermediate polyhedra representation are freed. */
	dd_FreeMatrix(dd_inputMatrix);
	dd_FreePolyhedra(poly); 
		
	/*DEBUG*/ #ifdef DEBUG_CDDLIB
	/*DEBUG*/ printf("\nOUTPUT\n------\n");
	/*DEBUG*/ dd_WriteMatrix(stdout,dd_outputMatrix);
	/*DEBUG*/ #endif

	/* Initialize the orbit adjacency tree. */
	orbitwise_adjacency_tree_initialize(&orbit_orbitwise_adjacenct_tree);

	/* For each ray in the CDD output, the orbit_processor_get_neighbor_orbit function is called
		obtaining the canonical representative of the neighbor vertex. Then, this value is inserted
		into the orbitwise adjacency tree (which will make the necesary insertions in the orbit
		databank while keeping track of the number of repeated orbits found. */
	for(i=0;i< (unsigned) dd_outputMatrix->rowsize;i++) {
		if(!*dd_outputMatrix->matrix[i][0]) {

			/* Enlarge ray so that its values are integers. */
			orbit_processor_enlarge_ray(dd_outputMatrix->matrix[i]+1, integer_ray, integer_ray_denominators);

			/* Obtain neighbour vertex from current ray. */
			orbit_processor_get_neighbor_vertex(orbit_canonical_representative, orbit_incidence, incidence_inequalities, integer_ray, vertex_coordinate);
			
			/*DEBUG*/ #ifdef DEBUG_CANON
			/*DEBUG*/ printf("\nCanon input:  "); orbit_print_coordinate(vertex_coordinate);
			/*DEBUG*/ #endif
			
			/* Obtain the canonical representative of the neighbour vertex. */
			canon((int)*(vertex_coordinate+owe_setting.metric_polytope.dimension),(int*)vertex_coordinate);

			/*DEBUG*/ #ifdef DEBUG_CANON
			/*DEBUG*/ printf("\nCanon output: "); orbit_print_coordinate(vertex_coordinate); printf("\n\n");
			/*DEBUG*/ #endif

			/* Insert the canonical representative of the neighbour vertex into the orbitwise adjacency tree. */
			orbitwise_adjacency_tree_insert(&orbit_orbitwise_adjacenct_tree, vertex_coordinate, slave_process_slave_process_tree_ptr);
		
		}
		else
			vertex_present_in_output = 1;
	}

	orbit_adjacency = dd_outputMatrix->rowsize - vertex_present_in_output;
	
	/*DEBUG*/ #ifdef DEBUG_ORBIT_NEIGHBORHOOD_CALCULATION
	/*DEBUG*/ printf("Total rays processed: %u\n\n", orbit_adjacency);
	/*DEBUG*/ #endif
	
	/* Track time of orbit computation (begin). */
	time(&orbit_timer_stop);
	orbit_computation_time = orbit_timer_stop - orbit_timer_start;

	/* Print the result. */
	printf("- Orbit ---------------------------------------------------------\n\n Canonical representative:\n ");
	orbit_print_coordinate(orbit_canonical_representative);
	printf("\n\n Incidence:  %lu\n Adjacency:  %lu\n\n Orbitwise Adjacency:\n\n", orbit_incidence, orbit_adjacency);
	orbitwise_adjacency_tree_print_tree(&orbit_orbitwise_adjacenct_tree);
	printf("\n Double description time:       "); owe_lib_print_time(cdd_computation_time);
	printf("\n Total orbit computation time:  "); owe_lib_print_time(orbit_computation_time);
	printf("\n\n");

	/* Free variables used in computation. */
	free(vertex_coordinate);
	free(integer_ray);
	free(incidence_inequalities);
	free(integer_ray_denominators);

	orbitwise_adjacency_tree_delete(&orbit_orbitwise_adjacenct_tree);

	return(1);

}



coordinate orbit_processor_get_neighbor_vertex(coordinate orbit_canonical_representative, unsigned long int orbit_incidence, metric_polytope_inequality* incidence_inequalities, int* integer_ray, coordinate new_vertex) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_processor_get_neighbor_orbit
	*  ----------------------------------
	*  Parameters:
	*		- orbit:	orbit whose neighborhood wants to be calculated.
	*		- ray:		ray that will be used for calculating the orbit neighborhood.
	*		- new_vertex:	the canonical representative of the neighbor vertex of the orbit
	*						will be stored in this coordinate. Note that this coordinate has
	*						to be allocated before calling this function (it is used as a
	*						reference parameter).
	*						
	*
	*  Return value:
	*		- This function first calculates the neighbor vertex of the orbit canonical representative as follows:
	*
	*						neighbor vertex = orbit canonical representative + t * ray
	*
	*		  The coordinate of the neighbor vertex is returned (with its corresponding denominator).
  	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	int dimension;	/* Variable used as a counter for elements of a coordinate. */
	int small_t_numerator = 0, small_t_denominator = 0; /* Numerator and denominator of smallest t. */
	int current_numerator, current_denominator;	/* Temporary numerator and denominator (their ratio will be
													compared to that of small_t_numerator and small_t_denominator
													to determine if that is the smallest t found so far. */
	double small_t=0;	/* Double precision value where the ratio of small_t_numerator and small_t_denominator used
							to perform quick "smaller than" comparisons agains the ratio current_numerator / current_denominator. */
	
	double current_small_t=0;

	int vertex_numerator_lcm_coefficient, ray_numerator_lcm_coefficient;	/* Intermediate calculations for obtaining
																				the coordinate of a new vertex from the sum
																				of the orbit canonical representative and
																				t times a ray. */

	coordinate vertex_coordinate;	/* Pointer (to coordinate item) used to parse the coordinate items of the orbit canonical representative (vertex). */
	
	coordinate new_vertex_coordinate;	/* Pointer (to coordinate item) used to parse the coordinate items of the new vertex. */
	
	int* integer_ray_coordinate;	/* Pointer (to ray item) used to parse the ray items. */

	metric_polytope_inequality inequality_coordinate;	/* Pointer (to metric polytope inequality) used to parse the set
															of non-incident inequalities (which are metric_polytope.inequality_count
															- metric_polytope.incidence; and they are stored in parameter inequality_set. */
	int new_vertex_coordinate_greatest_common_denominator;	/* Integer used for reducing all items of the new vertex coordinate. */

	int inequality_constant;	/* This variable will be used for the calculation of the neighbor vertex through small t. */
	metric_polytope_inequality *inequality_set = incidence_inequalities+orbit_incidence;	/* Array of pointer to inequalities that do not
																										contain the canonical representative of the
																										orbit. */
	unsigned int inequality_count = owe_setting.metric_polytope.inequality_count-orbit_incidence;	/* Number of inequalities that do not contain the canonical
																							representative of the orbit (total number of inequalities
																							minus orbit incidence). */

	/*DEBUG*/ #ifdef DEBUG_ORBIT_NEIGHBORHOOD_CALCULATION
	/*DEBUG*/ printf("\nRay coordinate: "); dimension = owe_setting.metric_polytope.dimension; while(dimension--) printf("%d ",*integer_ray++); integer_ray-=owe_setting.metric_polytope.dimension;
	/*DEBUG*/ #endif

	/* Calculation of the smallest t such that vertex + t*ray = new_vertex. */
	while(inequality_count--) {
		inequality_coordinate = *inequality_set++;
		inequality_constant = *inequality_coordinate++ * *(orbit_canonical_representative+owe_setting.metric_polytope.dimension);
		current_numerator = 0;
		current_denominator = 0;
		vertex_coordinate = orbit_canonical_representative;
		integer_ray_coordinate = integer_ray;
		dimension = owe_setting.metric_polytope.dimension;
		while(dimension--) {
			current_numerator -= *inequality_coordinate * *vertex_coordinate++;
			current_denominator -= *integer_ray_coordinate++ * *inequality_coordinate++;
		}

		current_numerator = inequality_constant - current_numerator;
		current_denominator *= *(orbit_canonical_representative+owe_setting.metric_polytope.dimension);

		current_small_t = (double) current_numerator/ (double) current_denominator;
		/* Keep the smallest t found so far. */
		if(current_denominator&&current_small_t > 0) {
			if((((double) current_numerator/ (double) current_denominator)<small_t)||!small_t) {
				small_t = (double) current_numerator/ (double) current_denominator;
				small_t_numerator = current_numerator;
				small_t_denominator = current_denominator;
			}
		}
	}

	
	/* At this point small_t_numerator and small_t_denominator contain the fraction that
	corresponds	to the smallest t. This fraction is reduced. */
	owe_lib_reduce_fraction(&small_t_numerator, &small_t_denominator);

	/*DEBUG*/ #ifdef DEBUG_ORBIT_NEIGHBORHOOD_CALCULATION
	/*DEBUG*/ printf("\nsmall t: %f (%d/%d)\n",small_t,small_t_numerator,small_t_denominator); 
	/*DEBUG*/ #endif

	/* Set the denominator of the new orbit to the least common multiple of the denominator of the orbit canonical representative
		and the denominator of small t. Then prepare the vertex and ray coefficients. */
	*(new_vertex+owe_setting.metric_polytope.dimension) = new_vertex_coordinate_greatest_common_denominator = owe_lib_least_common_multiple(*(orbit_canonical_representative+owe_setting.metric_polytope.dimension), small_t_denominator);
	vertex_numerator_lcm_coefficient = *(new_vertex+owe_setting.metric_polytope.dimension) / *(orbit_canonical_representative+owe_setting.metric_polytope.dimension);
	ray_numerator_lcm_coefficient = small_t_numerator * *(new_vertex+owe_setting.metric_polytope.dimension) / small_t_denominator;
		
	/* Prepare the values of the pointers used in each iteration for parsing the elements. */
	vertex_coordinate = orbit_canonical_representative;
	new_vertex_coordinate = new_vertex-1;
	integer_ray_coordinate = integer_ray;
	dimension = owe_setting.metric_polytope.dimension;
	/* Assign each coordinate item of the new vertex to its corresponding value. */
	while(dimension--) {
		*++new_vertex_coordinate = (vertex_numerator_lcm_coefficient * *vertex_coordinate++) + (ray_numerator_lcm_coefficient * *integer_ray_coordinate++);
		
				/* For each coordinate, the greatest common denominator is obtained (as long as its value is greater than 1. */
		if(new_vertex_coordinate_greatest_common_denominator > 1 && *new_vertex_coordinate)
			new_vertex_coordinate_greatest_common_denominator = owe_lib_greatest_common_denominator(*new_vertex_coordinate, new_vertex_coordinate_greatest_common_denominator);
	}

	/* If new_vertex_coordinate_greatest_common_denominator contains a value greater than 1, then the entire coordinate (including the denominator) can be reduced by it. */
	if(new_vertex_coordinate_greatest_common_denominator > 1) {
		new_vertex_coordinate = new_vertex;
		dimension = owe_setting.metric_polytope.dimension + 1;
		while(dimension--)
			*new_vertex_coordinate++ /= new_vertex_coordinate_greatest_common_denominator;
	}
		
	/*DEBUG*/ #ifdef DEBUG_ORBIT_NEIGHBORHOOD_CALCULATION
	/*DEBUG*/ printf("vertex:  "); orbit_print_coordinate(new_vertex); printf("\n");
	/*DEBUG*/ #endif

	return(new_vertex);

}



int orbit_processor_enlarge_ray(mytype* ray, int* integer_ray_numerators, int* integer_ray_denominators) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_processor_enlarge_ray
	*  ---------------------------
	*  Parameters:
	*		- ray:	coordinates of ray to enlarge (array of type "mytype").
	*		- integer_ray_numerators:	pointer to already allocated memory that will contain the enlarged
	*									integer	ray.
	*		- integer_ray_denominators:	pointer to already allocated memory that will contain the enlarged
	*									integer	ray denominators. These values
	*
	*	This function takes a ray of type "mytype" (double precision) and enlarges it so that all its
	*	coordinates are integers. For this all double precision coordinates of the ray are converted
	*	to fractions using the continuous fraction method (implemented in function
	*	owe_lib_double_to_fraction). The least common multiplier of all fractions is computed and
	*	used to eliminate all denominators. Finally, the enlarged integer ray is stored into the
	*	memory address pointed by integer_ray_numerators.
	*
	*  Return value:
	*		- The least common multiplier used for the calculation of the ray enlargement.
  	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


	int dimension = owe_setting.metric_polytope.dimension;
	int least_common_multiple = 1;
	
	while(dimension--) 
	least_common_multiple = owe_lib_least_common_multiple(least_common_multiple, owe_lib_double_to_fraction(**ray++, integer_ray_numerators++, integer_ray_denominators++));
	
	dimension = owe_setting.metric_polytope.dimension;
	integer_ray_numerators -= dimension;
	integer_ray_denominators -= dimension;

	while(dimension--)
		*integer_ray_numerators++ *= least_common_multiple / *integer_ray_denominators++;

	return least_common_multiple;
}



long unsigned int orbit_processor_calculate_orbit_incidence(coordinate orbit_canonical_representative,metric_polytope_inequality* incidence_inequalities) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_processor_calculate_orbit_incidence
	*  -----------------------------------------
	*  Parameters:
	*		- orbit: pointer to the orbit whose incidence is to be calculated
	*
	*	This function stores the orbit incidence in the "incidence" attribute of the orbit structure.
	*	It also assigns the corresponding values to the "incidence_inequalities" attribute of the orbit
	*	structure as follows:
	*	
	*		The "incidence_inequalities" is an array of pointers to inequalities whose size is equal
	*		to the number of inequalities of the H-representation of the "working" metric polytope
	*		(its memory is allocated in the	orbit_create_orbit function). The the first (incidence)
	*		values of the array point to the inequalities that contain the canonical representative
	*		of the orbit, while the rest doesn't.
	*
	*	Return value:
	*		- The incidence of the orbit (integer value).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned long int orbit_incidence = 0;
	metric_polytope_inequality inequality_value = owe_setting.metric_polytope.inequalities;
	unsigned short int inequality_count = owe_setting.metric_polytope.inequality_count;
	unsigned short int dimension;
	signed int equality_value;
	coordinate canonical_representative_value;


	/* For each inequality of the metric polytope*/
	while(inequality_count--) {

		/* Set the pointer value of the canonical representative. */
		canonical_representative_value = orbit_canonical_representative;
		
		/* Set the value of the equality to the product of the metric polytope inequality constant and the
		canonical representative denominator. */
		equality_value = *(inequality_value++) * *(canonical_representative_value+owe_setting.metric_polytope.dimension);
		
		/* Add to the equality value the product of each element of the canonical representative
		coordinate times the corresponding coheficient of the metric polytope inequality. */
		dimension = owe_setting.metric_polytope.dimension;
		while(dimension--) {
			equality_value += *(inequality_value++) * *(canonical_representative_value++);
		}

		/* If the equality is 0, increment incidence and update incident inequalities/non incident inequalities. */
		incidence_inequalities[(!equality_value)?orbit_incidence++:inequality_count+orbit_incidence]=inequality_value-owe_setting.metric_polytope.dimension-1;
	}
	return(orbit_incidence);
}
