\ignore{
\begin{code}
{-# LANGUAGE 
    ScopedTypeVariables
  , MultiParamTypeClasses
  , FunctionalDependencies
  , FlexibleInstances
  , UndecidableInstances
  , TypeFamilies
  , DataKinds
  , PolyKinds
  , TypeOperators
  , ConstraintKinds
  #-}
module TypeComputation.Numeric.Integer 
  ( module TypeComputation.Numeric.Integer
  , module TypeComputation.Numeric.Natural
  ) where 

import TypeComputation.Numeric.Natural
import TypeComputation.List

import Data.Ratio ((%))

\end{code}
}

\subsection{Integers}\label{sec:Integer}

This module implements arithmetic for integers. 

Basic definitions: sign and integer. 
\begin{code}
data Sign = Neg | Pos
type (:+) = Pos
type (:-) = Neg

class SignVal (s :: Sign) where signVal :: p s -> Integer 
instance SignVal Pos where signVal _ = 1
instance SignVal Neg where signVal _ = (-1)

data INT (s :: Sign) (n :: Nat) = INT
instance (SignVal s, KnownNat n) => Show (INT s n) where 
  show s = show (signVal (Proxy :: Proxy s) * natVal s)

instance (SignVal s, KnownNat n) => GenericNumber (INT s n) where
    numToRational a = (signVal (Proxy :: Proxy s) * natVal a) % 1
    numToFloating a = fromIntegral (signVal (Proxy :: Proxy s) * natVal a)

intVal :: forall n s proxy . (KnownNat n, SignVal s) => proxy s n -> Integer
intVal a = signVal (Proxy :: Proxy s) * natVal a

\end{code}
Determine if a type is an integer.
\begin{code}
type family IsInt n where 
  IsInt (INT s n)  = Always

isInt :: IsInt n => n -> ()
isInt _ = ()

\end{code}
Determine if an integer is zero.
\begin{code}
instance (n == 0) => IsZero (INT s n) 
instance (n /= 0) => NonZero (INT s n) 

\end{code}
Conversion to an integer.
\begin{code}
type family ToInt a where 
  ToInt (NAT    n)  =  INT Pos n
  ToInt (INT s  n)  =  INT s n

\end{code}
Negation.
\begin{code}
instance Negate  (INT Pos n)  (INT Neg n)  where negate' _ = INT
instance Negate  (INT Neg n)  (INT Pos n)  where negate' _ = INT

type instance Negate' (INT s n) = NegateInt (INT s n)

type family NegateInt n where 
  NegateInt  (INT  sa   0)  = INT Pos 0 
  NegateInt  (INT  Pos  n)  = INT Neg n
  NegateInt  (INT  Neg  n)  = INT Pos n

\end{code}
Absolute value.
\begin{code}
instance Abs (INT s n) (INT Pos n) where abs' _ = INT 

type instance Abs' (INT s n) = INT Pos n

\end{code}
Comparison. There are many special cases for comparing two integers which are included in an effort to speed up typical computations,
such as comparison to 0 or 1.
\begin{code}
instance (CompareInt (INT sa a) (INT sb b) ~ c) => Compare (INT sa a) (INT sb b) c 

type instance Compare' (INT sa a) (INT sb b) = CompareInt (INT sa a) (INT sb b)

type family CompareInt a b where 
  CompareInt (k        0)  (k        0) = EQ
  CompareInt (INT Pos  x)  (k        0) = GT
  CompareInt (INT Neg  x)  (k        0) = LT
  CompareInt (k        0)  (INT Pos  x) = LT
  CompareInt (k        0)  (INT Neg  x) = GT

  CompareInt (INT Pos a) (INT Pos b) = CompareNat (NAT a) (NAT b)
  CompareInt (INT Neg a) (INT Neg b) = CompareNat (NAT b) (NAT a)

  CompareInt (INT Pos a) (INT Neg b) = GT
  CompareInt (INT Neg a) (INT Pos b) = LT 

compareInt :: a -> b -> Proxy (CompareInt a b)
compareInt _ _ = Proxy 

\end{code}
Addition.
\begin{code}
instance (a ~ (INT sa an), b ~ (INT sb bn), c ~ (INT sc cn),
          AddInt a b ~ c , AddInt c (NegateInt b) ~ a, AddInt c (NegateInt a) ~ b
         ) => Add (INT sa an) (INT sb bn) (INT sc cn) where add _ _ = INT 

type instance INT sa a + INT sb b = AddInt (INT sa a) (INT sb b)

type family AddInt a b where 
  AddInt (INT Pos  a) (INT Neg  a) = INT Pos 0 -- These two lines are required to prevent `INT Neg 0`
  AddInt (INT Neg  a) (INT Pos  a) = INT Pos 0 

  AddInt (INT Pos  a) (INT Pos  b) = INT Pos (a + b)
  AddInt (INT Neg  a) (INT Neg  b) = INT Neg (a + b)

  AddInt (INT sa   a) (INT sb   b) = AddInt' (a <=? b) (INT sa  a) (INT sb  b)


type family AddInt' cmp a b where 
  AddInt' True   (INT Pos a) (INT Neg b) = INT Neg (b - a)
  AddInt' False  (INT Pos a) (INT Neg b) = INT Pos (a - b)
  AddInt' True   (INT Neg a) (INT Pos b) = INT Pos (b - a)
  AddInt' False  (INT Neg a) (INT Pos b) = INT Neg (a - b)

addInt :: (AddInt (INT sa a) (INT sb b) ~ (INT sc c)) => INT sa a -> INT sb b -> INT sc c
addInt _ _ = INT

\end{code}
Subtraction.
\begin{code}
instance (a ~ (INT sa an), b ~ (INT sb bn), c ~ (INT sc cn),
          AddInt a (NegateInt b) ~ c, AddInt c b ~ a, AddInt a (NegateInt c) ~ b
         ) => Subtract (INT sa an) (INT sb bn) (INT sc cn) where sub _ _ = INT 

type instance INT sa a - INT sb b = SubtractInt (INT sa a) (INT sb b)

type family SubtractInt a b where 
  SubtractInt a b = AddInt a (NegateInt b)

\end{code}
Multiplication.
\begin{code}
instance ((an * bn) ~ cn, MultSign sa sb sc) => Mult (INT sa an) (INT sb bn) (INT sc cn) where mult _ _ = INT 

type instance INT sa a * INT sb b = MultInt (INT sa a) (INT sb b)

class MultSign a b c | a b -> c, b c -> a, a c -> b 
instance MultSign Pos Pos Pos 
instance MultSign Neg Neg Pos 
instance MultSign Neg Pos Neg
instance MultSign Pos Neg Neg

type family MultSign' a b where 
  MultSign' Pos Pos = Pos 
  MultSign' Neg Neg = Pos 
  MultSign' Neg Pos = Neg
  MultSign' Pos Neg = Neg


type family MultInt a b where 
  MultInt (INT sa an) (INT sb bn) = INT (MultSign' sa sb) (an * bn)

\end{code}
Multiplication of non-zero integers.
\begin{code}
instance (an /= 0, bn /= 0, cn /= 0, (an * bn) ~ cn, MultSign sa sb sc
         ) => MultNZ (INT sa an) (INT sb bn) (INT sc cn) where multNZ _ _ = INT 

\end{code}
Exponentiation of integers. A negative exponent is an error. 
\begin{code}
type family ExpInt n e where 
  ExpInt n             (INT s    0 ) = INT Pos 1 
  ExpInt n             (INT Pos  1 ) = n
  ExpInt (INT Pos n0)  (INT Pos  n1) = INT Pos (n0^n1)
  ExpInt (INT Neg n0)  (INT Pos  n1) = ExpIntNeg (IsEven n1) (n0^n1)

type ExpIntNeg b = If b (INT Pos) (INT Neg)

type instance (^) (INT s0 n0) (INT s1 n1) = ExpInt (INT s0 n0) (INT s1 n1) 
instance ((INT s0 n0 ^ INT s1 n1) ~ INT s2 n2)
         => Pow (INT s0 n0) (INT s1 n1) (INT s2 n2)  
            where pow _ _ = INT

\end{code}
Minimum and maximum of integers.
\begin{code}
instance MinMaxInt (INT sa a) (INT sb b) ~ (INT sc c, t) => Min (INT sa a) (INT sb b) (INT sc c) where 
  min' _ _ = INT

instance MinMaxInt (INT sa a) (INT sb b) ~ (t, INT sc c) => Max (INT sa a) (INT sb b) (INT sc c) where 
  max' _ _ = INT

type MinMaxInt a b = If (CompareInt a b ==? LT) (a, b) (b, a)  
type MinInt a b = Fst (MinMaxInt a b)
type MaxInt a b = Snd (MinMaxInt a b)


\end{code}
