\ignore{
\begin{code}
{-# LANGUAGE 
    EmptyDataDecls
  , MultiParamTypeClasses
  , FunctionalDependencies
  , FlexibleInstances
  , RankNTypes
  , TemplateHaskell
  , DataKinds 
  , KindSignatures
  , PolyKinds
  , GADTs
  , TypeFamilies
  , UndecidableInstances
  , ScopedTypeVariables
  , TypeOperators
  , ConstraintKinds
  #-}

module TypeComputation.UnitDefinitions where

import TypeComputation.Numeric.Integer

import GHC.Prim (Constraint)

\end{code}
}

\section{Unit Definitions}\label{sec:UnitDefinitions}

In this module we define a type-level number for the power of units.
The units include six of the SI base units: meter, second, kilogram, Ampere, mole, and Kelvin. 


Basic unit types definitions. These data types will be lifted to kinds. 
\begin{code}
data M_U    =  M    Nat  |  M_    Nat 
data S_U    =  S    Nat  |  S_    Nat
data Kg_U   =  Kg   Nat  |  Kg_   Nat
data A_U    =  A    Nat  |  A_    Nat
data Mol_U  =  Mol  Nat  |  Mol_  Nat
data K_U    =  K    Nat  |  K_    Nat

\end{code}
Showing units. 
\begin{code}

showUnit' :: String -> Integer -> Integer -> String 
showUnit'  _  _  0 = "" 
showUnit'  s  1  1 = s 
showUnit'  s  _  1 = s ++ "^-1"
showUnit'  s  m  n = s ++ "^" ++ show (m*n)

class ShowUnit k where 
  showUnit :: p k -> String 

instance KnownNat n => ShowUnit (M n) where 
  showUnit _ = showUnit' "m" 1 (natVal (Proxy :: Proxy n))

instance KnownNat n => ShowUnit (M_ n) where 
  showUnit _ = showUnit' "m" (-1) (natVal (Proxy :: Proxy n))     

\end{code}
\ignore{
\begin{code}
instance KnownNat n => ShowUnit (S     n) where showUnit _ = showUnit' ("s"  )  1     (natVal (Proxy :: Proxy n)) 
instance KnownNat n => ShowUnit (S_    n) where showUnit _ = showUnit' ("s"  )  (-1)  (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (Kg    n) where showUnit _ = showUnit' ("kg" )  1     (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (Kg_   n) where showUnit _ = showUnit' ("kg" )  (-1)  (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (A     n) where showUnit _ = showUnit' ("A"  )  1     (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (A_    n) where showUnit _ = showUnit' ("A"  )  (-1)  (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (Mol   n) where showUnit _ = showUnit' ("mol")  1     (natVal (Proxy :: Proxy n))  
instance KnownNat n => ShowUnit (Mol_  n) where showUnit _ = showUnit' ("mol")  (-1)  (natVal (Proxy :: Proxy n)) 
\end{code}
}
$\cdots$
\begin{code}
instance KnownNat n => ShowUnit (K n) where 
  showUnit _ = showUnit' "K" 1 (natVal (Proxy :: Proxy n)) 
   
instance KnownNat n => ShowUnit (K_ n) where 
  showUnit _ = showUnit' "K" (-1) (natVal (Proxy :: Proxy n))     

\end{code}

\subsection{SI Units}\label{subsec:SIUnit}

Units are represented as sums of SI base units.
\begin{code}

data SIUnit (m :: M_U) (s :: S_U) (kg :: Kg_U) (a :: A_U) (mol :: Mol_U) (k :: K_U) val 
   = SIUnit val 


instance (ShowUnit m, ShowUnit s, ShowUnit kg, ShowUnit a,
          ShowUnit mol, ShowUnit k, Show val) => Show (SIUnit m s kg a mol k val) where 
  show (SIUnit val) = unwords $ filter (not . null)
                      [show val, showUnit (Proxy :: Proxy m), showUnit (Proxy :: Proxy s), 
                                 showUnit (Proxy :: Proxy kg), showUnit (Proxy :: Proxy a),
                                 showUnit (Proxy :: Proxy mol), showUnit (Proxy :: Proxy k)]


\end{code}
We have a type class for known units. It is not really a type class, but it behaves like one.
We use a closed type family to ensure that no other instances can ever be created.
\begin{code}

type family KnownUnit (x :: k) :: Constraint where 
  KnownUnit (SIUnit m s kg a mol k) = ()
  KnownUnit (SIUnit m s kg a mol k v) = ()

knownUnit :: KnownUnit x => x -> ()
knownUnit _ = ()

\end{code}
Addition of numbers with units is simple as it requires only that type be the same.
\begin{code}
addU :: (u ~ SIUnit m s kg a mol k, Num v) => u v -> u v -> u v
addU (SIUnit a) (SIUnit b) = SIUnit (a+b)

instance (a0 ~ a1, a1 ~ a2, Num a0, Num a1, Num a2) =>
  Add (SIUnit m s kg amp mol k a0) (SIUnit m s kg amp mol k a1) (SIUnit m s kg amp mol k a2) 
    where add (SIUnit a) (SIUnit b) = SIUnit (a + b)

\end{code}

\subsection{Addition of units}\label{sec:UnitAdd}

It is simpler to perform conversions of units to integers and back than to rewrite integer arithmetic
for the exponent type.
\begin{code}

type family UnitToSign (a :: k) where 
  UnitToSign  M    = Pos; UnitToSign M_    = Neg;  
  UnitToSign  S    = Pos; UnitToSign S_    = Neg; 
  UnitToSign  Kg   = Pos; UnitToSign Kg_   = Neg; 
  UnitToSign  A    = Pos; UnitToSign A_    = Neg; 
  UnitToSign  Mol  = Pos; UnitToSign Mol_  = Neg; 
  UnitToSign  K    = Pos; UnitToSign K_    = Neg;


type family UnitToInt (a :: k) where 
  UnitToInt (k 0) = INT Pos 0
  UnitToInt (k a) = INT (UnitToSign k) a

unitToInt ::  UnitToInt u ~ INT s a => p u -> INT s a
unitToInt _ = INT 

\end{code}
The first type in `IntToUnit` acts as a witness to determine the output type.
\begin{code}

type family IntToUnit (w :: k) a :: k0 where 
  IntToUnit (M     x) (INT Pos a) = M    a; IntToUnit  (M     x) (INT Neg a) = M_    a;
  IntToUnit (M_    x) (INT Pos a) = M    a; IntToUnit  (M_    x) (INT Neg a) = M_    a;  
  IntToUnit (S     x) (INT Pos a) = S    a; IntToUnit  (S     x) (INT Neg a) = S_    a;
  IntToUnit (S_    x) (INT Pos a) = S    a; IntToUnit  (S_    x) (INT Neg a) = S_    a; 
  IntToUnit (Kg    x) (INT Pos a) = Kg   a; IntToUnit  (Kg    x) (INT Neg a) = Kg_   a;
  IntToUnit (Kg_   x) (INT Pos a) = Kg   a; IntToUnit  (Kg_   x) (INT Neg a) = Kg_   a; 
  IntToUnit (A     x) (INT Pos a) = A    a; IntToUnit  (A     x) (INT Neg a) = A_    a;
  IntToUnit (A_    x) (INT Pos a) = A    a; IntToUnit  (A_    x) (INT Neg a) = A_    a; 
  IntToUnit (Mol   x) (INT Pos a) = Mol  a; IntToUnit  (Mol   x) (INT Neg a) = Mol_  a;
  IntToUnit (Mol_  x) (INT Pos a) = Mol  a; IntToUnit  (Mol_  x) (INT Neg a) = Mol_  a; 
  IntToUnit (K     x) (INT Pos a) = K    a; IntToUnit  (K     x) (INT Neg a) = K_    a;
  IntToUnit (K_    x) (INT Pos a) = K    a; IntToUnit  (K_    x) (INT Neg a) = K_    a; 


\end{code}
Adding units; used to multiply values with units. 
First define addition of base units, then of composite units.
\begin{code}

class AddUnitOne' (a :: k) (b :: k) (c :: k) | a b -> c, a c -> b, b c -> a 
instance (UnitToInt u0 ~ i0, UnitToInt u1 ~ i1, IntToUnit u0 i2 ~ u2, Add i0 i1 i2) => AddUnitOne' u0 u1 u2 


instance (Num v0, Num v1, Num v2, v0 ~ v1, v1 ~ v2, 
          AddUnitOne' m0 m1 m2, AddUnitOne'   s0   s1   s2, AddUnitOne' kg0 kg1 kg2, 
          AddUnitOne' a0 a1 a2, AddUnitOne' mol0 mol1 mol2, AddUnitOne'  k0  k1  k2
         ) => Mult (SIUnit m0 s0 kg0 a0 mol0 k0 v0) (SIUnit m1 s1 kg1 a1 mol1 k1 v1) (SIUnit m2 s2 kg2 a2 mol2 k2 v2) where 
  mult (SIUnit a) (SIUnit b) = SIUnit (a * b) 
    
\end{code}

\subsection{Composing units}\label{sec:UnitComp}

Some useful functions for composing units naturally and concisely.
\begin{code}

type AddUnitOne a b = IntToUnit a (AddInt (UnitToInt a) (UnitToInt b))
  
type family AddUnit (a :: * -> *) (b :: * -> *) :: * -> * where 
  SIUnit m0 s0 kg0 a0 mol0 k0 `AddUnit` SIUnit m1 s1 kg1 a1 mol1 k1 = 
    SIUnit (AddUnitOne m0 m1) (AddUnitOne s0 s1) (AddUnitOne kg0 kg1) (AddUnitOne a0 a1) (AddUnitOne mol0 mol1) (AddUnitOne k0 k1)

type (:*:) a b = AddUnit (ToSIUnit a) (ToSIUnit b)

addUnit :: ((SIUnit m s kg am mol k) ~ AddUnit a b) => Proxy a -> Proxy b -> Proxy (SIUnit m s kg am mol k)
addUnit _ _ = Proxy 


type family ToSIUnit (a :: k) where 
  ToSIUnit (SIUnit m s kg a mol k) = SIUnit m s kg a mol k
  ToSIUnit (M     x) = SIUnit (M   x) (S   0) (Kg   0) (A   0) (Mol   0) (K   0)
  ToSIUnit (M_    x) = SIUnit (M_  x) (S   0) (Kg   0) (A   0) (Mol   0) (K   0)
  ToSIUnit (S     x) = SIUnit (M   0) (S   x) (Kg   0) (A   0) (Mol   0) (K   0)
  ToSIUnit (S_    x) = SIUnit (M   0) (S_  x) (Kg   0) (A   0) (Mol   0) (K   0)
  ToSIUnit (Kg    x) = SIUnit (M   0) (S   0) (Kg   x) (A   0) (Mol   0) (K   0)
  ToSIUnit (Kg_   x) = SIUnit (M   0) (S   0) (Kg_  x) (A   0) (Mol   0) (K   0) 
  ToSIUnit (A     x) = SIUnit (M   0) (S   0) (Kg   0) (A   x) (Mol   0) (K   0)         
  ToSIUnit (A_    x) = SIUnit (M   0) (S   0) (Kg   0) (A_  x) (Mol   0) (K   0)         
  ToSIUnit (Mol   x) = SIUnit (M   0) (S   0) (Kg   0) (A   0) (Mol   x) (K   0) 
  ToSIUnit (Mol_  x) = SIUnit (M   0) (S   0) (Kg   0) (A   0) (Mol_  x) (K   0) 
  ToSIUnit (K     x) = SIUnit (M   0) (S   0) (Kg   0) (A   0) (Mol   0) (K   x)
  ToSIUnit (K_    x) = SIUnit (M   0) (S   0) (Kg   0) (A   0) (Mol   0) (K_  x) 


type family Recip (a :: k) :: k where 
  Recip (SIUnit m s kg a mol k) = SIUnit (Recip m) (Recip s) (Recip kg) (Recip a) (Recip mol) (Recip k)
  Recip (M     0) = M     0; Recip (M     x) = M_    x 
  Recip (M_    0) = M     0; Recip (M_    x) = M     x 
  Recip (S     0) = S     0; Recip (S     x) = S_    x 
  Recip (S_    0) = S     0; Recip (S_    x) = S     x 
  Recip (Kg    0) = Kg    0; Recip (Kg    x) = Kg_   x 
  Recip (Kg_   0) = Kg    0; Recip (Kg_   x) = Kg    x  
  Recip (A     0) = A     0; Recip (A     x) = A_    x          
  Recip (A_    0) = A     0; Recip (A_    x) = A     x          
  Recip (Mol   0) = Mol   0; Recip (Mol   x) = Mol_  x  
  Recip (Mol_  0) = Mol   0; Recip (Mol_  x) = Mol   x  
  Recip (K     0) = K     0; Recip (K     x) = K_    x 
  Recip (K_    0) = K     0; Recip (K_    x) = K     x 

\end{code}
Division and reciprocation of units 
\begin{code}

recipU :: Fractional v => SIUnit m s kg a mol k v -> SIUnit (Recip m) (Recip s) (Recip kg) (Recip a) (Recip mol) (Recip k) v
recipU (SIUnit a) = SIUnit (1/a)

type (:/:) a b = a :*: (Recip b)
divU :: (Fractional v, u0 ~ SIUnit m0 s0 kg0 a0 mol0 k0, u1 ~ SIUnit m1 s1 kg1 a1 mol1 k1) => u0 v -> u1 v -> (u0 :/: u1) v
divU (SIUnit a) (SIUnit b) = SIUnit (a/b)

\end{code}
Useful unit synonyms. 
\begin{code}

type Unitless = SIUnit (M 0) (S 0) (Kg 0) (A 0) (Mol 0) (K 0)
unitless :: a -> Unitless a 
unitless = SIUnit

type Meter = ToSIUnit (M 1)
meter :: a -> Meter a
meter = SIUnit

type Second = ToSIUnit (S 1)
second :: a -> Second a
second = SIUnit

type Kilogram = ToSIUnit (Kg 1)
kilogram :: a -> Kilogram a
kilogram = SIUnit

type MetersPerSecond = Meter :/: Second
speed :: a -> MetersPerSecond a
speed = SIUnit

type Acceleration = MetersPerSecond :/: Second
accel :: a -> Acceleration a
accel = SIUnit

type Newton = Kilogram :*: Acceleration
newton :: a -> Newton a 
newton = SIUnit

type Joule = Newton :*: Meter 
joule :: a -> Joule a 
joule = SIUnit

type Watt = Joule :/: Second
watt :: a -> Watt a
watt = SIUnit

\end{code}
