\ignore{
\begin{code}
{-# LANGUAGE EmptyDataDecls,
    MultiParamTypeClasses, FunctionalDependencies, UndecidableInstances,
    FlexibleInstances, RankNTypes, FlexibleContexts,
    TypeFamilies, GADTs, ScopedTypeVariables, Rank2Types,  ConstraintKinds
  #-}
module Examples where

import qualified Prelude
import Prelude (Integer, Int, Double, undefined, Show, fromIntegral, fromRational
                ,(^),pi,sin,cos,(/),Eq,String,($),(&&), zip, unwords,unlines,writeFile, map, show, zipWith, (++), (.))
import SizeTypes
import SIUnits 
import qualified SIUnits as U
import UnitDefinitions

import Data.List (transpose)
import Data.Complex
import qualified Numeric.FFT as FFT
\end{code}
}

\section{Examples}\label{sec:Examples}
This module contains examples of applications of code in the other modules. 


\subsection{FFT}\label{subsec:FFT}
The examples below demonstrate an application of type level numbers and units to the Fast Fourier Transform.\\ 
The basic class defintion for a frame of reference, containing a function to show the name of a frame, and a type alias for the base unit of the frame. The type of \hask{BaseUnit} should be an instance of \hask{SIUnitVal} with an empty tuple as a value.
\begin{code}

class Frame frame where
  type BaseUnit frame
  name :: frame -> String

\end{code}
One application of frames of reference is to the Fast Fourier Transform, which has an inverse operation. The implementation of FFT is provided by the package pure-fft. The package contains two functions useful to us; \hask{fft},
which is the FFT and \hask{ifft}, which is the inverse operation. We make the assertions that inverse FFT applied to the result of FFT has the same type as the original arguement (equivalently, \hask{ ( (ifft . fft) (a :: t) ) :: t }). 
We let the user assert that two frames are dual to each other, and check that the units match.
The functional depency is asserting that the dual of the dual frame is the original frame.
This has to be argued outside the type system, with a definition of what the dual frame is.
Maybe we need to argue it in the context of discretizations, 
if there is no notion of dual coordinate frames, but there is a notion of dual vector spaces.

Before dual frames can be formalized, we first formalize dual units; these are units whose dimensions are reciprocals, or whose product is unitless.
Instances of the class which defines dual frames of reference must contain two frames whose units are dual.
\begin{code}

class DualUnits a b | a -> b, b -> a where
instance (Frame a, Frame b, U.Mult (BaseUnit a) (BaseUnit b) (U.Unitless ())) => DualUnits a b where


class (DualUnits a b) => AssertDualFrames a b | a -> b, b -> a where

\end{code}

The user must then define instances of \hask{Frame} for each frame of reference. They are responsible for defining
base units which are consistent; they then must create an instance of \hask{AssertDualFrames} 
to guarantee that the frames are in fact dual to each other.
\begin{code}
data LabFrame
instance Frame LabFrame where
  type BaseUnit LabFrame = (U.Meter ())
  name _ = "LabFrame"
  
data LabFrameT
instance Frame LabFrameT where
  type BaseUnit LabFrameT = (U.PerM ())
  name _ = "LabFrameDual"

instance AssertDualFrames LabFrame LabFrameT where

\end{code}

If the user defines a frame improperly and attempts to assert the duality of the improperly defined frames
\begin{code}
data BadFrame
instance Frame BadFrame where
  type BaseUnit BadFrame = (U.Unitless ())
  name _ = "LabFrameDual"
\end{code}
\begin{lstlisting}
instance AssertDualFrames LabFrame BadFrame where
\end{lstlisting}
they will meet this compile time error
\begin{lstlisting}
    Couldn't match type `UnitDefinitions.M0' with `UnitDefinitions.M1'
    When using functional dependencies to combine
      UnitDefinitions.AddUnit
        UnitDefinitions.M0 UnitDefinitions.M0 UnitDefinitions.M0,
        arising from the dependency `b c -> a'
\end{lstlisting}
The error is clear that there doesn't exist an instance for adding meters to meters to get meters. 



We defining phantom type wrapper for the \hask{Size} class just for better representation. We also define helper functions for each phantom type which will take a phantom type and give the underlying \hask{Size} instance. These wrappers will be used to define more readable datatypes for demonstration purposes below. The first will be used to represent the number of samples in a discretization; the second and third will represent the numerator and denominator, respectively, of the step size.
\begin{code}

class SizeWrapper w where 
    sizeOf :: w s -> s
    sizeOf = undefined

class NumSamplesC n where 
data NumSamples size 
instance (Size size, NonZero size) => NumSamplesC (NumSamples size) where
instance SizeWrapper NumSamples where 


class StepSizeNumC s where 
data StepSizeNum size 
instance (Size size, NonZero size) => StepSizeNumC (StepSizeNum size) where
instance SizeWrapper StepSizeNum where 

class StepSizeDenomC n where 
data StepSizeDenom size 
instance (Size size, NonZero size) => StepSizeDenomC (StepSizeDenom size) where
instance SizeWrapper StepSizeDenom where 



\end{code}

We define another sample frame.
\begin{code}

data CanalFrame
instance Frame CanalFrame where
  type BaseUnit CanalFrame = (U.Meter ())
  name _ = "CanalFrame"
\end{code}



We are now ready to define a datatype to represent a series of values belonging to one dimensional discretization. The discretization has several physical properties which are all present in the constraints of the datatype.
\begin{code}
data Discretization1D a b c d e f where
    Discretization1D :: (Frame frame, U.SIUnit (BaseUnit frame), U.SIUnit rangeU , Size numSamples , Size stepSizeNum, Size stepSizeDenom
                        ) => val ->
                         Discretization1D frame
                                          (NumSamples numSamples)
                                          (StepSizeNum stepSizeNum)
                                          (StepSizeDenom stepSizeDenom)
                                          rangeU 
                                          val


\end{code}


An instance for addition and subtraction of discretization is implemented.
\begin{code}
instance (U.Add val) => U.Add (Discretization1D frame numSamples stepSizeNum stepSizeDenom rangeU val) where
  (Discretization1D a) + (Discretization1D b) = Discretization1D (a U.+ b)
  (Discretization1D a) - (Discretization1D b) = Discretization1D (a U.- b)
  negate (Discretization1D a) = Discretization1D (U.negate a)
\end{code}


Using examples below, we can see that addition between discretizations which do not share all of the same properties produces errors.
\begin{code}
meas1 :: Discretization1D  LabFrame 
                           (NumSamples (SIZE3 D0 D0 D4))
                           (StepSizeNum(SIZE3 D0 D0 D1))
                           (StepSizeDenom(SIZE3 D5 D0 D0)) 
                           (U.Meter ()) 
                           [Complex Double]
meas1 = Discretization1D [0,1,2,3]

meas2 :: Discretization1D  LabFrame 
                           (NumSamples (SIZE3 D0 D0 D4))
                           (StepSizeNum(SIZE3 D0 D0 D1))
                           (StepSizeDenom(SIZE3 D5 D0 D0)) 
                           (U.Second ()) 
                           [Complex Double]
meas2 = Discretization1D [0.4,0.8,1.2,1.6]

canalSample1 :: Discretization1D CanalFrame 
                                 (NumSamples (SIZE3 D0 D1 D2))
                                 (StepSizeNum (SIZE3 D0 D0 D1))
                                 (StepSizeDenom (SIZE3 D1 D0 D0))
                                 (U.Meter ()) 
                                 [Double]
canalSample1 = Discretization1D [1, 1.2, 1.3, 1.12, 1.23, 1.12, 1.15, 1.25, 1.18, 1.20, 1.24, 1.28]

canalSample2 :: Discretization1D CanalFrame 
                                 (NumSamples (SIZE3 D0 D1 D2))
                                 (StepSizeNum (SIZE3 D0 D0 D1))
                                 (StepSizeDenom (SIZE3 D2 D0 D0))
                                 (U.Meter ()) 
                                 [Double]
canalSample2 = Discretization1D [1, 1.21, 1.2, 1.42, 1.3, 1.32, 1.12, 1.25, 1.23, 1.20, 1.12, 1.28]
\end{code}

\begin{lstlisting}
:t meas1 U.+ canalSample1
    Couldn't match expected type `LabFrame'
                with actual type `CanalFrame'
                
:t canalSample2 U.+ canalSample1
    Couldn't match expected type `D2' with actual type `D1'
    Expected type: Discretization1D
    
:t meas1 U.+ meas2
    Couldn't match expected type `UnitDefinitions.M1'
                with actual type `UnitDefinitions.M0'
    
\end{lstlisting}

We can now use the above defintions to define a class \hask{FT} which will be used to describe the relationship between a set of discrete data and the Fourier Transform of that data. The functional dependencies indicate that the intial data uniquely determines the Fourier transform, and the result of the Fourier transform uniquely determines the initial data.
\begin{code}
class FT a b | a -> b, b -> a where
  ft :: a -> b 
  invFt :: b -> a
\end{code}

The instance of \hask{FT} for one dimensional discretizations constrains the frames of reference of the original data and the result as being dual frames. 
One requirement of Fourier transform is the product of the step sizes of input and output data and the number of samples must be one. The \hask{MultDNonZero} class is used to apply this constraint. 
Note that in order to more easily generalize \hask{FT} to instances of \hask{Size} which have a width other than three, it would be helpful to define a special type \hask{One}. 
\begin{code}
instance (U.SIUnit (BaseUnit frame2),
          Size stepSizeDenom2,
          Size stepSizeNum2,
          Frame frame2,
          U.SIUnit (BaseUnit frame1),
          Size stepSizeDenom1,
          Size stepSizeNum1,
          Frame frame1,
          
          AssertDualFrames frame1 frame2,
          MultDNonZero (stepSizeNum1, stepSizeDenom1) (numSamp, (One numSamp)) (stepSizeDenom2, stepSizeNum2),
          DualUnits frame1 frame2
         ) =>
    FT (Discretization1D frame1 (NumSamples numSamp) (StepSizeNum stepSizeNum1) (StepSizeDenom stepSizeDenom1) rangeU [Complex Double])
       (Discretization1D frame2 (NumSamples numSamp) (StepSizeNum stepSizeNum2) (StepSizeDenom stepSizeDenom2) rangeU [Complex Double]) 
    where
    
    ft (Discretization1D x) = (Discretization1D $ FFT.fft x)
    invFt (Discretization1D x) = (Discretization1D $ FFT.ifft x)
\end{code}

We can now apply Fourier transform to a data set with type safety. The type system infers the type of the result; the result itself is not of particular interest, however.
\begin{lstlisting}
:t meas1
meas1 :: Discretization1D  LabFrame 
                           (NumSamples (SIZE3 D0 D0 D4))
                           (StepSizeNum(SIZE3 D0 D0 D1))
                           (StepSizeDenom(SIZE3 D5 D0 D0)) 
                           (U.Meter ()) 
                           [Complex Double] 
:t ft meas1
ft meas1
  :: Discretization1D
       LabFrameT
       (NumSamples (SIZE3 D0 D0 D4))
       (StepSizeNum (SIZE3 D5 D0 D0))
       (StepSizeDenom (SIZE3 D0 D0 D4))
       (U.Meter ())
       [Complex Double]
\end{lstlisting}
The step sizes of \hask{meas1} and \hask{ft meas1} are (1/500) = 0.002 and (500/4) = 125, respectively. We indeed have that $0.002 \times 125 \times 4 = 1$, as the constraint on the Fourier transform requires.


We can define a similair rectangular discretization in two dimensions. In this case, we need to indicate the number of samples in both perpendicular basis directions, as well as the step size. 
\begin{code}

data LabFrame2D
data LabFrame2DT

instance Frame LabFrame2D where
  type BaseUnit LabFrame2D = U.Meter ()
  name _ = "lab2"
  
instance Frame LabFrame2DT where
  type BaseUnit LabFrame2DT = U.PerM ()
  name _ = "lab2t"
  
instance AssertDualFrames LabFrame2D LabFrame2DT

data Discretization2D a b c d e f g h i where
        Discretization2D :: (Frame frame , U.SIUnit rangeU, U.SIUnit (BaseUnit frame), 
                             Size numSampX, Size numSampY, 
                             Size stepSizeXNum, Size stepSizeXDenom, 
                             Size stepSizeYNum, Size stepSizeYDenom
                            ) => val -> 
            Discretization2D frame
                             (NumSamples numSampX)
                             (StepSizeNum stepSizeXNum)
                             (StepSizeDenom stepSizeXDenom)
                             (NumSamples numSampY)
                             (StepSizeNum stepSizeYNum)
                             (StepSizeDenom stepSizeYDenom)
                             rangeU 
                             val

labSample2D :: Discretization2D LabFrame2D                   
                                (NumSamples    (SIZE3 D0 D0 D8))   
                                (StepSizeNum   (SIZE3 D0 D0 D1))           
                                (StepSizeDenom (SIZE3 D0 D2 D0))        
                                (NumSamples    (SIZE3 D0 D0 D4))           
                                (StepSizeNum   (SIZE3 D0 D0 D1))                
                                (StepSizeDenom (SIZE3 D1 D0 D0))         
                                (U.Meter ())                      
                                [[Complex Double]] 
labSample2D = Discretization2D
                    [[-0.2 ,-0.26 ,-0.32 ,-0.38 ,-0.44 ,-0.5 ,-0.56 ,-0.63]
                    ,[0.2  ,0.26  ,0.32  ,0.38  ,0.44  ,0.5  ,0.56  ,0.76 ]
                    ,[0.6  ,0.78  ,0.96  ,1.14  ,1.32  ,1.5  ,1.68  ,1.98 ]
                    ,[1.0  ,1.3   ,1.6   ,1.90  ,2.2   ,2.5  ,2.8   ,2.94 ]]


instance ( SIUnit (BaseUnit frame1),
           Size stepSizeYDenom1,
           Size stepSizeYNum1,
           Size stepSizeXDenom1,
           Size stepSizeXNum1,
           Frame frame1,
           AssertDualFrames frame1 frame2,
           SIUnit (BaseUnit frame2),
           Size stepSizeYDenom2,
           Size stepSizeYNum2,
           Size stepSizeXDenom2,
           Size stepSizeXNum2,
           Frame frame2,
           MultDNonZero (stepSizeXNum1, stepSizeXDenom1) (numSampX, (One numSampX)) (stepSizeXDenom2, stepSizeXNum2),
           MultDNonZero (stepSizeYNum1, stepSizeYDenom1) (numSampY, (One numSampY)) (stepSizeYDenom2, stepSizeYNum2)
         )
    => FT (Discretization2D  frame1
                             (NumSamples numSampX)
                             (StepSizeNum stepSizeXNum1)
                             (StepSizeDenom stepSizeXDenom1)
                             (NumSamples numSampY)
                             (StepSizeNum stepSizeYNum1)
                             (StepSizeDenom stepSizeYDenom1)
                             rangeU
                             [[Complex Double]])
          (Discretization2D  frame2
                             (NumSamples numSampX)
                             (StepSizeNum stepSizeXNum2)
                             (StepSizeDenom stepSizeXDenom2)
                             (NumSamples numSampY)
                             (StepSizeNum stepSizeYNum2)
                             (StepSizeDenom stepSizeYDenom2)
                             rangeU 
                             [[Complex Double]])
    where
  ft    (Discretization2D x) = (Discretization2D . transpose . (map FFT.fft ) . transpose . (map FFT.fft )) x
  invFt (Discretization2D x) = (Discretization2D . transpose . (map FFT.ifft) . transpose . (map FFT.ifft)) x
  
\end{code}

\subsection{Other examples}\label{subsec:ExOther}

Examples of the use of code in \hask{SIUnits}.
\begin{code}

eg1 = SIUnitVal 7 :: SIUnitVal M1 Kg0 S1 A1 Mol1 K0 Double
eg2 = SIUnitVal 8 :: SIUnitVal M1 Kg1 S1 A1 Mol1 K0 Double
eg3 = SIUnitVal 8 :: SIUnitVal Kg2 M1 S1 A1 Mol1 K0 Double

distance = SIUnitVal 2 :: Meter Double
time =     SIUnitVal 10 :: Second Double
velocity = SIUnitVal 0.2 :: MPerS Double 
testval =  SIUnitVal 4 :: Unitless Double
\end{code}

A simple function for calculation potential energy in the Earth's gravity given change in height and mass.
\begin{code}

potentialEnergy ::          Kilogram Double -> 
                            Meter Double -> 
                            Joule Double
potentialEnergy mass height = mass * g * height
    where g = SIUnitVal 9.81 :: SIUnitVal M1 Kg0 S_2 A0 Mol0 K0 Double
    
\end{code}
The following works correctly: 
\begin{lstlisting}
potentialEnergy (SIUnitVal 3.0 :: Kilogram Double) (SIUnitVal 25.0 :: Meter Double)

735.75 m^{2} kg s^{-2} :: Joule Double
\end{lstlisting}
While an incorrectly typed arguement will yield: 
\begin{lstlisting}
potentialEnergy (SIUnitVal 1.0 :: Second Double) (SIUnitVal 1.0 :: Unitless Double)

    Couldn't match expected type `Kg1' with actual type `Kg0'
    Expected type: Kilogram Double
      Actual type: Second Double
    In the first argument of `potentialEnergy', namely
      `(SIUnitVal 1.0 :: Second Double)'
    In the expression:
      potentialEnergy
        (SIUnitVal 1.0 :: Second Double) (SIUnitVal 1.0 :: Unitless Double)
\end{lstlisting}


A trivial function that demonstrates the functionality of the \hask{NonZero} class.
\begin{code}
toIntNonZero :: forall a . (NonZero a, Size a) => a -> Integer
toIntNonZero _ = toInt (undefined :: a) 
\end{code}

\begin{lstlisting}
toIntNonZero (undefined :: SIZE3 D0 D4 D2)
42

toIntNonZero (undefined :: SIZE3 D0 D0 D0)
    No instance for (Fail TypeLevelNumIsZero)
      arising from a use of `toIntNonZero'
    Possible fix: add an instance declaration for (Fail ())

\end{lstlisting}

We define a function \hask{mult10} which demonstrates multiplication of type level numbers. 
Some sample type level numbers are also defined. 
\begin{code}
mult10 :: (MultD10 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0 e9 e8 e7 e6 e5 e4 e3 e2 e1 e0 g9 g8 g7 g6 g5 g4 g3 g2 g1 g0) =>
          (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) -> (SIZE  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0) -> (SIZE  g9 g8 g7 g6 g5 g4 g3 g2 g1 g0)
mult10 _ _ = undefined

tl_16  = undefined :: SIZE D0 D0 D0 D0 D0 D0 D0 D0 D1 D6 
tl_3   = undefined :: SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D3 
tl_5e5 = undefined :: SIZE D0 D0 D0 D0 D0 D5 D0 D0 D0 D0
tl_3e8 = undefined :: SIZE D0 D2 D9 D9 D7 D7 D2 D4 D5 D8
\end{code}

GHCI gives the following results, as expected.
\begin{lstlisting}
:type mult10 tl_16 tl_3
mult10 tl_16 tl_3 :: SIZE D0 D0 D0 D0 D0 D0 D0 D0 D4 D8

:type mult10 tl_5e5 tl_3
mult10 tl_5e5 tl_3 :: SIZE D0 D0 D0 D0 D1 D5 D0 D0 D0 D0
\end{lstlisting}


However, the square of $3 \times 10^8$ is a number containing more than ten digits. When we try this arithmetic, we get:
\begin{lstlisting}
:type mult10 tl_3e8 tl_3e8
    Couldn't match type `D9' with `D0'
    When using functional dependencies to combine
      Times D3 D3 D0 D9,
        arising from the dependency `da db -> low'
        in the instance declaration at sizes.lhs:299:10
      Times D3 D3 D0 D0,
        arising from a use of `mult10' at <interactive>:1:1-6
    In the expression: mult10 tl_3e8 tl_3e8
\end{lstlisting}

