\ignore{
\begin{code}
{-# LANGUAGE 
    RankNTypes, 
    ScopedTypeVariables, 
    EmptyDataDecls, 
    MultiParamTypeClasses,
    FunctionalDependencies, 
    FlexibleInstances, 
    UndecidableInstances, 
    OverlappingInstances, 
    TypeFamilies , 
    ImpredicativeTypes, 
    FlexibleContexts, 
    DeriveDataTypeable,
    IncoherentInstances,
    ConstraintKinds,
    TypeOperators,
    GADTs
    #-}
module SizeTypes where
import Data.List
import Data.Typeable
import TypeLevelList
import Data.Ratio ((%))

\end{code}
}
\section{Numbers}\label{sec:SizeTypes}

This module implements fixed-width type-level decimal numbers, 
including constraints for non-zero numbers, and constraints for addition and multiplication of numbers.

The data constructors for digits, and instances of the \hask{Digit} class for each digit, which defines the function \hask{digit}, which represents
the numeric value of each digit.
Phantom types for digits.
\begin{code}
data D0 
data D1
data D2
data D3
data D4 
data D5  
data D6  
data D7  
data D8  
data D9  
\end{code}

Class for digits, including type-level to data-level conversion function \hask{digit}.
These instances identify the digit phantom types to the type system.
\begin{code}
class Digit a where
  digit :: a -> Integer

instance Digit D0 where
  digit _ = 0
instance Digit D1 where
   digit _ = 1
instance Digit D2 where
   digit _ = 2
instance Digit D3 where
   digit _ = 3
instance Digit D4 where
   digit _ = 4
instance Digit D5 where
   digit _ = 5
instance Digit D6 where
   digit _ = 6
instance Digit D7 where
   digit _ = 7
instance Digit D8 where
   digit _ = 8
instance Digit D9 where
   digit _ = 9
\end{code}

From digits we create fixed-width numbers.
\begin{code}
data SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0
data SIZE2 d1 d0
data SIZE3 d2 d1 d0
data SIZE4 d3 d2 d1 d0
data SIZE5 d4 d3 d2 d1 d0
data SIZE6 d5 d4 d3 d2 d1 d0
data SIZE7 d6 d5 d4 d3 d2 d1 d0
data SIZE8 d7 d6 d5 d4 d3 d2 d1 d0
data SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0
data SIZE10 d9 d8 d7 d6 d5 d4 d3 d2 d1 d0
data SIZE11 d10 d9 d8 d7 d6 d5 d4 d3 d2 d1 d0
data SIZE12 d11 d10 d9 d8 d7 d6 d5 d4 d3 d2 d1 d0

\end{code}

Defines the \hask{Size} class which creates the constraint that each phantom variable in a type level number is a digit.
Also defines the function \hask{toInt} which gives the numeric value of a type level number.
Type aliases for two special type level numbers, zero and one, are defined. Their sizes correspond with the width of the type level number.  
Instances are derived recursiely, except for \hask{SIZE2} where the value of each digit is calculated.
\begin{code}

class Size a where 
  type Zero a   
  type One  a
  toInt :: a -> Integer
  toInt' :: a -> Int
  toInt' = fromIntegral . toInt

instance (Digit d0, Digit d1, Digit d2, Digit d3, Digit d4, Digit d5, Digit d6, Digit d7, Digit d8, Digit d9) 
         => Size (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0 ) where
  type Zero (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0 ) = (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0 )
  type One  (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0 ) = (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D1 )
  toInt _ = toInt (undefined :: SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0) + 10^9 *(digit d9)
    where 
       d9 = undefined :: d9
            
instance (Digit d8, Digit d7, Digit d6, Digit d5, Digit d4, Digit d3, Digit d2, Digit d1, Digit d0)
         => Size (SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0) where
  type Zero (SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0) = (SIZE9 D0 D0 D0 D0 D0 D0 D0 D0 D0)
  type One  (SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0) = (SIZE9 D0 D0 D0 D0 D0 D0 D0 D0 D1)
  toInt _ = toInt (undefined :: SIZE8 d7 d6 d5 d4 d3 d2 d1 d0) + 10^8 *(digit d8)
    where
       d8 = undefined :: d8

instance (Digit d7, Digit d6, Digit d5, Digit d4, Digit d3, Digit d2, Digit d1, Digit d0)
         => Size (SIZE8 d7 d6 d5 d4 d3 d2 d1 d0) where
  type Zero (SIZE8 d7 d6 d5 d4 d3 d2 d1 d0) = (SIZE8 D0 D0 D0 D0 D0 D0 D0 D0)
  type One  (SIZE8 d7 d6 d5 d4 d3 d2 d1 d0) = (SIZE8 D0 D0 D0 D0 D0 D0 D0 D1)
  toInt _ = toInt (undefined :: SIZE7 d6 d5 d4 d3 d2 d1 d0) + 10^7 *(digit d7)
    where
       d7 = undefined :: d7

instance (Digit d6, Digit d5, Digit d4, Digit d3, Digit d2, Digit d1, Digit d0)
         => Size (SIZE7 d6 d5 d4 d3 d2 d1 d0) where
  type Zero (SIZE7 d6 d5 d4 d3 d2 d1 d0) = (SIZE7 D0 D0 D0 D0 D0 D0 D0)
  type One  (SIZE7 d6 d5 d4 d3 d2 d1 d0) = (SIZE7 D0 D0 D0 D0 D0 D0 D1)
  toInt _ = toInt (undefined :: SIZE6 d5 d4 d3 d2 d1 d0) + 10^6 *(digit d6)
    where
       d6 = undefined :: d6

instance (Digit d5, Digit d4, Digit d3, Digit d2, Digit d1, Digit d0)
         => Size (SIZE6 d5 d4 d3 d2 d1 d0) where
  type Zero (SIZE6 d5 d4 d3 d2 d1 d0) = (SIZE6 D0 D0 D0 D0 D0 D0)
  type One  (SIZE6 d5 d4 d3 d2 d1 d0) = (SIZE6 D0 D0 D0 D0 D0 D1)
  toInt _ = toInt (undefined :: SIZE5 d4 d3 d2 d1 d0) + 10^5 *(digit d5)
    where
       d5 = undefined :: d5

instance (Digit d4, Digit d3, Digit d2, Digit d1, Digit d0)
         => Size (SIZE5 d4 d3 d2 d1 d0) where
  type Zero (SIZE5 d4 d3 d2 d1 d0) = (SIZE5 D0 D0 D0 D0 D0)
  type One  (SIZE5 d4 d3 d2 d1 d0) = (SIZE5 D0 D0 D0 D0 D1)
  toInt _ = toInt (undefined :: SIZE4 d3 d2 d1 d0) + 10^4 *(digit d4)
    where
       d4 = undefined :: d4
       
instance (Digit d3, Digit d2, Digit d1, Digit d0) 
         => Size (SIZE4 d3 d2 d1 d0) where
  type Zero (SIZE4 d3 d2 d1 d0) = (SIZE4 D0 D0 D0 D0)   
  type One  (SIZE4 d3 d2 d1 d0) = (SIZE4 D0 D0 D0 D1)      
  toInt _ = toInt (undefined :: SIZE3 d2 d1 d0) + 10^3 * (digit d3)
    where 
       d3 = undefined :: d3

instance (Digit d2, Digit d1, Digit d0) 
         => Size (SIZE3 d2 d1 d0 ) where
  type Zero (SIZE3 d2 d1 d0) = (SIZE3 D0 D0 D0)
  type One  (SIZE3 d2 d1 d0) = (SIZE3 D0 D0 D1)      
  toInt _ = toInt (undefined :: SIZE2 d1 d0) + 10^2 *(digit d2)
    where 
       d2 = undefined :: d2
       
instance (Digit d1, Digit d0) 
         => Size (SIZE2 d1 d0 ) where
  type Zero (SIZE2 d1 d0) = (SIZE2 D0 D0)  
  type One  (SIZE2 d1 d0) = (SIZE2 D0 D1)      
  toInt _ = digit d0 + 10 *(digit d1)
    where 
       d0 = undefined :: d0
       d1 = undefined :: d1
       

\end{code}


We declare a class to indicate a digit is non-zero and create instances for single digit numbers, the simplest cases. With natural numbers and zero, we can create positive fractions,
but note that we cannot ensure that the fractions are in normal form.

\begin{code}

class NonZero a where
instance NonZero D1 where
instance NonZero D2 where
instance NonZero D3 where
instance NonZero D4 where
instance NonZero D5 where
instance NonZero D6 where
instance NonZero D7 where
instance NonZero D8 where
instance NonZero D9 where
\end{code}

Naturally, a non-zero number is one that is not equal to zero. In order to easily create instances for non-zero numbers of larger widths, we implement a class for comparing type-level numbers along with type level booleans. 
\begin{code}
class TypeEq x y b e

\end{code}
Type equivalence is trivial; we include it for completeness. 
\begin{code}

instance TypeEq x x TTrue e where 

\end{code}
We declare a class \hask{Fail} with no constructors; attempting to declare an instance with
\hask{Fail} in the constraints produces a compile-time error. We then declare two instances of \hask{TypeEq} 
with \hask{TFalse}. One instance contains two distinct type variables which match any two types 
(even types that are the same). Another instance is declared which has \hask{Fail} as the constraint 
and has two identical type variables. The compiler will always match the most specific instance 
first when using \hask{OverlappingInstances}; in this case, the instance containing the least amount 
of distinct type variables is matched first, so the compiler will match the first instance if 
and only if the types are not the same, and the second instance is not matched. 
Additionally, we define an empty data type \hask{TypeLevelNumIsZero} to use as a type 
arguement in order to clarify the error message recieved in the case of type check failure. 
If \hask{Fail} is used elsewhere the class implementing it can use a different data type as an ``error message''. 
\begin{code}

class Fail x

instance TypeEq x y TFalse e where 
instance (Fail e) => TypeEq x x TFalse e where 

data TypeLevelNumIsZero

class TypeIsEq a b 
instance TypeIsEq a a
instance (Fail ()) => TypeIsEq a b

class TypeNotEq a b
instance TypeNotEq a b
instance (Fail ()) => TypeNotEq a a

class TypeEq_ a b c | a b -> c
instance TypeEq_ a a TTrue
instance (Fail ()) => TypeEq_ c d TTrue



\end{code}

We create instances of \hask{NonZero} for type level numbers and fractions. The class constraint \hask{TypeEq (Zero n) n TFalse} indicates that the instance matches only those \hask{Size} types that are not equal to the \hask{Zero} type defined in the corresponding instance of \hask{Size}. 
\begin{code}

instance (Size n, zn ~ (Zero n), TypeEq zn n TFalse TypeLevelNumIsZero ) => NonZero n where

instance (Fraction (p,q), NonZero p) => NonZero (p,q) where

testTEQ :: (Size n, zn ~ (Zero n), TypeEq zn n TFalse ()) => n -> zn
testTEQ = undefined

testNonZero :: NonZero n => n
testNonZero = undefined

\end{code}


With natural numbers and zero, we can create positive fractions,
but note that we cannot insure that the fractions are in normal form.
\begin{code}
class Fraction a where
  toNumDenom :: a -> (Integer,Integer) 
  toFloating :: (Floating f) => a -> f
  toRatio :: a -> Rational

instance (Size p, Size q, NonZero q) => Fraction (p,q) where
  toNumDenom _ = (toInt (undefined :: p), toInt (undefined :: q))
  toFloating x = (fromIntegral a) / (fromIntegral b)
    where (a,b) = toNumDenom x
  toRatio x = (fromIntegral a) % (fromIntegral b)
    where (a,b) = toNumDenom x

\end{code}
Below are the definition for basic additon and multiplication. Meaningful arithmetic operations will be built from these.


The class \hask{Times}, which defines valid multiplication between two digit type level numbers.
The first two digits are the multiplicands; the second two digits are the product. 
\begin{code}
class Times da db high low | da db -> high, da db -> low where
instance Times D0 D0 D0 D0 where
instance Times D0 D1 D0 D0 where
instance Times D0 D2 D0 D0 where
instance Times D0 D3 D0 D0 where
instance Times D0 D4 D0 D0 where
instance Times D0 D5 D0 D0 where
instance Times D0 D6 D0 D0 where
instance Times D0 D7 D0 D0 where
instance Times D0 D8 D0 D0 where
instance Times D0 D9 D0 D0 where
instance Times D1 D0 D0 D0 where
instance Times D1 D1 D0 D1 where
instance Times D1 D2 D0 D2 where
instance Times D1 D3 D0 D3 where
instance Times D1 D4 D0 D4 where
instance Times D1 D5 D0 D5 where
instance Times D1 D6 D0 D6 where
instance Times D1 D7 D0 D7 where
instance Times D1 D8 D0 D8 where
instance Times D1 D9 D0 D9 where
instance Times D2 D0 D0 D0 where
instance Times D2 D1 D0 D2 where
instance Times D2 D2 D0 D4 where
instance Times D2 D3 D0 D6 where
instance Times D2 D4 D0 D8 where
instance Times D2 D5 D1 D0 where
instance Times D2 D6 D1 D2 where
instance Times D2 D7 D1 D4 where
instance Times D2 D8 D1 D6 where
instance Times D2 D9 D1 D8 where
instance Times D3 D0 D0 D0 where
instance Times D3 D1 D0 D3 where
instance Times D3 D2 D0 D6 where
instance Times D3 D3 D0 D9 where
instance Times D3 D4 D1 D2 where
instance Times D3 D5 D1 D5 where
instance Times D3 D6 D1 D8 where
instance Times D3 D7 D2 D1 where
instance Times D3 D8 D2 D4 where
instance Times D3 D9 D2 D7 where
instance Times D4 D0 D0 D0 where
instance Times D4 D1 D0 D4 where
instance Times D4 D2 D0 D8 where
instance Times D4 D3 D1 D2 where
instance Times D4 D4 D1 D6 where
instance Times D4 D5 D2 D0 where
instance Times D4 D6 D2 D4 where
instance Times D4 D7 D2 D8 where
instance Times D4 D8 D3 D2 where
instance Times D4 D9 D3 D6 where
instance Times D5 D0 D0 D0 where
instance Times D5 D1 D0 D5 where
instance Times D5 D2 D1 D0 where
instance Times D5 D3 D1 D5 where
instance Times D5 D4 D2 D0 where
instance Times D5 D5 D2 D5 where
instance Times D5 D6 D3 D0 where
instance Times D5 D7 D3 D5 where
instance Times D5 D8 D4 D0 where
instance Times D5 D9 D4 D5 where
instance Times D6 D0 D0 D0 where
instance Times D6 D1 D0 D6 where
instance Times D6 D2 D1 D2 where
instance Times D6 D3 D1 D8 where
instance Times D6 D4 D2 D4 where
instance Times D6 D5 D3 D0 where
instance Times D6 D6 D3 D6 where
instance Times D6 D7 D4 D2 where
instance Times D6 D8 D4 D8 where
instance Times D6 D9 D5 D4 where
instance Times D7 D0 D0 D0 where
instance Times D7 D1 D0 D7 where
instance Times D7 D2 D1 D4 where
instance Times D7 D3 D2 D1 where
instance Times D7 D4 D2 D8 where
instance Times D7 D5 D3 D5 where
instance Times D7 D6 D4 D2 where
instance Times D7 D7 D4 D9 where
instance Times D7 D8 D5 D6 where
instance Times D7 D9 D6 D3 where
instance Times D8 D0 D0 D0 where
instance Times D8 D1 D0 D8 where
instance Times D8 D2 D1 D6 where
instance Times D8 D3 D2 D4 where
instance Times D8 D4 D3 D2 where
instance Times D8 D5 D4 D0 where
instance Times D8 D6 D4 D8 where
instance Times D8 D7 D5 D6 where
instance Times D8 D8 D6 D4 where
instance Times D8 D9 D7 D2 where
instance Times D9 D0 D0 D0 where
instance Times D9 D1 D0 D9 where
instance Times D9 D2 D1 D8 where
instance Times D9 D3 D2 D7 where
instance Times D9 D4 D3 D6 where
instance Times D9 D5 D4 D5 where
instance Times D9 D6 D5 D4 where
instance Times D9 D7 D6 D3 where
instance Times D9 D8 D7 D2 where
instance Times D9 D9 D8 D1 where
\end{code}


The class \hask{Add2} defines addition of two one digit numbers. 
\begin{code}
class Add2 a b sh sl | a b -> sh, a b -> sl, sl sh a -> b, sl sh b -> a where
instance Add2 D0 D0 D0 D0
instance Add2 D1 D0 D0 D1
instance Add2 D2 D0 D0 D2
instance Add2 D3 D0 D0 D3
instance Add2 D4 D0 D0 D4
instance Add2 D5 D0 D0 D5
instance Add2 D6 D0 D0 D6
instance Add2 D7 D0 D0 D7
instance Add2 D8 D0 D0 D8
instance Add2 D9 D0 D0 D9
instance Add2 D0 D1 D0 D1
instance Add2 D1 D1 D0 D2
instance Add2 D2 D1 D0 D3
instance Add2 D3 D1 D0 D4
instance Add2 D4 D1 D0 D5
instance Add2 D5 D1 D0 D6
instance Add2 D6 D1 D0 D7
instance Add2 D7 D1 D0 D8
instance Add2 D8 D1 D0 D9
instance Add2 D9 D1 D1 D0
instance Add2 D0 D2 D0 D2
instance Add2 D1 D2 D0 D3
instance Add2 D2 D2 D0 D4
instance Add2 D3 D2 D0 D5
instance Add2 D4 D2 D0 D6
instance Add2 D5 D2 D0 D7
instance Add2 D6 D2 D0 D8
instance Add2 D7 D2 D0 D9
instance Add2 D8 D2 D1 D0
instance Add2 D9 D2 D1 D1
instance Add2 D0 D3 D0 D3
instance Add2 D1 D3 D0 D4
instance Add2 D2 D3 D0 D5
instance Add2 D3 D3 D0 D6
instance Add2 D4 D3 D0 D7
instance Add2 D5 D3 D0 D8
instance Add2 D6 D3 D0 D9
instance Add2 D7 D3 D1 D0
instance Add2 D8 D3 D1 D1
instance Add2 D9 D3 D1 D2
instance Add2 D0 D4 D0 D4
instance Add2 D1 D4 D0 D5
instance Add2 D2 D4 D0 D6
instance Add2 D3 D4 D0 D7
instance Add2 D4 D4 D0 D8
instance Add2 D5 D4 D0 D9
instance Add2 D6 D4 D1 D0
instance Add2 D7 D4 D1 D1
instance Add2 D8 D4 D1 D2
instance Add2 D9 D4 D1 D3
instance Add2 D0 D5 D0 D5
instance Add2 D1 D5 D0 D6
instance Add2 D2 D5 D0 D7
instance Add2 D3 D5 D0 D8
instance Add2 D4 D5 D0 D9
instance Add2 D5 D5 D1 D0
instance Add2 D6 D5 D1 D1
instance Add2 D7 D5 D1 D2
instance Add2 D8 D5 D1 D3
instance Add2 D9 D5 D1 D4
instance Add2 D0 D6 D0 D6
instance Add2 D1 D6 D0 D7
instance Add2 D2 D6 D0 D8
instance Add2 D3 D6 D0 D9
instance Add2 D4 D6 D1 D0
instance Add2 D5 D6 D1 D1
instance Add2 D6 D6 D1 D2
instance Add2 D7 D6 D1 D3
instance Add2 D8 D6 D1 D4
instance Add2 D9 D6 D1 D5
instance Add2 D0 D7 D0 D7
instance Add2 D1 D7 D0 D8
instance Add2 D2 D7 D0 D9
instance Add2 D3 D7 D1 D0
instance Add2 D4 D7 D1 D1
instance Add2 D5 D7 D1 D2
instance Add2 D6 D7 D1 D3
instance Add2 D7 D7 D1 D4
instance Add2 D8 D7 D1 D5
instance Add2 D9 D7 D1 D6
instance Add2 D0 D8 D0 D8
instance Add2 D1 D8 D0 D9
instance Add2 D2 D8 D1 D0
instance Add2 D3 D8 D1 D1
instance Add2 D4 D8 D1 D2
instance Add2 D5 D8 D1 D3
instance Add2 D6 D8 D1 D4
instance Add2 D7 D8 D1 D5
instance Add2 D8 D8 D1 D6
instance Add2 D9 D8 D1 D7
instance Add2 D0 D9 D0 D9
instance Add2 D1 D9 D1 D0
instance Add2 D2 D9 D1 D1
instance Add2 D3 D9 D1 D2
instance Add2 D4 D9 D1 D3
instance Add2 D5 D9 D1 D4
instance Add2 D6 D9 D1 D5
instance Add2 D7 D9 D1 D6
instance Add2 D8 D9 D1 D7
instance Add2 D9 D9 D1 D8




\end{code}
Classes defining the addition of $n$ single digit type level numbers, up to $n=20$.
\begin{code}
class Add3 a1 a2 a3 sh sl |  a1 a2 a3 -> sh, a1 a2 a3  -> sl, sl sh -> a1 a2 a3

instance (Add2 a1 a2 a1a2h a1a2l, Add2 a1a2l a3 a1a2la3h a1a2a3l, Add2 a1a2h a1a2la3h D0 a1a2a3h) => Add3 a1 a2 a3 a1a2a3h a1a2a3l where 

class Add4 a1 a2 a3 a4 sh sl | a1 a2 a3 a4 -> sh, a1 a2 a3 a4 -> sl, sl sh -> a1 a2 a3 a4 where
instance (Add3 a1 a2 a3 a1a2a3h a1a2a3l, Add2 a1a2a3l a4 a1a2a3la4h a1a2a3a4l, Add2 a1a2a3h a1a2a3la4h D0 a1a2a3a4h) => Add4 a1 a2 a3 a4 a1a2a3a4h a1a2a3a4l where 

class Add5 a1 a2 a3 a4 a5 sh sl | a1 a2 a3 a4 a5 -> sh, a1 a2 a3 a4 a5 -> sl where
instance (Add4 a1 a2 a3 a4 a1a2a3a4h a1a2a3a4l, Add2 a1a2a3a4l a5 a1a2a3a4la5h a1a2a3a4a5l, Add2 a1a2a3a4h a1a2a3a4la5h D0 a1a2a3a4a5h) => Add5 a1 a2 a3 a4 a5 a1a2a3a4a5h a1a2a3a4a5l where 

class Add6 a1 a2 a3 a4 a5 a6 sh sl | a1 a2 a3 a4 a5 a6 -> sh, a1 a2 a3 a4 a5 a6 -> sl where
instance (Add5 a1 a2 a3 a4 a5 a1a2a3a4a5h a1a2a3a4a5l, Add2 a1a2a3a4a5l a6 a1a2a3a4a5la6h a1a2a3a4a5a6l, Add2 a1a2a3a4a5h a1a2a3a4a5la6h D0 a1a2a3a4a5a6h) => Add6 a1 a2 a3 a4 a5 a6 a1a2a3a4a5a6h a1a2a3a4a5a6l where 

class Add7 a1 a2 a3 a4 a5 a6 a7 sh sl | a1 a2 a3 a4 a5 a6 a7 -> sh, a1 a2 a3 a4 a5 a6 a7 -> sl where
instance (Add6 a1 a2 a3 a4 a5 a6 a1a2a3a4a5a6h a1a2a3a4a5a6l, Add2 a1a2a3a4a5a6l a7 a1a2a3a4a5a6la7h a1a2a3a4a5a6a7l, Add2 a1a2a3a4a5a6h a1a2a3a4a5a6la7h D0 a1a2a3a4a5a6a7h) => Add7 a1 a2 a3 a4 a5 a6 a7 a1a2a3a4a5a6a7h a1a2a3a4a5a6a7l where 

class Add8 a1 a2 a3 a4 a5 a6 a7 a8 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 -> sl where
instance (Add7 a1 a2 a3 a4 a5 a6 a7 a1a2a3a4a5a6a7h a1a2a3a4a5a6a7l, Add2 a1a2a3a4a5a6a7l a8 a1a2a3a4a5a6a7la8h a1a2a3a4a5a6a7a8l, Add2 a1a2a3a4a5a6a7h a1a2a3a4a5a6a7la8h D0 a1a2a3a4a5a6a7a8h) => Add8 a1 a2 a3 a4 a5 a6 a7 a8 a1a2a3a4a5a6a7a8h a1a2a3a4a5a6a7a8l where 

class Add9 a1 a2 a3 a4 a5 a6 a7 a8 a9 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 -> sl where
instance (Add8 a1 a2 a3 a4 a5 a6 a7 a8 a1a2a3a4a5a6a7a8h a1a2a3a4a5a6a7a8l, Add2 a1a2a3a4a5a6a7a8l a9 a1a2a3a4a5a6a7a8la9h a1a2a3a4a5a6a7a8a9l, Add2 a1a2a3a4a5a6a7a8h a1a2a3a4a5a6a7a8la9h D0 a1a2a3a4a5a6a7a8a9h) => Add9 a1 a2 a3 a4 a5 a6 a7 a8 a9 a1a2a3a4a5a6a7a8a9h a1a2a3a4a5a6a7a8a9l where 

class Add10 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 -> sl where
instance (Add9 a1 a2 a3 a4 a5 a6 a7 a8 a9 a1a2a3a4a5a6a7a8a9h a1a2a3a4a5a6a7a8a9l, Add2 a1a2a3a4a5a6a7a8a9l a10 a1a2a3a4a5a6a7a8a9la10h a1a2a3a4a5a6a7a8a9a10l, Add2 a1a2a3a4a5a6a7a8a9h a1a2a3a4a5a6a7a8a9la10h D0 a1a2a3a4a5a6a7a8a9a10h) => Add10 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a1a2a3a4a5a6a7a8a9a10h a1a2a3a4a5a6a7a8a9a10l where 

class Add11 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 -> sl where
instance (Add10 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a1a2a3a4a5a6a7a8a9a10h a1a2a3a4a5a6a7a8a9a10l, Add2 a1a2a3a4a5a6a7a8a9a10l a11 a1a2a3a4a5a6a7a8a9a10la11h a1a2a3a4a5a6a7a8a9a10a11l, Add2 a1a2a3a4a5a6a7a8a9a10h a1a2a3a4a5a6a7a8a9a10la11h D0 a1a2a3a4a5a6a7a8a9a10a11h) => Add11 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a1a2a3a4a5a6a7a8a9a10a11h a1a2a3a4a5a6a7a8a9a10a11l where 

class Add12 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 -> sl where
instance (Add11 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a1a2a3a4a5a6a7a8a9a10a11h a1a2a3a4a5a6a7a8a9a10a11l, Add2 a1a2a3a4a5a6a7a8a9a10a11l a12 a1a2a3a4a5a6a7a8a9a10a11la12h a1a2a3a4a5a6a7a8a9a10a11a12l, Add2 a1a2a3a4a5a6a7a8a9a10a11h a1a2a3a4a5a6a7a8a9a10a11la12h D0 a1a2a3a4a5a6a7a8a9a10a11a12h) => Add12 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a1a2a3a4a5a6a7a8a9a10a11a12h a1a2a3a4a5a6a7a8a9a10a11a12l where 

class Add13 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 -> sl where
instance (Add12 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a1a2a3a4a5a6a7a8a9a10a11a12h a1a2a3a4a5a6a7a8a9a10a11a12l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12l a13 a1a2a3a4a5a6a7a8a9a10a11a12la13h a1a2a3a4a5a6a7a8a9a10a11a12a13l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12h a1a2a3a4a5a6a7a8a9a10a11a12la13h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13h) => Add13 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a1a2a3a4a5a6a7a8a9a10a11a12a13h a1a2a3a4a5a6a7a8a9a10a11a12a13l where 

class Add14 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 -> sl where
instance (Add13 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a1a2a3a4a5a6a7a8a9a10a11a12a13h a1a2a3a4a5a6a7a8a9a10a11a12a13l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13l a14 a1a2a3a4a5a6a7a8a9a10a11a12a13la14h a1a2a3a4a5a6a7a8a9a10a11a12a13a14l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13h a1a2a3a4a5a6a7a8a9a10a11a12a13la14h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14h) => Add14 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a1a2a3a4a5a6a7a8a9a10a11a12a13a14h a1a2a3a4a5a6a7a8a9a10a11a12a13a14l where 

class Add15 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 -> sl where
instance (Add14 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a1a2a3a4a5a6a7a8a9a10a11a12a13a14h a1a2a3a4a5a6a7a8a9a10a11a12a13a14l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14l a15 a1a2a3a4a5a6a7a8a9a10a11a12a13a14la15h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14h a1a2a3a4a5a6a7a8a9a10a11a12a13a14la15h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15h) => Add15 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15l where 

class Add16 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 -> sl where
instance (Add15 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15l a16 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15la16h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15la16h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16h) => Add16 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16l where 

class Add17 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 -> sl where
instance (Add16 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16l a17 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16la17h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16la17h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17h) => Add17 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17l where 

class Add18 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 -> sl where
instance (Add17 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17l a18 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17la18h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17la18h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18h) => Add18 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18l where 

class Add19 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 -> sl where
instance (Add18 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18l a19 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18la19h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18la19h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19h) => Add19 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19l where 

class Add20 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20 sh sl | a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20 -> sh, a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20 -> sl where
instance (Add19 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19l a20 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19la20h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19a20l, Add2 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19la20h D0 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19a20h) => Add20 a1 a2 a3 a4 a5 a6 a7 a8 a9 a10 a11 a12 a13 a14 a15 a16 a17 a18 a19 a20 a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19a20h a1a2a3a4a5a6a7a8a9a10a11a12a13a14a15a16a17a18a19a20l where 
\end{code}


The class \hask{MultD10} defines multiplication of two ten digit type level numbers. Equivalent classes exist for two and three digit numbers. Type level multiplication is defined through the principle of long multiplication, by building the product from additions, which are results of multiplying the first multiplicand by one digit of the second.
\begin{code}
class MultD10 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  g9 g8 g7 g6 g5 g4 g3 g2 g1 g0 
     | f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g9, 
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g8, 
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g7,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g6,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g5,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g4,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g3,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g2,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g1,
       f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  -> g0 where       
      
instance ( Times f0 e0 p00h p00l, Times f0 e1 p01h p01l, Times f0 e2 p02h p02l, Times f0 e3 p03h p03l,
           Times f0 e4 p04h p04l, Times f0 e5 p05h p05l, Times f0 e6 p06h p06l, Times f0 e7 p07h p07l, 
           Times f0 e8 p08h p08l, Times f0 e9 D0  p09l, Times f1 e0 p10h p10l, Times f1 e1 p11h p11l, 
           Times f1 e2 p12h p12l, Times f1 e3 p13h p13l, Times f1 e4 p14h p14l, Times f1 e5 p15h p15l, 
           Times f1 e6 p16h p16l, Times f1 e7 p17h p17l, Times f1 e8 D0  p18l, Times f1 e9 D0 D0 , 
           Times f2 e0 p20h p20l, Times f2 e1 p21h p21l, Times f2 e2 p22h p22l, Times f2 e3 p23h p23l,
           Times f2 e4 p24h p24l, Times f2 e5 p25h p25l, Times f2 e6 p26h p26l, Times f2 e7 D0  p27l,
           Times f2 e8 D0 D0 , Times f2 e9 D0 D0 , Times f3 e0 p30h p30l, Times f3 e1 p31h p31l, 
           Times f3 e2 p32h p32l, Times f3 e3 p33h p33l, Times f3 e4 p34h p34l, Times f3 e5 p35h p35l,
           Times f3 e6 D0  p36l, Times f3 e7 D0 D0 , Times f3 e8 D0 D0 , Times f3 e9 D0 D0 , Times f4 e0 p40h p40l,
           Times f4 e1 p41h p41l, Times f4 e2 p42h p42l, Times f4 e3 p43h p43l, Times f4 e4 p44h p44l,
           Times f4 e5 D0  p45l, Times f4 e6 D0 D0 , Times f4 e7 D0 D0 , Times f4 e8 D0 D0 , Times f4 e9 D0 D0 ,
           Times f5 e0 p50h p50l, Times f5 e1 p51h p51l, Times f5 e2 p52h p52l, Times f5 e3 p53h p53l,
           Times f5 e4 D0  p54l, Times f5 e5 D0 D0 , Times f5 e6 D0 D0 , Times f5 e7 D0 D0 , Times f5 e8 D0 D0 ,
           Times f5 e9 D0 D0 , Times f6 e0 p60h p60l, Times f6 e1 p61h p61l, Times f6 e2 p62h p62l, 
           Times f6 e3 D0  p63l, Times f6 e4 D0 D0 , Times f6 e5 D0 D0 , Times f6 e6 D0 D0 , Times f6 e7 D0 D0 ,
           Times f6 e8 D0 D0 , Times f6 e9 D0 D0 , Times f7 e0 p70h p70l, Times f7 e1 p71h p71l, 
           Times f7 e2 D0  p72l, Times f7 e3 D0 D0 , Times f7 e4 D0 D0 , Times f7 e5 D0 D0 , Times f7 e6 D0 D0 ,
           Times f7 e7 D0 D0 , Times f7 e8 D0 D0 , Times f7 e9 D0 D0 , Times f8 e0 p80h p80l, Times f8 e1 D0  p81l,
           Times f8 e2 D0 D0 , Times f8 e3 D0 D0 , Times f8 e4 D0 D0 , Times f8 e5 D0 D0 , Times f8 e6 D0 D0 , 
           Times f8 e7 D0 D0 , Times f8 e8 D0 D0 , Times f8 e9 D0 D0 , Times f9 e0 D0  p90l, Times f9 e1 D0 D0 , 
           Times f9 e2 D0 D0 , Times f9 e3 D0 D0 , Times f9 e4 D0 D0 , Times f9 e5 D0 D0 , Times f9 e6 D0 D0 , 
           Times f9 e7 D0 D0 , Times f9 e8 D0 D0 , Times f9 e9 D0 D0 ,
           Add4 p01l p10l p00h D0 c1h c1l,
           Add6 p02l p11l p20l p01h p10h c1h c2h c2l, 
           Add8 p03l p12l p21l p30l p02h p11h p20h c2h c3h c3l,
           Add10 p04l p13l p22l p31l p40l p03h p12h p21h p30h c3h c4h c4l,
           Add12 p05l p14l p23l p32l p41l p50l p04h p13h p22h p31h p40h c4h c5h c5l,
           Add14 p06l p15l p24l p33l p42l p51l p60l p05h p14h p23h p32h p41h p50h c5h c6h c6l, 
           Add16 p07l p16l p25l p34l p43l p52l p61l p70l p06h p15h p24h p33h p42h p51h p60h c6h c7h c7l,
           Add18 p08l p17l p26l p35l p44l p53l p62l p71l p80l p07h p16h p25h p34h p43h p52h p61h p70h c7h c8h c8l,
           Add20 p09l p18l p27l p36l p45l p54l p63l p72l p81l p90l p08h p17h p26h p35h p44h p53h p62h p71h p80h c8h D0 c9l
           
           )
           => MultD10 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  c9l c8l c7l c6l c5l c4l c3l c2l c1l p00l  where 

class MultD2 f1 f0  e1 e0  g1 g0 | f1 f0 e1 e0 -> g1, f1 f0 e1 e0 -> g0 where
instance ( Times f0 e0 p00h p00l, Times f1 e0 D0 p10l, Times f0 e1 D0 p01l, Times f1 e1 D0 D0
         , Add3 p00h p10l p01l D0 c1l) => MultD2 f1 f0 e1 e0 c1l p00l where 

class MultD3 f2 f1 f0  e2 e1 e0  g2 g1 g0 | f2 f1 f0 e2 e1 e0 -> g2, f2 f1 f0 e2 e1 e0 -> g1, f2 f1 f0 e2 e1 e0 -> g0 where
instance ( Times f0 e0 p00h p00l, Times f1 e0 p10h p10l, Times f2 e0 D0 p20l, Times f0 e1 p01h p01l, Times f1 e1 D0 p11l, Times f2 e1 D0 D0,
           Times f0 e2 D0 p02l, Times f1 e2 D0 D0, Times f2 e2 D0 D0, Add3 p00h p10l p01l c1h c1l, Add5 p20l p01h p11l p02l c1h D0 c2l)
           => MultD3 f2 f1 f0 e2 e1 e0 c2l c1l p00l where 
\end{code}
Note that we leave functional dependencies for the specific sizes,
and they only flow from multiplicands to product, not the other way, although we know they could
flow the other way with the addition of nonzero type constriants.



The class \hask{MultD} encapsulates multiplication of type level numbers of different lengths. The above classes which define valid multiplication for type level numbers of various lengths
are instances of this class. Note that the multiplication $F \times E = G$ will lead to an overflow in $G$ if $F$ and $E$ are too large. Overflows result in compile-time errors.

We only define multiplication instances for two sizes (ten digits for general use and three digits for examples).
\begin{code}
class MultD f e g | f e -> g where
instance (MultD10 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0  e9 e8 e7 e6 e5 e4 e3 e2 e1 e0  g9 g8 g7 g6 g5 g4 g3 g2 g1 g0) 
  => MultD (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0) (SIZE g9 g8 g7 g6 g5 g4 g3 g2 g1 g0) where 
instance (MultD3 f2 f1 f0  e2 e1 e0  g2 g1 g0) => MultD (SIZE3 f2 f1 f0) (SIZE3 e2 e1 e0) (SIZE3 g2 g1 g0) where 
instance (MultD2 f1 f0  e1 e0  g1 g0) => MultD (SIZE2 f1 f0) (SIZE2 e1 e0) (SIZE2 g1 g0) where 

instance (MultD a1 b1 c1, MultD a2 b2 c2) 
    => MultD (a1,a2) (b1,b2) (c1,c2) where


\end{code}
In order to simplify the multiplication of type level numbers, we create a helper class which 
constructs a width-ten number from any width number. We then describe the multiplcation of
two numbers, in general, as the multiplication of the two width-ten numbers created from the
original numbers by prepending zeroes.
\begin{code}
class Augment a b | a -> b
instance Augment (SIZE2 d1 d0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 d1 d0)
instance Augment (SIZE3 d2 d1 d0) (SIZE D0 D0 D0 D0 D0 D0 D0 d2 d1 d0)
instance Augment (SIZE4 d3 d2 d1 d0) (SIZE D0 D0 D0 D0 D0 D0 d3 d2 d1 d0)
instance Augment (SIZE5 d4 d3 d2 d1 d0) (SIZE D0 D0 D0 D0 D0 d4 d3 d2 d1 d0)
instance Augment (SIZE6 d5 d4 d3 d2 d1 d0) (SIZE D0 D0 D0 D0 d5 d4 d3 d2 d1 d0)
instance Augment (SIZE7 d6 d5 d4 d3 d2 d1 d0) (SIZE D0 D0 D0 d6 d5 d4 d3 d2 d1 d0)
instance Augment (SIZE8 d7 d6 d5 d4 d3 d2 d1 d0) (SIZE D0 D0 d7 d6 d5 d4 d3 d2 d1 d0)
instance Augment (SIZE9 d8 d7 d6 d5 d4 d3 d2 d1 d0) (SIZE D0 d8 d7 d6 d5 d4 d3 d2 d1 d0)
instance Augment (SIZE  d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) (SIZE  d9 d8 d7 d6 d5 d4 d3 d2 d1 d0)

instance (Augment a au_a, MultD (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) au_a r) 
    => MultD a (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) r
instance (Augment a au_a, MultD (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) au_a r) 
    => MultD (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) a r
instance (Augment a au_a, Augment b au_b, MultD au_a au_b r) => MultD a b r

\end{code}
Because multiplication by zero destroys information,
if we want to effectively use type inferencing, we need to also assert that 
multiplicands are nonzero, for which we have a special class.


\begin{code}
class MultDNonZero f e g | f e -> g, f g -> e, e g -> f where
instance (NonZero f, NonZero e, NonZero g, MultD e f g) => MultDNonZero e f g where
\end{code}


We define classes and for addition of two type level numbers of the same width. Addition of type level numbers is limited to those numbers whose sum contains no more digits than the original two numbers. 
\begin{code}
class Sum2 e1 e0 f1 f0 g1 g0 |
    e1 e0 f1 f0 -> g1,
    e1 e0 f1 f0 -> g0,
    f1 f0 g1 g0 -> e1,
    f1 f0 g1 g0 -> e0,
    f1 f0 e1 e0 -> f1,
    f1 f0 e1 e0 -> f0
instance (Add2 e0 f0 c1 g0, 
          Add3 f1 e1 c1  D0 g1) => Sum2 e1 e0 f1 f0 g1 g0
          
class Sum3 e2 e1 e0 f2 f1 f0 g2 g1 g0 | e2 e1 e0 f2 f1 f0 -> g0, e2 e1 e0 f2 f1 f0 -> g1, e2 e1 e0 f2 f1 f0 -> g2 where 
instance (Add2 e0 f0 c1 g0, 
          Add3 f1 e1 c1  c2 g1,
          Add3 f2 e2 c2  D0 g2) => Sum3 e2 e1 e0 f2 f1 f0 g2 g1 g0

class Sum4 e3 e2 e1 e0 f3 f2 f1 f0 g3 g2 g1 g0 | e3 e2 e1 e0 f3 f2 f1 f0 -> g3,e3 e2 e1 e0 f3 f2 f1 f0 -> g2,e3 e2 e1 e0 f3 f2 f1 f0 -> g1,e3 e2 e1 e0 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f3 e3 c3 D0 g3) => Sum4 e3 e2 e1 e0 f3 f2 f1 f0 g3 g2 g1 g0

class Sum5 e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 g4 g3 g2 g1 g0 | e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 -> g4,e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 -> g3,e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 -> g2,e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 -> g1,e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f4 e4 c4 D0 g4) => Sum5 e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 g4 g3 g2 g1 g0

class Sum6 e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 g5 g4 g3 g2 g1 g0 | e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g5,e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g4,e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g3,e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g2,e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g1,e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f4 e4 c4 c5 g4,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f5 e5 c5 D0 g5) => Sum6 e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 g5 g4 g3 g2 g1 g0

class Sum7 e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 g6 g5 g4 g3 g2 g1 g0 | e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g6,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g5,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g4,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g3,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g2,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g1,e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f5 e5 c5 c6 g5,Add3 f4 e4 c4 c5 g4,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f6 e6 c6 D0 g6) => Sum7 e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 g6 g5 g4 g3 g2 g1 g0

class Sum8 e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 g7 g6 g5 g4 g3 g2 g1 g0 | e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g7,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g6,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g5,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g4,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g3,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g2,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g1,e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f6 e6 c6 c7 g6,Add3 f5 e5 c5 c6 g5,Add3 f4 e4 c4 c5 g4,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f7 e7 c7 D0 g7) => Sum8 e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 g7 g6 g5 g4 g3 g2 g1 g0

class Sum9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 g8 g7 g6 g5 g4 g3 g2 g1 g0 | e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g8,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g7,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g6,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g5,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g4,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g3,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g2,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g1,e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g0 where
instance (Add2 e0 f0 c1 g0,Add3 f7 e7 c7 c8 g7,Add3 f6 e6 c6 c7 g6,Add3 f5 e5 c5 c6 g5,Add3 f4 e4 c4 c5 g4,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f8 e8 c8 D0 g8) => Sum9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 g8 g7 g6 g5 g4 g3 g2 g1 g0

--

class Sum10 e9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0 g9 g8 g7 g6 g5 g4 g3 g2 g1 g0 
    | e9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0 -> g9 g8 g7 g6 g5 g4 g3 g2 g1 g0 where
--      
instance (Add2 e0 f0 c1 g0,Add3 f8 e8 c8 c9 g8,Add3 f7 e7 c7 c8 g7,Add3 f6 e6 c6 c7 g6,Add3 f5 e5 c5 c6 g5,Add3 f4 e4 c4 c5 g4,Add3 f3 e3 c3 c4 g3,Add3 f2 e2 c2 c3 g2,Add3 f1 e1 c1 c2 g1,Add3 f9 e9 c9 D0 g9) => Sum10 e9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0 g9 g8 g7 g6 g5 g4 g3 g2 g1 g0


\end{code}
We define a superclass for addition of two type level numbers, of which all above classes are instances. 
\begin{code}
class Sum e f g | e f -> g, f g -> e, e g -> f where

instance (Sum2 e1 e0 f1 f0 g1 g0) => Sum (SIZE2 e1 e0) (SIZE2 f1 f0) (SIZE2 g1 g0) where 
instance (Sum3 e2 e1 e0 f2 f1 f0 g2 g1 g0) => Sum (SIZE3 e2 e1 e0) (SIZE3 f2 f1 f0) (SIZE3 g2 g1 g0) where
instance (Sum4 e3 e2 e1 e0 f3 f2 f1 f0 g3 g2 g1 g0) => Sum (SIZE4 e3 e2 e1 e0) (SIZE4 f3 f2 f1 f0) (SIZE4 g3 g2 g1 g0) where
instance (Sum5 e4 e3 e2 e1 e0 f4 f3 f2 f1 f0 g4 g3 g2 g1 g0) => Sum (SIZE5 e4 e3 e2 e1 e0) (SIZE5 f4 f3 f2 f1 f0) (SIZE5 g4 g3 g2 g1 g0) where
instance (Sum6 e5 e4 e3 e2 e1 e0 f5 f4 f3 f2 f1 f0 g5 g4 g3 g2 g1 g0) => Sum (SIZE6 e5 e4 e3 e2 e1 e0) (SIZE6 f5 f4 f3 f2 f1 f0) (SIZE6 g5 g4 g3 g2 g1 g0) where
instance (Sum7 e6 e5 e4 e3 e2 e1 e0 f6 f5 f4 f3 f2 f1 f0 g6 g5 g4 g3 g2 g1 g0) => Sum (SIZE7 e6 e5 e4 e3 e2 e1 e0) (SIZE7 f6 f5 f4 f3 f2 f1 f0) (SIZE7 g6 g5 g4 g3 g2 g1 g0) where
instance (Sum8 e7 e6 e5 e4 e3 e2 e1 e0 f7 f6 f5 f4 f3 f2 f1 f0 g7 g6 g5 g4 g3 g2 g1 g0) => Sum (SIZE8 e7 e6 e5 e4 e3 e2 e1 e0) (SIZE8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE8 g7 g6 g5 g4 g3 g2 g1 g0) where
instance (Sum9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f8 f7 f6 f5 f4 f3 f2 f1 f0 g8 g7 g6 g5 g4 g3 g2 g1 g0) => Sum (SIZE9 e8 e7 e6 e5 e4 e3 e2 e1 e0) (SIZE9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE9 g8 g7 g6 g5 g4 g3 g2 g1 g0) where
instance (Sum10 e9 e8 e7 e6 e5 e4 e3 e2 e1 e0 f9 f8 f7 f6 f5 f4 f3 f2 f1 f0 g9 g8 g7 g6 g5 g4 g3 g2 g1 g0) => Sum (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0) (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE g9 g8 g7 g6 g5 g4 g3 g2 g1 g0) where

\end{code}

The only way to garuntee the denominators are the same is to cross multiply. However this creates a high probability
of overflow, unfortunatley, and is very slow. 
\begin{code}
instance (Fraction (e1,e0), Fraction (d1,d0), Fraction (f1,f0),
          MultD e1 d0 q1, MultD d1 e0 r1, MultD e0 d0 f0, -- q1 and r1 are the resulting numerators
          Sum q1 r1 f1 
        ) =>
            Sum (e1,e0) (d1,d0) (f1,f0)
instance (Sum e1 d1 f1) => Sum (e1,n) (d1,n) (f1,n) 
\end{code}

A class using datatypes from TypeLevelList to measure the length of a list.
\begin{code}
class TLength a b | a -> b
instance TLength (x :$ T) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D1)
instance (TLength xs n1, Sum (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D1) n1 n) => TLength (x :$ xs) n
tLength :: TLength a b => a -> b
tLength = undefined
\end{code}


Example usage: sum of a list as a fold.
We only seperate \hask{x :$ xs} to get the zero.
\begin{code}
class TSum xs r | xs -> r
instance (TFold Sum (Zero x) (x :$ xs) r) => TSum (x :$ xs) r
tSum :: TSum xs r => xs -> r
tSum = undefined
\end{code}

We define some simple constructs which will be used to implement type level comparison. 
There is a data type for each ordering, as well as a class for comparison.
We define helper classes as well for readability.
\begin{code}

--class (:=:) a b r | a b -> r
--instance                     (:=:) a a TTrue
--instance                     (:=:) a b TFalse

class (:<:) a b r | a b -> r
instance (Compare a b LT) => (:<:) a b TTrue

--instance (Compare a b EQ) => (:<:) a b TFalse
--
--class (:>:) a b r | a b -> r
--instance ((b :<: a) r) => (:>:) a b r
--
--
class (:<=:) a b r | a b -> r
instance (Compare a b r, TypeEq r GT TFalse ()) 
                          => (a :<=: b) TTrue

class (:>=:) a b r | a b -> r
instance (Compare a b r, TypeEq r LT TFalse ()) 
                          => (a :>=: b) TTrue


class TOrd a where 
    toStr :: a -> String
instance TOrd LT where 
    toStr _ = "LT"
instance TOrd EQ where 
    toStr _ = "EQ"
instance TOrd GT where 
    toStr _ = "GT"

data LT 
data EQ 
data GT 

class Compare a b r | a b -> r where 

\end{code}
\hask{a} is the comparator of the most significant digit, 
\hask{b} is the comparator of the rest of the number:
\begin{code}
class CombComp a b c | a b -> c 
instance CombComp LT z  LT
instance CombComp GT z  GT
instance CombComp EQ LT LT
instance CombComp EQ GT GT
instance CombComp EQ EQ EQ

\end{code}
\hask{Compare} is defined recursively for \hask{Size} types.
The \hask{SIZE2} instance is special because it compares only \hask{Digit}s.
All other instances will compare the first digits of the two numbers,
and the numbers which are made of the rest of the digits.
The result is deduced by combined the two comparisons according to the above rules.
\begin{code}
instance (Compare d1 e1 c1, Compare d0 e0 c0, CombComp c1 c0 c) => 
    Compare (SIZE2 d1 d0) (SIZE2 e1 e0) c

instance (Compare d0 e0 c0, Compare (SIZE2 d1 d2) (SIZE2 e1 e2) c1, CombComp c0 c1 c) =>
    Compare (SIZE3 d0 d1 d2) (SIZE3 e0 e1 e2) c

instance (Compare d0 e0 c0, Compare (SIZE3 d1 d2 d3) (SIZE3 e1 e2 e3) c1, CombComp c0 c1 c) =>
    Compare (SIZE4 d0 d1 d2 d3) (SIZE4 e0 e1 e2 e3) c

instance (Compare d0 e0 c0, Compare (SIZE4 d1 d2 d3 d4) (SIZE4 e1 e2 e3 e4) c1, CombComp c0 c1 c) =>
    Compare (SIZE5 d0 d1 d2 d3 d4) (SIZE5 e0 e1 e2 e3 e4) c

instance (Compare d0 e0 c0, Compare (SIZE5 d1 d2 d3 d4 d5) (SIZE5 e1 e2 e3 e4 e5) c1, CombComp c0 c1 c) =>
    Compare (SIZE6 d0 d1 d2 d3 d4 d5) (SIZE6 e0 e1 e2 e3 e4 e5) c

instance (Compare d0 e0 c0, Compare (SIZE6 d1 d2 d3 d4 d5 d6) (SIZE6 e1 e2 e3 e4 e5 e6) c1, CombComp c0 c1 c) =>
    Compare (SIZE7 d0 d1 d2 d3 d4 d5 d6) (SIZE7 e0 e1 e2 e3 e4 e5 e6) c

instance (Compare d0 e0 c0, Compare (SIZE7 d1 d2 d3 d4 d5 d6 d7) (SIZE7 e1 e2 e3 e4 e5 e6 e7) c1, CombComp c0 c1 c) =>
    Compare (SIZE8 d0 d1 d2 d3 d4 d5 d6 d7) (SIZE8 e0 e1 e2 e3 e4 e5 e6 e7) c

instance (Compare d0 e0 c0, Compare (SIZE8 d1 d2 d3 d4 d5 d6 d7 d8) (SIZE8 e1 e2 e3 e4 e5 e6 e7 e8) c1, CombComp c0 c1 c) =>
    Compare (SIZE9 d0 d1 d2 d3 d4 d5 d6 d7 d8) (SIZE9 e0 e1 e2 e3 e4 e5 e6 e7 e8) c

instance (Compare d0 e0 c0, Compare (SIZE9 d1 d2 d3 d4 d5 d6 d7 d8 d9) (SIZE9 e1 e2 e3 e4 e5 e6 e7 e8 e9) c1, CombComp c0 c1 c) =>
    Compare (SIZE d0 d1 d2 d3 d4 d5 d6 d7 d8 d9) (SIZE e0 e1 e2 e3 e4 e5 e6 e7 e8 e9) c

\end{code}

Comparison of fractions involves cross multiplication to normalize the fractions.
Note that this has a very high chance of causing overflow so we first augment the fractions.
\begin{code}
instance (Size d1, Size d0, Size e1, Size e0,
          Augment d1 d1a, Augment d0 d0a, Augment e1 e1a, Augment e0 e0a, MultD d0a e1a t0, MultD e0a d1a t1,
          Compare t0 t1 eq)
    => Compare (d0,d1) (e0,e1) eq

instance Compare D0 D0 EQ
instance Compare D0 D1 LT
instance Compare D0 D2 LT
instance Compare D0 D3 LT
instance Compare D0 D4 LT
instance Compare D0 D5 LT
instance Compare D0 D6 LT
instance Compare D0 D7 LT
instance Compare D0 D8 LT
instance Compare D0 D9 LT
instance Compare D1 D0 GT
instance Compare D1 D1 EQ
instance Compare D1 D2 LT
instance Compare D1 D3 LT
instance Compare D1 D4 LT
instance Compare D1 D5 LT
instance Compare D1 D6 LT
instance Compare D1 D7 LT
instance Compare D1 D8 LT
instance Compare D1 D9 LT
instance Compare D2 D0 GT
instance Compare D2 D1 GT
instance Compare D2 D2 EQ
instance Compare D2 D3 LT
instance Compare D2 D4 LT
instance Compare D2 D5 LT
instance Compare D2 D6 LT
instance Compare D2 D7 LT
instance Compare D2 D8 LT
instance Compare D2 D9 LT
instance Compare D3 D0 GT
instance Compare D3 D1 GT
instance Compare D3 D2 GT
instance Compare D3 D3 EQ
instance Compare D3 D4 LT
instance Compare D3 D5 LT
instance Compare D3 D6 LT
instance Compare D3 D7 LT
instance Compare D3 D8 LT
instance Compare D3 D9 LT
instance Compare D4 D0 GT
instance Compare D4 D1 GT
instance Compare D4 D2 GT
instance Compare D4 D3 GT
instance Compare D4 D4 EQ
instance Compare D4 D5 LT
instance Compare D4 D6 LT
instance Compare D4 D7 LT
instance Compare D4 D8 LT
instance Compare D4 D9 LT
instance Compare D5 D0 GT
instance Compare D5 D1 GT
instance Compare D5 D2 GT
instance Compare D5 D3 GT
instance Compare D5 D4 GT
instance Compare D5 D5 EQ
instance Compare D5 D6 LT
instance Compare D5 D7 LT
instance Compare D5 D8 LT
instance Compare D5 D9 LT
instance Compare D6 D0 GT
instance Compare D6 D1 GT
instance Compare D6 D2 GT
instance Compare D6 D3 GT
instance Compare D6 D4 GT
instance Compare D6 D5 GT
instance Compare D6 D6 EQ
instance Compare D6 D7 LT
instance Compare D6 D8 LT
instance Compare D6 D9 LT
instance Compare D7 D0 GT
instance Compare D7 D1 GT
instance Compare D7 D2 GT
instance Compare D7 D3 GT
instance Compare D7 D4 GT
instance Compare D7 D5 GT
instance Compare D7 D6 GT
instance Compare D7 D7 EQ
instance Compare D7 D8 LT
instance Compare D7 D9 LT
instance Compare D8 D0 GT
instance Compare D8 D1 GT
instance Compare D8 D2 GT
instance Compare D8 D3 GT
instance Compare D8 D4 GT
instance Compare D8 D5 GT
instance Compare D8 D6 GT
instance Compare D8 D7 GT
instance Compare D8 D8 EQ
instance Compare D8 D9 LT
instance Compare D9 D0 GT
instance Compare D9 D1 GT
instance Compare D9 D2 GT
instance Compare D9 D3 GT
instance Compare D9 D4 GT
instance Compare D9 D5 GT
instance Compare D9 D6 GT
instance Compare D9 D7 GT
instance Compare D9 D8 GT
instance Compare D9 D9 EQ


\end{code}
Subtraction classes.
\begin{code}
class Sub2 a b c | a b -> c
instance Sub2 D0 D0 D0
instance Sub2 D0 D1 D9
instance Sub2 D0 D2 D8
instance Sub2 D0 D3 D7
instance Sub2 D0 D4 D6
instance Sub2 D0 D5 D5
instance Sub2 D0 D6 D4
instance Sub2 D0 D7 D3
instance Sub2 D0 D8 D2
instance Sub2 D0 D9 D1
instance Sub2 D1 D0 D1
instance Sub2 D1 D1 D0
instance Sub2 D1 D2 D9
instance Sub2 D1 D3 D8
instance Sub2 D1 D4 D7
instance Sub2 D1 D5 D6
instance Sub2 D1 D6 D5
instance Sub2 D1 D7 D4
instance Sub2 D1 D8 D3
instance Sub2 D1 D9 D2
instance Sub2 D2 D0 D2
instance Sub2 D2 D1 D1
instance Sub2 D2 D2 D0
instance Sub2 D2 D3 D9
instance Sub2 D2 D4 D8
instance Sub2 D2 D5 D7
instance Sub2 D2 D6 D6
instance Sub2 D2 D7 D5
instance Sub2 D2 D8 D4
instance Sub2 D2 D9 D3
instance Sub2 D3 D0 D3
instance Sub2 D3 D1 D2
instance Sub2 D3 D2 D1
instance Sub2 D3 D3 D0
instance Sub2 D3 D4 D9
instance Sub2 D3 D5 D8
instance Sub2 D3 D6 D7
instance Sub2 D3 D7 D6
instance Sub2 D3 D8 D5
instance Sub2 D3 D9 D4
instance Sub2 D4 D0 D4
instance Sub2 D4 D1 D3
instance Sub2 D4 D2 D2
instance Sub2 D4 D3 D1
instance Sub2 D4 D4 D0
instance Sub2 D4 D5 D9
instance Sub2 D4 D6 D8
instance Sub2 D4 D7 D7
instance Sub2 D4 D8 D6
instance Sub2 D4 D9 D5
instance Sub2 D5 D0 D5
instance Sub2 D5 D1 D4
instance Sub2 D5 D2 D3
instance Sub2 D5 D3 D2
instance Sub2 D5 D4 D1
instance Sub2 D5 D5 D0
instance Sub2 D5 D6 D9
instance Sub2 D5 D7 D8
instance Sub2 D5 D8 D7
instance Sub2 D5 D9 D6
instance Sub2 D6 D0 D6
instance Sub2 D6 D1 D5
instance Sub2 D6 D2 D4
instance Sub2 D6 D3 D3
instance Sub2 D6 D4 D2
instance Sub2 D6 D5 D1
instance Sub2 D6 D6 D0
instance Sub2 D6 D7 D9
instance Sub2 D6 D8 D8
instance Sub2 D6 D9 D7
instance Sub2 D7 D0 D7
instance Sub2 D7 D1 D6
instance Sub2 D7 D2 D5
instance Sub2 D7 D3 D4
instance Sub2 D7 D4 D3
instance Sub2 D7 D5 D2
instance Sub2 D7 D6 D1
instance Sub2 D7 D7 D0
instance Sub2 D7 D8 D9
instance Sub2 D7 D9 D8
instance Sub2 D8 D0 D8
instance Sub2 D8 D1 D7
instance Sub2 D8 D2 D6
instance Sub2 D8 D3 D5
instance Sub2 D8 D4 D4
instance Sub2 D8 D5 D3
instance Sub2 D8 D6 D2
instance Sub2 D8 D7 D1
instance Sub2 D8 D8 D0
instance Sub2 D8 D9 D9
instance Sub2 D9 D0 D9
instance Sub2 D9 D1 D8
instance Sub2 D9 D2 D7
instance Sub2 D9 D3 D6
instance Sub2 D9 D4 D5
instance Sub2 D9 D5 D4
instance Sub2 D9 D6 D3
instance Sub2 D9 D7 D2
instance Sub2 D9 D8 D1
instance Sub2 D9 D9 D0

class CompToDig a b | a -> b
instance CompToDig GT D0
instance CompToDig LT D1
instance CompToDig EQ D0


tnum0 = undefined :: (SIZE D0 D0 D0 D3 D0 D0 D0 D1 D3 D2)
tnum1 = undefined :: (SIZE D0 D0 D0 D0 D0 D2 D0 D0 D4 D4)
tnum2 = undefined :: (SIZE D0 D0 D0 D0 D0 D2 D0 D1 D2 D3)
tnum3 = undefined :: (SIZE D0 D0 D0 D0 D0 D0 D2 D1 D2 D3)
tnum4 = undefined :: (SIZE D0 D9 D9 D9 D9 D9 D9 D9 D9 D9)

class Sign s where toVal :: s -> Integer
data (:+) 
data (:-)
instance Sign (:+) where toVal _ = 1
instance Sign (:-) where toVal _ = (-1)

data SSIZE s n
data SFRAC s n 

class SSize s 
instance (Sign s, Size n) => SSize (SSIZE s n) 
class SFrac s
instance (Sign s, Fraction n) => SFrac (SFRAC s n) 

class If a b c d | a b c -> d
instance (InvAddSafe a b c) => If GT a b (SSIZE (:+) c)
instance (InvAddSafe b a c) => If LT a b (SSIZE (:-) c)

class InvAddSafe a b c | a b -> c
instance InvAddSafe (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance InvAddSafe f (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) f
instance InvAddSafe f f (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance ( fn ~ (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0),
           en ~ (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0),
           
           fl ~ (f9:$f8:$f7:$f6:$f5:$f4:$f3:$f2:$f1:$f0:$T),
           el ~ (e9:$e8:$e7:$e6:$e5:$e4:$e3:$e2:$e1:$e0:$T),
           
           (fn `Compare` en) GT,
           
           TZipWith Sub2 fl el rl, rl ~ (r9:$r8:$r7:$r6:$r5:$r4:$r3:$r2:$r1:$r0:$T),
           TZipWith Compare fl el cl,
           TMap CompToDig cl nl, nl ~ (n9:$n8:$n7:$n6:$n5:$n4:$n3:$n2:$n1:$n0:$T), 
           InvAddSafe (SIZE r9 r8 r7 r6 r5 r4 r3 r2 r1 r0) (SIZE n8 n7 n6 n5 n4 n3 n2 n1 n0 D0) result 
           
    ) => InvAddSafe fn en result 
    
class InvAdd a b c | a b -> c
instance InvAdd (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance InvAdd f (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) f
instance InvAdd f f (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance ((fn `Compare` en) r,
           If r fn en result
    ) => InvAdd fn en result 

class Subtract a b c | a b -> c
instance (Augment e ea, Augment f fa, InvAdd ea fa g) => Subtract e f g
instance (Fraction (e1,e0), Fraction (d1,d0), Fraction (f1,f0),
          MultD e1 d0 q1, MultD d1 e0 r1, MultD e0 d0 f0, 
          Subtract q1 r1 f1 
        ) =>
            Subtract (e1,e0) (d1,d0) (f1,f0)

tInvAdd :: Subtract a b c => a -> b -> c
tInvAdd = undefined

\end{code}
Altenate sum classes.
\begin{code}

class Add2_ a b c | a b -> c
instance (Add2 a b c d) => Add2_ a b (c,d)

class SplitTuple a b | a -> b
instance SplitTuple T (T,T)
instance (SplitTuple xs (ax, bx)) => SplitTuple ((a,b) :$ xs) ((a :$ ax), (b :$ bx))

class Sum_ a b c | a b -> c
instance Sum_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance Sum_ (SIZE r9 r8 r7 r6 r5 r4 r3 r2 r1 r0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE r9 r8 r7 r6 r5 r4 r3 r2 r1 r0)
instance Sum_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE r9 r8 r7 r6 r5 r4 r3 r2 r1 r0) (SIZE r9 r8 r7 r6 r5 r4 r3 r2 r1 r0)

instance ( fn ~ (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0),
           en ~ (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0),
           
           fl ~ (f9:$f8:$f7:$f6:$f5:$f4:$f3:$f2:$f1:$f0:$T),
           el ~ (e9:$e8:$e7:$e6:$e5:$e4:$e3:$e2:$e1:$e0:$T),
           
           
           TZipWith Add2_ fl el tl,  
           SplitTuple tl (highDigs, lowDigs), 
           highDigs ~ (h9:$h8:$h7:$h6:$h5:$h4:$h3:$h2:$h1:$h0:$T),
           lowDigs ~ (l9:$l8:$l7:$l6:$l5:$l4:$l3:$l2:$l1:$l0:$T), 
           
           shifted ~ (SIZE h8 h7 h6 h5 h4 h3 h2 h1 h0 D0),
           lowDigsNum ~ (SIZE l9 l8 l7 l6 l5 l4 l3 l2 l1 l0),
           Sum_ shifted lowDigsNum result 
           
    ) => Sum_ fn en result
instance (Fraction (e1,e0), Fraction (d1,d0), Fraction (f1,f0),
          MultD e1 d0 q1, MultD d1 e0 r1, MultD e0 d0 f0, 
          Sum_ q1 r1 f1 
        ) =>
            Sum_ (e1,e0) (d1,d0) (f1,f0)
\end{code}
This instance will greatly increase speed on the offchance d1 == d2.
\begin{code}
instance (Sum_ e1 d1 f1) => Sum_ (e1,n) (d1,n) (f1,n) 

tSum_ :: Sum_ a b c => a -> b -> c
tSum_ = undefined



\end{code}
Alternate multiplication classes.
\begin{code}
class Times_ a b c | a b -> c
instance (Times a b c d) => Times_ a b (c,d)

class MultHelper num dig r | num dig -> r
instance MultHelper (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) D1 (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0)
instance MultHelper (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) D0 (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance (TMap (Times_ d) (f9:$f8:$f7:$f6:$f5:$f4:$f3:$f2:$f1:$f0:$T) rl,
          rl ~ ((D0, r91):$(r80, r81):$(r70, r71):$(r60, r61):$(r50, r51):$(r40, r41):$(r30, r31):$(r20, r21):$(r10, r11):$(r00, r01):$T),
          nums ~ 
           ((SIZE r91 D0   D0  D0  D0  D0  D0  D0  D0  D0) :$
            (SIZE r80 r81  D0  D0  D0  D0  D0  D0  D0  D0) :$
            (SIZE D0  r70 r71  D0  D0  D0  D0  D0  D0  D0) :$
            (SIZE D0  D0  r60 r61  D0  D0  D0  D0  D0  D0) :$
            (SIZE D0  D0  D0  r50 r51  D0  D0  D0  D0  D0) :$
            (SIZE D0  D0  D0  D0  r40 r41  D0  D0  D0  D0) :$
            (SIZE D0  D0  D0  D0  D0  r30 r31  D0  D0  D0) :$
            (SIZE D0  D0  D0  D0  D0  D0  r20 r21  D0  D0) :$
            (SIZE D0  D0  D0  D0  D0  D0  D0  r10 r11  D0) :$
            (SIZE D0  D0  D0  D0  D0  D0  D0  D0  r00 r01) :$ T),
          TFold Sum_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) nums result
          ) =>
    MultHelper (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) d result
\end{code}
The constraint \hask{f9 ~ D0} is what prevents overflow.
\begin{code}
class LShift a b | a -> b
instance (f9 ~ D0) => LShift (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE f8 f7 f6 f5 f4 f3 f2 f1 f0 D0)

class Shifts a b | a -> b
instance Shifts T T
instance (TMap LShift (x :$ xs) (r :$ rs), Shifts rs rss) => Shifts (x :$ xs) (r :$ rss)

class Mult_ a b c | a b -> c
instance Mult_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance Mult_ (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance Mult_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0)
instance (TMap (MultHelper (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0)) (e9:$e8:$e7:$e6:$e5:$e4:$e3:$e2:$e1:$e0:$T) t0,
          TReverse t0 (t1 :$ t1s), 
          Shifts t1s t2s, 
          TFold Sum_ (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D0) (t1 :$ t2s) r, result ~ r
    ) => Mult_ (SIZE f9 f8 f7 f6 f5 f4 f3 f2 f1 f0) (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0) result
    
instance (Mult_ a1 b1 c1, Mult_ a0 b0 c0) => Mult_ (a1,a0) (b1,b0) (c1,c0)

instance (Augment a au, Augment b bu, Mult_ au bu c) => Mult_ a b c


tMult_ :: Mult_ a b c => a -> b -> c
tMult_ = undefined

tMult :: MultD a b c => a -> b -> c
tMult = undefined




\end{code}
Show instances for anything that it is useful to have a show instance of.
\begin{code}
instance (Size s, s ~ (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0)) => Show (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0) where
    show _ = (filter (not . (`elem` "D ")) $ show $ toInt s )
        where s = undefined :: s

instance (Fraction (s,t), Size s, Size t, s ~ (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0), t ~ (SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0)) 
    => Show (SIZE d9 d8 d7 d6 d5 d4 d3 d2 d1 d0, SIZE e9 e8 e7 e6 e5 e4 e3 e2 e1 e0) where
    show _ = (filter (not . (`elem` "D ")) $ show $ toInt s ) ++ "/" ++ (filter (not . (`elem` "D ")) $ show $ toInt t )
        where s = undefined :: s
              t = undefined :: t

instance Show (:+) where show _ = "+"
instance Show (:-) where show _ = "-"
instance (Show sign, Show num) => Show (SSIZE sign num) where
    show _ = show sign ++ filter (`elem` (map toEnum [48 .. 57])) (show num)
        where sign = undefined :: sign
              num = undefined :: num
\end{code}




