This module contains definitions of frames.
\begin{code}
{-# LANGUAGE 
    FunctionalDependencies,
    FlexibleInstances,
    ScopedTypeVariables,
    UndecidableInstances,
    TypeFamilies ,
    FlexibleContexts,
    ConstraintKinds,
    GADTs,
    RankNTypes,
    ImpredicativeTypes,
    IncoherentInstances,
    TypeOperators,
    ConstraintKinds,
    InstanceSigs,
    TemplateHaskell,
    QuasiQuotes
    #-}

module Frame where

import SIUnits hiding ((+),(-),(*),negate, (/))
import SIUnits as U ((+),(-),(*),negate)
import NumericTypes hiding (TFalse, TTrue, ShowList, showL, ShowListH, showListH)
import FloatingPoint
import TypeGen
import UnitDefinitions
import TypeLevelList

import GHC.Prim (Constraint)
import Data.Complex
import Data.Array.IO
import Prelude hiding ((+), (-), negate, (*))
import qualified Prelude as P ((+), (-), negate, (*))
import Data.List



\end{code}
The basic class defintion for a frame of reference, containing a function to show the name of a frame, 
and a type alias for the base unit of the frame. The type of |BaseUnit| should be an instance of |SIUnitVal| with an empty tuple as a value.
\begin{code}

class NumSamplesC n where 
instance (Nat size, NonZero size) => NumSamplesC size where

class StepSizeC s where 
instance (FloatT size, NonZero size, Positive size) => StepSizeC size where

class (FList fl) => Frame f fl | f -> fl
class (DList dl) => Disc d dl | d -> dl

class (SIUnit (BaseUnit f)) => AtomicFrame f where
    nameA :: f -> String
    type BaseUnit f 

class (NumSamplesC (NumSamples d), StepSizeC (StepSize d)
  ) => AtomicDiscretization d where
    type NumSamples d
    type StepSize d
    
\end{code}
Two types of list are defined; one is a list of frames, and another a list of discretizations.
\begin{code}
class FList a
instance (FList b, AtomicFrame a) => FList (a :$ b)
instance FList T
class DList a
instance (DList b, AtomicDiscretization a) => DList (a :$ b)
instance DList T


-- test
data EmptyFrame
testf = undefined :: (DiffFrame, (SecondD :$ MeterXD :$ MeterYD :$ MeterZD :$ T))
test :: (s `SubFrameOf` f, Frame f fl) => s -> (f, fl) -> String
test = undefined
-------

\end{code}
Helper classes for constructing discretizations.
\begin{code}
class ElemOf a b
instance (TElem a b TTrue) => ElemOf a b

class SubFrameOf s f
instance (Frame f fl, s `ElemOf` fl) => SubFrameOf s f




\end{code}
The discretization datatype.
\begin{code}
data Discretization frame disc rangeU val where
    Discretization :: (SIUnit rangeU, Frame frame fl, Disc disc dl, {-this is bugged, cant call TLength directly-}TLengthH fl (D0 :$ T) l, TLengthH dl (D0 :$ T) l) => val ->
                         Discretization frame
                                        disc
                                        rangeU 
                                        val
                                        
instance (Show v) => Show (Discretization frame disc rangeU v) where
    show (Discretization v) = "Discretization: " ++ show v -- put more details


{- canal sample 1 -}
data CanalFrame
data CanalDisc
data MeterC

instance AtomicFrame MeterC where
    nameA _ = "Meter"
    type BaseUnit MeterC = Meter ()
instance AtomicDiscretization MeterC where
    type NumSamples MeterC = [nat| 12 |]
    type StepSize MeterC = [float| 1e-2 |]

instance Frame CanalFrame (MeterC :$ T)
instance Disc CanalDisc (MeterC :$ T)

canalSample1 :: Discretization CanalFrame CanalDisc (Meter ()) [Double]
canalSample1 = Discretization [1, 1.2, 1.3, 1.12, 1.23, 1.12, 1.15, 1.25, 1.18, 1.20, 1.24, 1.28]

{- lab sample 1 -}
data LabFrame
data LabDisc
data MeterL

instance AtomicFrame MeterL where
    nameA _ = "Meter"
    type BaseUnit MeterL = Meter ()
instance AtomicDiscretization MeterL where 
    type NumSamples MeterL = [nat| 4 |]
    type StepSize MeterL = [float| 2e-3 |]
   
instance Frame LabFrame (MeterL :$ T)
instance Disc LabDisc (MeterL :$ T)

meas1 :: Discretization LabFrame LabDisc (Meter ())  [Complex Double]
meas1 = Discretization [0,1,2,3]

{- canal sample 2 -}

canalSample2 :: Discretization CanalFrame CanalDisc (Meter ()) [Double]
canalSample2 = Discretization [1, 1.21, 1.2, 1.42, 1.3, 1.32, 1.12, 1.25, 1.23, 1.20, 1.12, 1.28]

{- diffusion frame -}
data SecondD
instance AtomicFrame SecondD where
    nameA _ = "Second"
    type BaseUnit SecondD = Second ()
instance AtomicDiscretization SecondD where
    type NumSamples SecondD = [nat| 500 |]
    type StepSize SecondD = [num| 1/256 |]
    
data MeterXD
instance AtomicFrame MeterXD where
    nameA _ = "MeterX"
    type BaseUnit MeterXD = Meter ()
instance AtomicDiscretization MeterXD where
    type NumSamples MeterXD = [nat| 32 |]
    type StepSize MeterXD = [num| 1/8 |]
   
data MeterYD
instance AtomicFrame MeterYD where
    nameA _ = "MeterY"
    type BaseUnit MeterYD = Meter ()
instance AtomicDiscretization MeterYD where
    type NumSamples MeterYD = [nat| 32 |]
    type StepSize MeterYD = [num| 1/8 |]
   
data MeterZD
instance AtomicFrame MeterZD where
    nameA _ = "MeterZ"
    type BaseUnit MeterZD = Meter ()
instance AtomicDiscretization MeterZD where
    type NumSamples MeterZD = [nat| 32 |]
    type StepSize MeterZD = [num| 1/8 |]
    
data DiffFrame
instance Frame DiffFrame (SecondD :$ MeterXD :$ MeterYD :$ MeterZD :$ T)
data DiffDisc
instance Disc DiffDisc (SecondD :$ MeterXD :$ MeterYD :$ MeterZD :$ T)

type Kelvin   val = SIUnitVal M0 Kg0 S0 A0 Mol0 K1 val

diffFrame1 :: Discretization DiffFrame DiffDisc (Kelvin ()) (IOUArray (Int, Int, Int) Int)
diffFrame1 = undefined

\end{code}

An instance for addition and subtraction of discretization is implemented.
\begin{code}
instance (Add val) => Add (Discretization frame disc rangeU val) where
    (Discretization a) + (Discretization b) = Discretization (a U.+ b)
    (Discretization a) - (Discretization b) = Discretization (a U.- b)
    negate (Discretization a) = Discretization (U.negate a)
  
  
\end{code}

Since |Mult| has a functional dependancy we can't define two instances of it with a discretization as the third arguement.
So we first define |MultND| in order to be able to multiply a discretization by a discretization, 
and a discretization by a scalar. 
We also need this class for multiplication of a list by a value.
\begin{code}
class MultND a b c | a b -> c where 
    ($*) :: a -> b -> c

{- currently it is ambiguous what this instance will be used for, will it be pointwise multiplication of discretizations or matrix mult? -}
instance (Mult v1 v2 v3, SIUnit u1, SIUnit u2, SIUnit u3, Mult u1 u2 u3, Frame f fl) 
   => MultND (Discretization f d u1 v1) (Discretization f d u2 v2) (Discretization f d u3 v3) where
    (Discretization a) $* (Discretization b) = Discretization (a U.* b)
--
instance (Mult v1 v2 v3, SIUnit u1, SIUnit u3, Mult u1 (SIUnitVal w1 w2 w3 w4 w5 w6 ()) u3, Frame f fl) 
   => MultND (Discretization f d u1 v1) (SIUnitVal w1 w2 w3 w4 w5 w6 v2) (Discretization f d u3 v3) where
   (Discretization a) $* (SIUnitVal b) = Discretization (a U.* b)

instance (Mult v1 v2 v3, SIUnit u1, SIUnit u3, Mult u1 (SIUnitVal w1 w2 w3 w4 w5 w6 ()) u3) 
   => MultND (SIUnitVal w1 w2 w3 w4 w5 w6 v2) (Discretization f d u1 v1) (Discretization f d u3 v3) where
    (SIUnitVal b) $* (Discretization a)= Discretization (a U.* b)
    
instance (Num a) => MultND a [a] [a] where
    a $* b = map (P.* a) b 
instance (Num a) => MultND [a] a [a] where
    a $* b = map (P.* b) a 

instance MultND a b c => Mult a b c where 
    (*) = ($*)

testmult :: Mult a b c => a -> b -> c
testmult a b = a U.* b

type Newton val      = SIUnitVal M1 Kg1 S_2 A0 Mol0 K0 val
testS = SIUnitVal 5.0 :: Newton (Complex Double)
{- meas1 U.* testS correctly infers units -}



instance Mult [Double] [Double] [Double] where
    (*) = zipWith (P.*)

\end{code}
These classes can be used as predicates on other things that only make sense with a discretizations of a certain dimension.
\begin{code}
class DiscretizationND d s
instance (Frame frame fl, TLength fl s ) => DiscretizationND (Discretization frame d rangeU val) s

class Discretization1D d
instance (DiscretizationND d (D1 :$ T) ) => Discretization1D d

class Discretization2D d
instance (DiscretizationND d (D2 :$ T)) => Discretization2D d

class Discretization3D d
instance (DiscretizationND d (D3 :$ T)) => Discretization3D d

class Discretization4D d
instance (DiscretizationND d (D4 :$ T)) => Discretization4D d

testC :: Discretization2D d => d -> String
testC = undefined

\end{code}

The differential class infers units and checks that the given atomic frame is a subframe of the frame of the discretization.
It would be most generalizable if the subframe contained information how to take the differential.
\begin{code}
class Differential d1 s d2 | d1 s -> d2 where
    diff :: d1 -> s -> d2
    diff = undefined
    
instance (AtomicFrame s, s `SubFrameOf` frame, Mult (BaseUnit s) sinv (Unitless ()), Mult r sinv nr) =>
    Differential (Discretization frame disc r v) s (Discretization frame disc nr v)

testdiff :: (Differential d1 s d2) => d1 -> s -> d2
testdiff = undefined

class Is a b c | a b -> c
instance Is a a TTrue
instance (Fail ()) => Is a b c

-- i is #rows, j is #cols
data Matrix i j vals where
    Matrix :: (TLength vals i, TMap TLength vals lengths, TMap (Is j) lengths bools, TAnd bools TTrue) 
              => i -> j -> vals -> Matrix i j vals

instance (TTranspose a b) => TTranspose (Matrix i j a) (Matrix j i b)

class ShowRow r where
    showRow :: r -> [String]
instance ShowRow T where
    showRow _ = []
instance (Show x, ShowRow xs) => ShowRow (x :$ xs) where
    showRow _ = (show x):(showRow xs)
        where x = undefined :: x
              xs = undefined :: xs

class ShowMat m where
    showMat :: m -> String
instance ShowMat T where
    showMat _ = ""
instance (ShowRow x, ShowMat xs) => ShowMat (x :$ xs) where
    showMat _ = intercalate ", " (showRow x) ++ "\r\n" ++ showMat xs
        where x = undefined :: x
              xs = undefined :: xs

instance (ShowMat mat) => Show (Matrix i j mat) where
    show _ = showMat (undefined :: mat)

v0 = [list| +3, +4, +9 |]
v1 = [list| +4, -11, +1 |]


m4 = Matrix [nat| 1 |] [nat| 1 |] [mat|
    1 |]

m0 = Matrix [nat| 2 |] [nat| 2 |] [mat|
    6. , 2. 
    4. , 4. |]
 -- {{6,2},{4,4}} * {{3,8,3},{2,1,4}}
m1 = Matrix [nat| 2 |] [nat| 3 |] [mat|
    3. , 8. , 3. 
    2. , 1. , 4. |]

--m3 = Matrix [nat| 10 |] [nat| 10 |] [mat|
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90
--    3, 6, 7, 23, 31, 32, 52, 22, 32, 90 |]


class TDot a b c | a b -> c
instance (TZipWith MultD a b t , t ~ (x :$ xs), GenericNumber x, TFold Sum (Zero x) t c
    ) => TDot a b c

tDot :: TDot a b c => a -> b -> c
tDot = undefined 


mmulth :: Num b => [[b]] -> [b] -> [b]
mmulth tval col = map (dot col) tval
dot x y = sum $ zipWith (P.*) x y


class MMultH tval col r | col tval -> r
instance (TMap (TDot col) tval r) => MMultH tval col r

class MMult m0 m1 mr | m0 m1 -> mr
--instance (TTranspose m1 m1t, TMap (MMultH m1t) m0 mr) =>
--    MMult (Matrix i z m0) (Matrix z j m1) (Matrix i j mr)
instance (TTranspose m1 m1t
         ,TMap (MMultH m1t) m0 mr
         
    ) => MMult (Matrix i z m0) (Matrix z j m1) (Matrix i j mr)    
    
tMMult :: MMult m0 m1 mr => m0 -> m1 -> mr
tMMult = undefined

tTranspose2 :: TTranspose a b => (Matrix i j a) -> b
tTranspose2 = undefined

mmult a b = [ [ sum $ zipWith (P.*) ar bc | bc <- transpose b ] | ar <- a ]
testmm = [[1, 2],[3, 4]] `mmult` [[3, 8, 3],[2,  1, 4]]


--det too slow look up better algorithms

--class Determinant m0 r | m0 -> r 
--instance (Augment a (SIZE D0 D0 D0 D0 D0 D0 D0 D0 D0 D2), MultDNonZero i00 i11 t0, MultDNonZero i01 i10 t1 , Sum r t1 t0
--    ) => 
--    Determinant (Matrix a a ((i00 :$ i01 :$ T) :$
--                             (i10 :$ i11 :$ T) :$ T)) (t0,t1,r)


--tDet :: Determinant m0 r => m0 -> r
--tDet = undefined
--
---- a - b = c => c + b = a
--testsub :: Add2 c b a0 a1 => (a0,a1) -> b -> c
--testsub = undefined
--
--tests2 :: Sum c b a => a -> b -> c
--tests2 = undefined


\end{code}
For every member there are 6 DOF, which produces a 6x6 matrix.
Each row has a different tangent vector space.
The member connects two nodes.
a0 a1 ... a5 
...
f0 f1 ... f5 

Attempting to sum rows in different vector spaces results in a type error. 

\begin{code}


--class TangentVectorSpace a b | a -> b where
class VectorSpace a 

--data NodeVector a v n where
--    NodeVector :: (Add a, TangentVectorSpace v0 v1, Nat n) => [a] -> v0 -> n -> NodeVector [a] v0 n

data NodeVector a v where
    NodeVector :: (Add a, VectorSpace v0) => a -> v0 -> NodeVector a v0

--data NodeVector a v where
--    NodeVector :: (Add a, TangentVectorSpace v0 v1) => a -> v0 -> NodeVector a v0

--instance (Show a) => Show (NodeVector a v n) where
--    show (NodeVector a v n) = "NodeVector " ++ show a

instance (Show a, Show v) => Show (NodeVector a v) where
    show (NodeVector a v) = "NodeVector " ++ show a ++ " " ++ show v


--data PlaneFrameVectorSpace

data DisplacementX = DisplacementX deriving Show
data DisplacementY = DisplacementY deriving Show
data Moment = Moment deriving Show

instance VectorSpace DisplacementX  
instance VectorSpace DisplacementY  
instance VectorSpace Moment         

--instance TangentVectorSpace DisplacementX PlaneFrameVectorSpace 
--instance TangentVectorSpace DisplacementY PlaneFrameVectorSpace 
--instance TangentVectorSpace Moment        PlaneFrameVectorSpace 

--instance (Add a, TangentVectorSpace v0 v1) => Add (NodeVector a v0 n) where
--    (NodeVector a v0 n0) + (NodeVector b v1 n1) = NodeVector (a + b) v0 n0
--    negate (NodeVector a v0 n) = NodeVector (negate a) v0 n
--    (NodeVector a v0 n0) - (NodeVector b v1 n1) = NodeVector (a - b) v0 n0

instance (Add a, VectorSpace v0) => Add (NodeVector a v0) where
    (NodeVector a v0) + (NodeVector b v1) = NodeVector (a + b) v0
    negate (NodeVector a v0) = NodeVector (negate a) v0
    (NodeVector a v0) - (NodeVector b v1) = NodeVector (a - b) v0

\end{code}
A single node has 3 dof in plane frame analysis. The node number of each vector must be the same.
For Add to be used the three rows must be in the same order.
\begin{code}
--data NodeData a b c where
--    NodeData :: NodeVector a0 v0 n -> NodeVector a1 v1 n -> NodeVector a2 v2 n -> NodeData (NodeVector a0 v0 n) (NodeVector a1 v1 n) (NodeVector a2 v2 n)


data NodeData n a b c where
    NodeData :: (Nat n) => n -> NodeVector a0 v0 -> NodeVector a1 v1 -> NodeVector a2 v2 -> NodeData n (NodeVector a0 v0) (NodeVector a1 v1) (NodeVector a2 v2)

instance (Show n, Nat n, Show a, Show b, Show c) => Show (NodeData n a b c) where
    show (NodeData n a b c) = "NodeData (node " ++ show n ++ ") " ++ "(" ++ intercalate ", " [show a, show b, show c] ++ ")"

instance (Add a, Add b, Add c) => Add (NodeData n a b c) where
    (NodeData n0 a0 b0 c0) + (NodeData n1 a1 b1 c1) = NodeData n0 (a0+a1) (b0+b1) (c0+c1)
    (NodeData n0 a0 b0 c0) - (NodeData n1 a1 b1 c1) = NodeData n0 (a0-a1) (b0-b1) (c0-c1)
    negate (NodeData n a b c) = NodeData n (negate a) (negate b) (negate c)

\end{code}
This class makes sure that MemberMatrix can only be called on a list of NodeData.
\begin{code}
class NodeDataC a b | a -> b
instance NodeDataC (NodeData n a b c) TTrue

data LocalStiffnessMatrix a b where
    LocalStiffnessMatrix :: (NodeData n0 a0 b0 c0) -> (NodeData n1 a1 b1 c1) -> LocalStiffnessMatrix (NodeData n0 a0 b0 c0) (NodeData n1 a1 b1 c1)

data GlobalStiffnessMatrix a where
    GlobalStiffnessMatrix :: (TMap NodeDataC a b, TAnd b TTrue) => a -> GlobalStiffnessMatrix a

\end{code}
Then we can define a class for transforming data from the local frame to the global frame.
\begin{code}
class Transform local global | local -> global where
    transform :: local -> global

class TransformMatrix a where
    transformMatrix :: a -> [[Double]]

--instance (TransformMatrix (b,a)) => TransformMatrix (a,b) where
--    transformMatrix (a,b) = transformMatrix (b,a)

instance (TransformMatrix (n0,n1), a ~ [Double], 
          m0 ~ (NodeVector a v0), m1 ~ (NodeVector a v1), m2 ~ (NodeVector a v2),
          m3 ~ (NodeVector a v3), m4 ~ (NodeVector a v4), m5 ~ (NodeVector a v5)) => 
  Transform (LocalStiffnessMatrix   (NodeData n0 m0 m1 m2)    (NodeData n1 m3 m4 m5)) 
            (GlobalStiffnessMatrix ((NodeData n0 m0 m1 m2) :$ (NodeData n1 m3 m4 m5) :$ T)) where
            
    transform (LocalStiffnessMatrix (NodeData n0 (NodeVector a0 v0) (NodeVector a1 v1) (NodeVector a2 v2)) 
                                    (NodeData n1 (NodeVector a3 v3) (NodeVector a4 v4) (NodeVector a5 v5))) = 
       GlobalStiffnessMatrix
        ((NodeData n0 (NodeVector q0 v0) (NodeVector q1 v1) (NodeVector q2 v2)) :$
         (NodeData n1 (NodeVector q3 v3) (NodeVector q4 v4) (NodeVector q5 v5)) :$ End )
        
            where localkm = [a0, a1, a2, a3, a4, a5]
                  tm = transformMatrix (undefined :: (n0,n1))
                  tmt = transpose tm
                  global = (tmt `mmult` localkm) `mmult` tm
                  [n0v, n1v] = map fromIntegral [integerValue n0, integerValue n1]
                  [q0,q1,q2,q3,q4,q5] = aug n0v n1v global
                  aug n0v n1v global = zipWith (++) z0 z1 
                    where (-) = (P.-); (+) = (P.+); (*) = (P.*)
                          first = map (take 3) global
                          last = map (drop 3) global
                          z0 = map ((replicate (3*(n0v - 1)) 0) ++) first
                          z1 = map ((replicate (3*((n1v - n0v) - 1)) 0) ++) last

data SpaceFrame n a where
    SpaceFrame :: (Nat a, Nat b, Nat c) => c -> a -> b -> SpaceFrame c (a,b)
    
data TangentFrame a where
    TangentFrame :: (SpaceFrame n a) -> TangentFrame (SpaceFrame n a)

\end{code}
The instances are of the form:
instance Transform (LocalStiffnessMatrix (NodeData n0 a0 b0 c0) (NodeData n1 a1 b1 c1)) (GlobalStiffnessMatrix d) where
    transform (LocalStiffnessMatrix (NodeData n0 a0 b0 c0) (NodeData n1 a1 b1 c1)) = .....

A global stiffness matrix can also be directly declared:
GlobalStiffnessMatrix (
    (NodeData
        NodeVector [...] DisplacementX (D0 :$ T)
        NodeVector [...] DisplacementY (D0 :$ T)
        NodeVector [...] Moment (D0 :$ T)) 
    :$
    ...
    :$
    (NodeData
        NodeVector [...] DisplacementX (D5 :$ T)
        NodeVector [...] DisplacementY (D5 :$ T)
        NodeVector [...] Moment (D5 :$ T))
    :$ End)
\begin{code}

class ShowListH a where
    showListH :: a -> [String]
instance ShowListH T where
    showListH _ = []
instance (ShowListH b, Show a) => ShowListH (a :$ b) where
    showListH (a :$ b) = (show a):(showListH b) -- (show (undefined :: a)):(showListH (undefined :: b))

class ShowList a where
    showL :: a -> String
instance (ShowListH a) => ShowList a where
    showL a = (++ "]") ("[" ++ (intercalate ",\r\n     " $ showListH a))
    
instance (ShowList a) => Show (GlobalStiffnessMatrix a) where
    show (GlobalStiffnessMatrix a) = "GlobalStiffnessMatrix \r\n    " ++ showL a

instance (ShowList (a :$ b :$ End)) => Show (LocalStiffnessMatrix a b) where
    show (LocalStiffnessMatrix a b) = "LocalStiffnessMatrix \r\n    " ++ showL (a :$ b :$ End)

\end{code}
There can't be a Add instance for MemberMatrix, since the types would be different.
So there must be a different class. Since the computations need to be done on the data level, 
each class must contain a class function which actually adds the values.
For specific implementations, these classes don't need to be touched; only a new instace of Add must be declared.
\begin{code}

class CombineElem e xs r | e xs -> r where 
    combineElem :: e -> xs -> r
    
instance CombineElem e T (e :$ T) where
    combineElem e End = (e :$ End)
    
instance (CombineElem e xs r) => CombineElem e (x :$ xs) (x :$ r) where
    combineElem e (x :$ xs) = x :$ (combineElem e xs)
    
instance (Add x) => CombineElem x (x :$ xs) (x :$ xs) where
    combineElem e (x :$ xs) = (e + x) :$ xs --note that '+' is the one defined in the Add class

class Combine xs ys r | xs ys -> r where
    combine :: xs -> ys -> r

instance Combine End ys ys where
    combine End ys = ys
    
instance (CombineElem x ys t0, Combine xs t0 r) => Combine (x :$ xs) ys r where
    combine (x :$ xs) ys = combine xs (combineElem x ys)


\end{code}
There are instances for adding two global matrices, a local to a global, and two local (to produce a global).
\begin{code}
class Add3 a b c | a b -> c where
    add3 :: a -> b -> c
    
instance (Combine xs ys zs, TMap NodeDataC zs b, TAnd b TTrue
    ) => Add3 (GlobalStiffnessMatrix xs) (GlobalStiffnessMatrix ys) (GlobalStiffnessMatrix zs) where
        add3 (GlobalStiffnessMatrix xs) (GlobalStiffnessMatrix ys) = GlobalStiffnessMatrix (combine xs ys)

instance (Transform (LocalStiffnessMatrix n0 n1) (GlobalStiffnessMatrix xs),
          Add3 (GlobalStiffnessMatrix xs) (GlobalStiffnessMatrix ys) (GlobalStiffnessMatrix r)
    ) => Add3 (LocalStiffnessMatrix n0 n1) (GlobalStiffnessMatrix ys) (GlobalStiffnessMatrix r) where
        add3 local0 global0 = add3 (transform local0) global0  

instance (Transform (LocalStiffnessMatrix n0 n1) (GlobalStiffnessMatrix ys),
          Add3 (GlobalStiffnessMatrix xs) (GlobalStiffnessMatrix ys) (GlobalStiffnessMatrix r)
    ) => Add3 (GlobalStiffnessMatrix xs) (LocalStiffnessMatrix n0 n1) (GlobalStiffnessMatrix r) where
        add3 global0 local0 = add3 global0 (transform local0) 

instance (Transform (LocalStiffnessMatrix n0 n1) (GlobalStiffnessMatrix xs),
          Transform (LocalStiffnessMatrix m0 m1) (GlobalStiffnessMatrix ys),
          Add3 (GlobalStiffnessMatrix xs) (GlobalStiffnessMatrix ys) (GlobalStiffnessMatrix r)
    ) => Add3 (LocalStiffnessMatrix n0 n1) (LocalStiffnessMatrix m0 m1) (GlobalStiffnessMatrix r) where
        add3 local0 local1 = add3 (transform local0) (transform local1) 


--node0 = NodeData (D0 :$ End)
--            (NodeVector [1] DisplacementX)
--            (NodeVector [1] DisplacementY)
--            (NodeVector [1] Moment) 
--node1 = NodeData (D1 :$ End)
--            (NodeVector [1] DisplacementX)
--            (NodeVector [1] DisplacementY)
--            (NodeVector [1] Moment) 
--node2 = NodeData (D2 :$ End)
--            (NodeVector [1] DisplacementX)
--            (NodeVector [1] DisplacementY)
--            (NodeVector [1] Moment) 
--member0 = LocalStiffnessMatrix node0 node1
--member1 = LocalStiffnessMatrix node2 node1


data Row a = Local a a | Global a



\end{code}
Functions which generate data.
\begin{code}
mkLocalKM :: (Nat n0, Nat n1, d ~ Double) => n0 -> n1 -> d -> d -> d -> d -> d -> d -> d-> 
    LocalStiffnessMatrix (NodeData n0 (NodeVector [d] Moment) 
                                   (NodeVector [d] DisplacementX) 
                                   (NodeVector [d] DisplacementY)) 
                      (NodeData n1 (NodeVector [d] Moment) 
                                   (NodeVector [d] DisplacementX) 
                                   (NodeVector [d] DisplacementY)) 
mkLocalKM n0 n1 e a i x1 y1 x2 y2 = LocalStiffnessMatrix node0 node1
    where node0 = NodeData n0 (NodeVector q0 Moment) 
                              (NodeVector q1 DisplacementX) 
                              (NodeVector q2 DisplacementY) 
          node1 = NodeData n1 (NodeVector (map negate q0) Moment) 
                              (NodeVector (map negate q1) DisplacementX) 
                              (NodeVector q3 DisplacementY) 
          q0 = let t = (e*a)/l in [t, 0, 0, (-1)*t, 0, 0]
          q1 = let t0 = (12*e*i)/l^3; t1 = (6*e*i)/l^2 in [0, t0, t1, 0, (-1)*t0, t1]
          q2 = let t0 = (6*e*i)/l^2; t1 = (2*e*i)/l in [0, t0, 2*t1, 0, (-1)*t0, t1]
          q3 = let t0 = (6*e*i)/l^2; t1 = (2*e*i)/l in [0, t0, t1, 0, (-1)*t0, 2*t1]
          l = sqrt((x2 P.- x1) ** 2 P.+  (y2 P.- y1) ** 2)
          (-) = (P.-) 
          (+) = (P.+)
          (*) = (P.*)

mkTransformMatrix x1 y1 x2 y2 = 
   [[cx  , cy , 0 , 0  , 0  , 0 ],
    [-cy , cx , 0 , 0  , 0  , 0 ],
    [ 0  , 0  , 1 , 0  , 0  , 0 ],
    [ 0  , 0  , 0 , cx , cy , 0 ],
    [ 0  , 0  , 0 ,-cy , cx , 0 ],
    [ 0  , 0  , 0 , 0  , 0  , 1 ]]
        where (-) = (P.-) 
              (+) = (P.+)
              (*) = (P.*)
              l = sqrt((x2 P.- x1) ** 2 P.+  (y2 P.- y1) ** 2)
              cx = (x2 P.- x1) / l
              cy = (y2 P.- y1) / l

instance TransformMatrix (D1 :$ T, D2 :$ T) where
    transformMatrix _ = mkTransformMatrix 100 75 0 75

instance TransformMatrix (D1 :$ T, D3 :$ T) where
    transformMatrix _ = mkTransformMatrix 100 75 200 0

member1 = mkLocalKM [nat| 1 |] [nat| 2 |] 1e4 10 1e3 100 75 0 75
member2 = mkLocalKM [nat| 1 |] [nat| 3 |] 1e4 10 1e3 100 75 200 0


--class AddM a b c | a b -> c where
--    addM :: a -> b -> c
--
--instance AddM 

--data NatI where
--    NatI :: (Nat a) => a -> NatI
--
--mkClassFunc ''Compare
--
--instance Eq NatI where
--    (NatI a) == (NatI b) = tCompare a b
--
--instance Ord NatI where
--    (NatI a) `compare` (NatI b) = (integerValue a) `compare` (integerValue b)

--data Dat = Dat Char Int deriving Show
--d0 = [Dat 'a' 3, Dat 'b' 5, Dat 'c' 8]
--d1 = [Dat 'd' 3, Dat 'a' 5, Dat 'q' 8, Dat 'b' 2]
--
--sumDats [] ys = ys
--sumDats (x:xs) ys = sumDats xs (search x ys)
--    where search d [] = [d]
--          search (Dat c0 n0) ((Dat c1 n1):ys) 
--            | c0 == c1 = (Dat c0 (n0 P.+ n1)):ys
--            | otherwise = (Dat c1 n1):(search (Dat c0 n0) ys)
--
--class AddM a b c | a b -> c where
--    addM :: a -> b -> c



--instance AddM End End End where
--    addM _ _ = End
--instance (Add a0 b0 c0, AddM a1 b1 c1) => AddM (x :$ a) (x :$ b) (c0 :$ c1) where
--    addM (a0 :$ a1) (b0 :$ b1) = (a0 + b0) :$ (addM a1 b1)

--instance (AddM a b c, TLength c (D6 :$ T)) => AddM (MemberMatrix a) (MemberMatrix b) (MemberMatrix c) where
--    addM (MemberMatrix a) (MemberMatrix b) = (MemberMatrix (addM a b) :: MemberMatrix c)


--data NatB = forall a t0 t1 t2 t3 t4 t5 t6 t7 . (Nat a, IxRange a t0 t1, IxRange t2 a t3, IxIndex t4 a t5, IxInRange t6 a t7) => NatB a
--
--instance Show NatB where
--    show (NatB a) = "NatB " ++ show (integerValue a)
--    
--
--liftNat1 f (NatB q) = f (integerValue q) 
--liftNat2 f (NatB q) (NatB r) = (integerValue q) `f` (integerValue r)
--
--instance Eq NatB where
--    (==) = liftNat2 (==)
--    
--instance Ord NatB where 
--    compare = liftNat2 compare
--
--class IxRange a b c | a b -> c
--instance (
--          Nat min, Nat max,
--          Compare min max c,
--          TypeEq c LT ce,
--          Succ min nMin,
--          If ce {-then-} (IxRange nMin) max {-else-} Id T  {-out-} next
--    ) => IxRange min max (min :$ next)
--ixRange :: IxRange a b c => a -> b -> c
--ixRange = undefined
--
--data IndexOutOfRange
--class FailWith a b 
--
--class IxIndex a b c| a b -> c
--instance (
--          Compare i max cmax,
--          TypeEq cmax GT cmaxe,
--          Compare i min cmin, 
--          TypeEq cmin LT cmine,
--          If cmaxe {-then-} FailWith IndexOutOfRange {-else-} Id Int {-out-} __ , 
--          If cmine {-then-} FailWith IndexOutOfRange {-else-} (Subtract i) min {-out-} r
--    ) => IxIndex (min, max) i r
--ixIndex :: IxIndex a b c => a -> b -> c
--ixIndex = undefined
--
--class IxInRange a b c | a b -> c
--instance (Compare i min c0,
--          TypeEq c0 LT t0, Not t0 t0n,
--          Compare i max c1,
--          TypeEq c1 GT t1, Not t1 t1n,
--          (t1n :&& t0n) r
--    ) => IxInRange (min, max) i r
--ixInRange :: IxInRange a b c => a -> b -> c
--ixInRange = undefined
--
--instance Ix NatB where
--    range ((NatB q) , (NatB r)) = ixRange q r
--    index = undefined
--    inRange = undefined

\end{code}

