\ignore{
\begin{code}
{-# LANGUAGE 
    RankNTypes, 
    ScopedTypeVariables, 
    EmptyDataDecls, 
    MultiParamTypeClasses,
    FunctionalDependencies, 
    FlexibleInstances, 
    UndecidableInstances, 
    OverlappingInstances, 
    TypeFamilies , 
    ImpredicativeTypes, 
    FlexibleContexts, 
    DeriveDataTypeable,
    IncoherentInstances,
    ConstraintKinds,
    TypeOperators,
    GADTs,
    TemplateHaskell,
    QuasiQuotes
    #-}
module NumericTypes where
import Data.List
import Data.Typeable
import TypeLevelList
import Data.Ratio ((%))
import GHC.Prim (Constraint)
import TypeGen
\end{code}
}


\section{Natural numbers}\label{sec:Natural}

Variable-width type-level natural numbers, including constraints for non-zero numbers,
and constraints for addition and multiplication of numbers.


Basic data definitions: digits and comparison operators. These must come before TH splices.
\begin{code}
data D0 = D0
data D1 = D1
data D2 = D2
data D3 = D3
data D4 = D4
data D5 = D5
data D6 = D6
data D7 = D7
data D8 = D8
data D9 = D9

data LT {- a < b -}
data EQ {- a = b -}
data GT {- a > b -}

\end{code}
Template haskell splices for basic two digit instances. For their definitions, see the \hask{TypeGen} module
or compile with \hask{-ddump-splices}.
\begin{code}
compareDefinition
add2Definition
timesDefinition
sub2Definition


\end{code}
Digit class, which allows data level values to be calculated.
\begin{code}
class Digit a where
  digit :: a -> Integer
instance Digit D0 where digit _ = 0
instance Digit D1 where digit _ = 1
instance Digit D2 where digit _ = 2
instance Digit D3 where digit _ = 3
instance Digit D4 where digit _ = 4
instance Digit D5 where digit _ = 5
instance Digit D6 where digit _ = 6
instance Digit D7 where digit _ = 7
instance Digit D8 where digit _ = 8
instance Digit D9 where digit _ = 9

instance Show D0 where show _ = "0"
instance Show D1 where show _ = "1"
instance Show D2 where show _ = "2"
instance Show D3 where show _ = "3"
instance Show D4 where show _ = "4"
instance Show D5 where show _ = "5"
instance Show D6 where show _ = "6"
instance Show D7 where show _ = "7"
instance Show D8 where show _ = "8"
instance Show D9 where show _ = "9"

class IsDigit a b | a -> b
instance (TMap (TypeEq a) [nat| 9876543210 |] r, TOr r b) => IsDigit a b

\end{code}
Sign class, which contains a function to retrieve the value (1 or -1) and a type which determines the negation.
\begin{code}
class Sign s where 
    sgnVal :: s -> Integer
    type Negate s
data Pos = Pos 
data Neg = Neg 
instance Sign Pos where 
    sgnVal _ = 1
    type Negate Pos = Neg
instance Sign Neg where 
    sgnVal _ = (-1)
    type Negate Neg = Pos
    
type (:+) = Pos
type (:-)  = Neg

\end{code}
Class for type level comparators; also includes a function for getting the data level value.
\begin{code}
class Comparator c where
    toDataComp :: c -> Ordering
instance Comparator LT where toDataComp _ = LT
instance Comparator GT where toDataComp _ = GT
instance Comparator EQ where toDataComp _ = EQ


\end{code}
Natural number class. The only instance is a type level list of digits terminated by \hask{T}.
\begin{code}
class () => Nat s where
    integerValue :: s -> Integer
    intValue :: s -> Int
    intValue = fromIntegral . integerValue
instance Nat T where
    integerValue _ = undefined
instance (Digit a, Nat b, SizeValHelper ys, TReverse xs ys, xs ~ (a :$ b)) => Nat (a :$ b) where
    integerValue _ = sizeValHelper (undefined :: ys)

class IsNat a b | a -> b
instance IsNat T TFalse
instance (IsDigit a a0) => IsNat (a :$ T) a0
instance (IsDigit a a0, IsNat b b0, (a0 :&& b0) c) => IsNat (a :$ b) c

class ShowLH a where
    showListH :: a -> [String]
instance ShowLH T where
    showListH _ = []
instance (ShowLH b, Show a) => ShowLH (a :$ b) where
    showListH (a :$ b) = (show a):(showListH b)

class ShowL a where
    showL :: a -> String
instance (ShowLH a, IsNat a b, TBool b) => ShowL a where
    showL a 
        | toBool (undefined :: b) = concat $ showListH a
        | otherwise = (++ "]") ("[" ++ (intercalate ", " $ showListH a))

instance (ShowL a) => Show a where show = showL

\end{code}
The natural numbers are interpreted as having the most significant digit as the head of the list.
The opposite representation is much more convenient for calculating the value recursively.
\begin{code}
class SizeValHelper s where
    sizeValHelper :: s -> Integer
instance SizeValHelper T where sizeValHelper _ = 0
instance (SizeValHelper xs, Digit x) =>  SizeValHelper (x :$ xs) where 
    sizeValHelper _ = (digit (undefined :: x)) + 10 * (sizeValHelper (undefined :: xs))


\end{code}
Data wrapper (for consistency with float/int).
\begin{code}
data NAT a where
    NAT :: (Nat a) => a -> NAT a

\end{code}
Generic number class, as well as an instance for Nat. Used for varius data-level manipulations. 
\begin{code}
class GenericNumber a where 
    type Zero a
    type One a
    numToRational :: a -> Rational 
    numToFloating :: (Floating b) => a -> b
    numToFloating = fromRational . numToRational 
    numToIntegral :: (Integral b) => a -> b
    numToIntegral = round . numToFloating 

instance (Nat (a :$ b)) => GenericNumber (a :$ b) where
    type Zero (a :$ b) = D0 :$ T
    type One  (a :$ b) = D1 :$ T
    numToRational a = (integerValue a) % 1
    numToFloating = fromIntegral . integerValue
    numToIntegral = fromIntegral . integerValue

\end{code}
Determine if a number is zero?
\begin{code}
class IsZero a b | a -> b
instance IsZero D0 TTrue
instance IsZero D1 TFalse
instance IsZero D2 TFalse
instance IsZero D3 TFalse
instance IsZero D4 TFalse
instance IsZero D5 TFalse
instance IsZero D6 TFalse
instance IsZero D7 TFalse
instance IsZero D8 TFalse
instance IsZero D9 TFalse

\end{code}
\hask{IsZero} instance for natural; this simply tests if each digit is zero.
\begin{code}
instance (IsZero x xz,
          If xz {-then-} IsZero xs {-else-} Id TFalse {-out-} r 
    )=> IsZero (x :$ xs) r
instance IsZero T TTrue

\end{code}
\hask{NonZero a} is an alias for \hask{IsZero a TFalse}.
\begin{code}
class NonZero a 
instance (IsZero a TFalse) => NonZero a

\end{code}
Before writing instances for arithmetic, we need some helper classes.

EqualizeLists takes an element and two lists, and pads it the the front of the shorter list until they have the same length.
Both lists are returned (the longer one is unchanged).
\begin{code}
class EqualizeLists c a b | c a -> b
instance (MkPadding c (a,b) (p0,p1), TAppend p0 a r0, TAppend p1 b r1) => EqualizeLists c (a,b) (r0,r1)

class MkPadding c a b | c a -> b
instance MkPadding c (T,T) (T,T)
instance (MkPadding c (a0 :$ ax, c :$ T) (ar, br) ) => MkPadding c (a0 :$ ax, T) (ar, c :$ br)
instance (MkPadding c (c :$ T, a0 :$ ax) (ar, br) ) => MkPadding c (T, a0 :$ ax) (c :$ ar, br)
instance (MkPadding c (ax,bx) (ra,rb) ) => MkPadding c (a0 :$ ax, b0 :$ bx) (ra, rb)

\end{code}
Pads a single zero to the right hand side.
\begin{code}
class PadZeroR a b | a -> b
instance PadZeroR T T
instance PadZeroR (a :$ T) (a :$ D0 :$ T)
instance (PadZeroR b c) => PadZeroR (a :$ b) (a :$ c)

\end{code}
If the most significant digit is zero, rotate left one digit. (Remove the zero and pad a zero on the right)
Otherwise, pad one zero to the right.
\begin{code}
class LShiftAndExpand a b | a -> b
instance (PadZeroR a b) => LShiftAndExpand (D0 :$ a) b
instance (PadZeroR (D1 :$ a) b) => LShiftAndExpand (q :$ a) b

\end{code}
Two classes for accessing individual digits of addition, as well as both digits as a pair.
\begin{code}
class Add2Low a b c | a b -> c
instance (Add2 a b c d) => Add2Low a b d
class Add2High a b c | a b -> c
instance (Add2 a b c d) => Add2High a b c
class Add2Pair a b c | a b -> c
instance (Add2 a b c d) => Add2Pair a b (c,d)

\end{code}
Removes leading zeros.
\begin{code}
class TruncateLeadingZeroes a b | a -> b
instance TruncateLeadingZeroes T T
instance (TruncateLeadingZeroes x y) => TruncateLeadingZeroes (D0 :$ x) y
instance TruncateLeadingZeroes (D1 :$ x) (D1 :$ x)
instance TruncateLeadingZeroes (D2 :$ x) (D2 :$ x)
instance TruncateLeadingZeroes (D3 :$ x) (D3 :$ x)
instance TruncateLeadingZeroes (D4 :$ x) (D4 :$ x)
instance TruncateLeadingZeroes (D5 :$ x) (D5 :$ x)
instance TruncateLeadingZeroes (D6 :$ x) (D6 :$ x)
instance TruncateLeadingZeroes (D7 :$ x) (D7 :$ x)
instance TruncateLeadingZeroes (D8 :$ x) (D8 :$ x)
instance TruncateLeadingZeroes (D9 :$ x) (D9 :$ x)

class Sum a b c | a b -> c

\end{code}
Addition is performed as one would expect.
Note that the numbers are reversed first so that they can be traversed from low digit to high digit.
\begin{code}
instance (--TruncateLeadingZeroes a at, TruncateLeadingZeroes b bt, -- this will break floating point multiplication
          EqualizeLists D0 (a,b) (ae,be), 
          TReverse ae aer, TReverse be ber,
          SumH aer ber D0 r,
          TReverse r rr
    ) => Sum a b rr


class SumH a b c r | a b c -> r
instance (TypeEq c D0 b,
          If b {-then-} Id T {-else-} Id (c :$ T) {-out-} r
    ) => SumH T T c r

instance (Add2Pair x y (t0,q0), Add2Pair q0 c (t1,q1), Add2Low t0 t1 t2,
          SumH xs ys t2 rs
    ) => SumH (x :$ xs) (y :$ ys) c (q1 :$ rs)

\end{code}
Now we can define the sum of a list.
\begin{code}
class SumList xs r | xs -> r
instance SumList (x :$ T) x
instance (SumList xs rs, Sum x rs r) => SumList (x :$ xs) r

\end{code}
Multiplication.
\begin{code}
class MultD a b c | a b -> c

\end{code}
Two classes for getting high and low digits individually. 
\begin{code}
class Times2Low a b c | a b -> c
instance (Times a b c d) => Times2Low a b d
class Times2High a b c | a b -> c
instance (Times a b c d) => Times2High a b c
class Times2Pair a b c | a b -> c
instance (Times a b c d) => Times2Pair a b (c,d)

\end{code}
Pad a zero to every number in the tail, and recurse on the tail.
\begin{code}
class PadMany a b | a -> b
instance PadMany T T
instance (TMap PadZeroR (x :$ xs) (y :$ ys), PadMany ys rs) => PadMany (x :$ xs) (y :$ rs)

\end{code}
Multiplication digit-wise. Two simple cases that will speed up multiplication.
\begin{code}
instance MultD D0 s (D0 :$ T)
instance MultD D1 s s

\end{code}
Again there are two possibility for the implentation. The second one seems to work better.
\begin{code}
instance (TReverse num numr,
          MultDH numr dig D0 r,
          TReverse r rr
    ) => MultD num dig rr
    
class MultDH a b c d | a b c -> d
instance (TypeEq c D0 b,
          If b {-then-} Id T {-else-} Id (c :$ T) {-out-} r
    ) => MultDH T d c r
instance (Times2Pair x y (t0,q0), Add2Pair q0 c (t1,q1), Add2Low t0 t1 t2,
          MultDH xs y t2 rs
    ) => MultDH (x :$ xs) y c (q1 :$ rs)


\end{code}
Get a list of digit-wise multiplications. Shift them to get the correct place values, and find the sum. 
\begin{code}
instance (Nat az, Nat bz, az ~ (q :$ qq), bz ~ (r :$ rr), az ~ a, bz ~ b,
--          TruncateLeadingZeroes az a, TruncateLeadingZeroes bz b, -- this will break floating multiplication
          TMap (MultD a) b t0,
          TReverse t0 (t1 :$ t1s),
          PadMany t1s t1s_p , 
          TFold Sum t1 t1s_p c
    ) =>  MultD (q :$ qq) (r :$ rr) c


\end{code}
Basic comparison operators.
\begin{code}
instance Show LT where 
    show _ = "LT"
instance Show EQ where 
    show _ = "EQ"
instance Show GT where 
    show _ = "GT"

\end{code}
Compare the numbers digit-wise, the zips the comparisons with the function below. 
\begin{code}
instance (EqualizeLists D0 (a,b) (ae,be), TZipWith Compare ae be z, ZipComp z c) => Compare a b c

\end{code}
If the head of the list is EQ then the next digit(s) must be examined. Otherwise, we know the result.
If the end of the list is reached we know the numbers are equal.
\begin{code}
class ZipComp a b | a -> b
instance                   ZipComp T          EQ
instance                   ZipComp (LT :$ xs) LT
instance                   ZipComp (GT :$ xs) GT
instance (ZipComp xs r) => ZipComp (EQ :$ xs) r


\end{code}
Subtraction. \hask{InvAdd} performs subtraction 'unsafely'; it assumes that \hask{a - b >= 0}. 
\begin{code}
class CompToDig a b | a -> b
instance CompToDig GT D0
instance CompToDig LT D1
instance CompToDig EQ D0

class InvAdd a b c | a b -> c
instance (TruncateLeadingZeroes a at, TruncateLeadingZeroes b bt,
--          a ~ at, b ~ bt,
          EqualizeLists D0 (at,bt) (fl,el) ,
          
          TZipWith Sub2 fl el rl,   -- result
          TZipWith Compare fl el cl,
          TMap CompToDig cl nl ,     -- carries
          IsZero nl nlz,
          PadZeroR nl nl_p, yuriy
          If nlz {-then-} Id rl {-else-} (InvAdd rl) nl_p {-out-} result
          
    ) => InvAdd a b result   

\end{code}
InvAddSafe takes a comparison and two numbers, and should call InvAdd properly as to prevent type check failure.
\begin{code}
data NaturalSubtractionProducesNegativeNumber
class InvAddSafe a b c d | a b c -> d
instance (InvAdd (a :$ ax) (b :$ bx) c) => InvAddSafe GT (a :$ ax) (b :$ bx) c
instance (Fail NaturalSubtractionProducesNegativeNumber) => InvAddSafe LT (a :$ ax) (b :$ bx) r
instance InvAddSafe EQ (a :$ ax) (b :$ bx) (D0 :$ T)

\end{code}
The actual subtract class simply compares the two numbers and calls InvAddSafe with the comparison. 
\begin{code}
class Subtract a b c | a b -> c
instance ((fn `Compare` en) r,
           InvAddSafe r fn en result
    ) => Subtract fn en result 


\end{code}
