-- | The Main module of the SAGA program. The entry point function 'main' is defined here
module Main (main) where

-------------
-- Imports --
-------------

import SAGA.Code (Code(..))
import SAGA.CodeGeneration (createCodeFiles,makeCode)
import SAGA.CodeGeneration.AbstractCode (makeAbstractCode)
import SAGA.Parsers.ConfigParser (Configuration(..),readConfig)
import SAGA.Parsers.StoryParser (readStory)
import SAGA.StoryManager.DataTypes (Story(storyLabel))
import SAGA.StoryManager.DotOutput (dotFileFromStory)
import SAGA.StoryManager.Helper (makeVarNameValid)

import List (intersperse)
import System (getArgs)
import System.Process (runCommand)
import Text.ParserCombinators.Parsec (ParseError)

------------------
-- IO Functions --
------------------

{-|
The entry point for the SAGA program. The following two
command-line arguments are required: a configuration file (to specify the
desired generation language), and a story description file.

Ex. SAGA.exe config.txt story.txt
-}
main :: IO ()
main = do
    args <- getArgs
    if length args /= 2
        then error "SAGA.main: Program only accepts two arguments: configuration\
            \file and story file to parse (ie. SAGA.exe config.txt story.saga)"
        else let configName = args !! 0
                 storyName = args !! 1
             in do
                configInput <- readFile configName
                storyInput <- readFile storyName
                putStrLn ""
                putStrLn "Parsing story description..."
                let story = readStory storyName storyInput
                processStory story configName configInput storyName

processStory :: Either ParseError Story -> String -> String -> String -> IO ()
processStory (Left err) _ _ _ = putStrLn $ "No Match: " ++ show err
processStory (Right story) cName cInput sName = do
    putStrLn $ "Done parsing " ++ sName
    dotFileFromStory story
    let fileName = makeVarNameValid (storyLabel story)
    putStrLn $ fileName ++ ".dot successfully created"
    runCommand $ "dot -K dot -T png -o "
        ++ fileName ++ ".png " ++ fileName ++ ".dot"
    putStrLn $ fileName ++ ".png successfully created"
    putStrLn ""
    putStrLn "Parsing configuration..."
    let configResult = readConfig cName cInput
    case configResult of
        Left err -> putStrLn $ "No Match: " ++ show err
        Right (Config lang) -> do
            putStrLn $ "Done parsing " ++ cName
            putStrLn ""
            putStrLn "Generating code..."
            let absCode = makeAbstractCode story
            let code@(Code codePairs) = makeCode lang absCode
            createCodeFiles code
            let fileCreationText = concat (intersperse " successfully created\n"
                    (map fst codePairs)) ++ " successfully created"
            putStrLn fileCreationText
