-- | Contains the high-level functionality to create 'Code' and then produce the actual generated code files
module SAGA.CodeGeneration (
    -- * Preparing the code files
    makeCode,
    
    -- * Creating the code files
    createCodeFiles
    ) where

import SAGA.Code (Code(..))
import SAGA.CodeGeneration.AbstractCode (AbstractCode)
import SAGA.CodeGeneration.LanguageRenderer (LanguageRenderer(renderCode))
import SAGA.CodeGeneration.LanguageRenderer.CSharpRenderer (CSharpRenderer(..))
import SAGA.CodeGeneration.LanguageRenderer.CppRenderer (CppRenderer(..))
import SAGA.CodeGeneration.LanguageRenderer.JavaRenderer (JavaRenderer(..))
import SAGA.Parsers.ConfigParser (cSharpLabel, cppLabel, javaLabel)

import Text.PrettyPrint.HughesPJ (Doc, render)
import IO

-- | Takes a language parameter and passes an 'AbstractCode' to the required rendering function, which produces a 'Code'
makeCode :: String -> (AbstractCode -> Code)
makeCode l c
    | l == cSharpLabel  = renderCode CSharpRenderer c
    | l == cppLabel     = renderCode CppRenderer c
    | l == javaLabel    = renderCode JavaRenderer c
    | otherwise         = error $ "SAGA.CodeGeneration.makeCode: must supply " ++ quoteString cSharpLabel ++ ", " ++ quoteString cppLabel ++ ", or " ++ quoteString javaLabel ++ " for the \"Generation Language\" option in the configuration file"

quoteString :: String -> String
quoteString s = "\"" ++ s ++ "\""

------------------
-- IO Functions --
------------------

-- | Creates the requested 'Code' by producing files
createCodeFiles :: Code -> IO ()
createCodeFiles (Code cs) = mapM_ createCodeFile cs

createCodeFile :: (FilePath, Doc) -> IO ()
createCodeFile (path, code) = do
    h <- openFile path WriteMode
    hPutStrLn h (render code)
    hClose h
