-- | An IR (intermediate representation) datatype of the generated code is
--   defined here, as well as a function to generate this IR datatype
module SAGA.CodeGeneration.AbstractCode (
    -- * AbstractCode
    -- ** Statement Structure
    Block(..),
    Statement(..), Assignment(..), Declaration(..), Conditional(..),
    Iteration(..), Jump(..), Return(..), Value(..), Comment(..),
    Literal(..), Function(..),
    Expression(..), UnaryRelation(..), BinaryRelation(..),
    -- ** Overall AbstractCode Structure
    Body, BaseType(..), StateType(..), TransType(..),
    Scope(..), Parameter(..), State(..), Transformation(..), Module(..), Package(..),
    AbstractCode(..),

    -- * Some Functions to Do a ForEach -> For Conversion in 'CppRenderer'
    varDecDef, litInt, ($.), plusPlus, less, at,
    eventLabels,

    -- * Generated Class Names
    nodeName,
    sectName,
    nodeTransName,
    sectTransName,
    storyName,
    storyManagerName,

    -- * Constructing 'AbstractCode'
    makeAbstractCode
) where

import SAGA.StoryManager.DataTypes
import SAGA.StoryManager.Helper (makeVarNameValid)

import List (zipWith4)
import Prelude hiding (break)

data Block = Block [Statement]
data Statement = AssignState Assignment | DeclState Declaration
               | CondState Conditional | IterState Iteration | JumpState Jump
               | RetState Return
               | ValState Value
               | CommentState Comment
data Assignment = Assign Value Value
                | PlusEquals Label Value
                | PlusPlus Label
data Declaration = VarDec Label StateType
                 | ListDec Label StateType Int
                 | ListDecLiterals Label Label StateType [Literal]
                 | VarDecDef Label StateType Value
                 | ObjDecDef Label StateType Value
data Conditional = If [(Value, Body)] Body
                 | Switch Value [(Literal, Body)]
data Iteration = For Statement Value Statement Body
               | ForEach StateType Label Value Body
data Jump = Break | Continue
data Return = Ret Value
data Value = Lit Literal
           | Var Label
           | FuncApp Label [Value]
           | ObjAccess Value Function
           | StateObj StateType [Value]
           | Expr Expression
    deriving Eq
-- data LValue =
-- Values should be split up into some LValue/RValue structures
data Literal = LitBool Bool
             | LitInt Int
             | LitFloat Float
             | LitChar Char
             | LitStr String
    deriving Eq
data Function = Func {funcName :: Label, funcParams :: [Value]}
              | IndexOf Value
              | ListSize
              | ListAccess Value
              | ListAdd Value Int
              | IterBegin
              | IterEnd
    deriving Eq
data Comment = Comment Label | CommentDelimit Label Int
data Expression = UnaryExpr UnaryRelation Value
                | BinaryExpr Value BinaryRelation Value
    deriving Eq
data UnaryRelation = Negation deriving Eq
data BinaryRelation = Equal   | NotEqual
                    | Greater | GreaterEqual
                    | Less    | LessEqual
    deriving Eq

type Body = [Block]
data BaseType = Boolean | Integer | Float | Character | String
    deriving Eq
data StateType = NodeT | NodeTransT | SectT | SectTransT | StoryT | StoryManT
               | List StateType | Base BaseType
    deriving Eq
data TransType = TState StateType
               | Void
               | Construct Label
data Scope = Private | Public
    deriving Eq
data Parameter = StateParam Label StateType
               | FuncParam Label TransType [Parameter]
data State = State Label Scope StateType
data Transformation = Transform Label Scope TransType [Parameter] Body
data Module = Mod
    {moduleName :: Label,
     moduleScope :: Scope,
     moduleVars :: [State],
     moduleFuncs :: [Transformation]}
data Package = Pack Label [Module]
data AbstractCode = AbsCode Package

makeAbstractCode :: Story -> AbstractCode
makeAbstractCode s =
    let ms = [nodeModule, nodeTransModule, sectionModule, sectionTransModule, storyModule, storyManagerModule s]
    in AbsCode $ Pack "StoryDSL" ms

-- Class names required globally
nodeName, sectName, nodeTransName, sectTransName, storyName, storyManagerName :: String
nodeName = "Node"
sectName = "Section"
nodeTransName = "NodeTransition"
sectTransName = "SectionTransition"
storyName = "Story"
storyManagerName = "StoryManager"

-- Function names required by multiple Module definitions
getNodeLabel, getNodeOutTrans, setNodeOutTrans, setNodeParent, getNodeTransEvents, getDstNode, getNodeTrans, getStoryState, setStoryState, getNodes :: String
getNodeLabel = "GetNodeLabel"
getNodeOutTrans = "GetNodeOutTrans"
setNodeOutTrans = "SetNodeOutTrans"
setNodeParent = "SetNodeParent"
getNodeTransEvents = "GetNodeTransEvents"
getDstNode = "GetDstNode"
getNodeTrans = "GetNodeTrans"
getStoryState = "GetStoryState"
setStoryState = "SetStoryState"
getNodes = "GetNodes"

-- Variable names required globally
managerStoryVar, eventLabels, label, sections, headVar, sectionTransVar :: String
managerStoryVar = "story"
eventLabels = "eventLabels"
label = "label"
sections = "sections"
headVar = "headVar"
sectionTransVar = "sectionTransVar"

privVarSep, evtsPrefix, nodePrefix, nodesPrefix, transPrefix, sectPrefix, outTransPrefix, toLabel :: String
privVarSep = "__"
evtsPrefix = "events" ++ privVarSep
nodePrefix = "node" ++ privVarSep
nodesPrefix = "nodes" ++ privVarSep
transPrefix = "trans" ++ privVarSep
sectPrefix = "sect" ++ privVarSep
outTransPrefix = "outTrans" ++ privVarSep
toLabel = privVarSep ++ "To" ++ privVarSep

-- Helper/Convenience functions
pubMod :: Label -> [State] -> [Transformation] -> Module
pubMod n vs fs = Mod n Public vs fs

privVar :: StateType -> Label -> State
privVar t n = State n Private t

privFunc :: TransType -> Label -> [Parameter] -> Body -> Transformation
privFunc t n ps b = Transform n Private t ps b

pubFunc :: TransType -> Label -> [Parameter] -> Body -> Transformation
pubFunc t n ps b = Transform n Public t ps b

bool, int, float, char, string, node, nodeTransition, section, sectionTransition, story, storyManager :: StateType
bool = Base Boolean
int = Base Integer
float = Base Float
char = Base Character
string = Base String
node = NodeT
nodeTransition = NodeTransT
section = SectT
sectionTransition = SectTransT
story = StoryT
storyManager = StoryManT

noElse :: Body
noElse = []

typ :: StateType -> TransType
typ = TState

false :: Value
false = Lit $ LitBool False

true :: Value
true = Lit $ LitBool True

litInt :: Int -> Value
litInt = Lit . LitInt

litString :: Label -> Value
litString = Lit . LitStr

litObj :: StateType -> [Value] -> Value
litObj t vs = StateObj t vs

negation :: Value -> Value
negation v = Expr $ UnaryExpr Negation $ v

less :: Value -> Value -> Value
v1 `less` v2 = Expr $ BinaryExpr v1 Less v2

binExpr :: Value -> BinaryRelation -> Value -> Value
binExpr v1 r v2 = Expr $ BinaryExpr v1 r v2

($.) :: Value -> Function -> Value
infixl 5 $.
v $. f = ObjAccess v f

varDecDef :: Label -> StateType -> Value -> Statement
varDecDef n t v = DeclState $ VarDecDef n t v

listDec :: Label -> StateType -> Int -> Statement
listDec n t s = DeclState $ ListDec n t s

listDecLiterals :: Label -> Label -> StateType -> [Literal] -> Statement
listDecLiterals n temp t ls = DeclState $ ListDecLiterals n temp t ls

objDecDef :: Label -> StateType -> Value -> Statement
objDecDef n t v = DeclState $ ObjDecDef n t v

returnVar :: Label -> Statement
returnVar = RetState . Ret . Var

infixr 5 $=
($=) :: Label -> Value -> Statement
($=) a b = AssignState $ Assign (Var a) b

(+=) :: Label -> Value -> Statement
infixl 6 +=
n += v = AssignState $ PlusEquals n v

plusPlus :: Label -> Statement
plusPlus n = AssignState $ PlusPlus n

ifCond :: [(Value, Body)] -> Body -> Statement
ifCond ifResults elseResult = CondState $ If ifResults elseResult

for :: Statement -> Value -> Statement -> Body -> Statement
for initv guard update b = IterState $ For initv guard update b

forEach :: StateType -> Label -> Value -> Body -> Statement
forEach t x xs b = IterState $ ForEach t x xs b

break :: Statement
break = JumpState Break

at :: Label -> Function
at = ListAccess . Var

addElem :: Label -> Value -> Int -> Statement
addElem n v i = ValState $ Var n $. ListAdd v i

param :: Label -> StateType -> Parameter
param = StateParam

params :: [(Label, StateType)] -> [Parameter]
params = map (\p -> StateParam (fst p) (snd p))
-- ex. (params [ (List NodeTransT,transParam) ])

oneLiner :: Statement -> Body
oneLiner s = [Block [s]]

-----------------------
-- Comment Functions --
-----------------------

commentLength = 75
endCommentLabel = "End"

comment :: Label -> Statement
comment = CommentState . Comment

commentDelimit :: Label -> Statement
commentDelimit s = CommentState $ CommentDelimit s commentLength

endCommentDelimit :: Label -> Statement
endCommentDelimit s = CommentState $ CommentDelimit (endCommentLabel ++ " " ++ s) commentLength

sectionLabelComment :: Section -> Statement
sectionLabelComment s = comment $ "\"" ++ sectionLabel s ++ "\""

prefixFirstBlock :: Statement -> [Block] -> [Block]
prefixFirstBlock s ((Block ss):bs) = Block(s : ss) : bs
prefixFirstBlock _ [] = error "prefixFirstBlock called without a block"

addComments :: Label -> [Block] -> [Block]
addComments c ((Block ss):[]) =
    let cStart = commentDelimit c
        cEnd = endCommentDelimit c
    in [Block $ cStart : ss ++ [cEnd]]
addComments c ((Block ss1):bs) =
    let cStart = commentDelimit c
        cEnd = endCommentDelimit c
        (Block ssn) = last bs
    in Block(cStart : ss1) : (init bs) ++ [Block $ ssn ++ [cEnd]]
addComments _ [] = error "addComments on an empty Block"

-- IR (intermediate representation) of the various modules
nodeModule :: Module
nodeModule =
    let modName = "Node"
        nodeLabel = "nodeLabel"
        label = "label"
        nodeOutTrans = "nodeOutTrans"
        trans = "trans"
        nodeParent = "nodeParent"
        parent = "parent"
    in pubMod modName [
        privVar string              nodeLabel,
        privVar (List NodeTransT)   nodeOutTrans,
        privVar section             nodeParent
    ] [
        pubFunc (Construct modName) modName [param label string] [
            Block [
                nodeLabel $= Var label,
                nodeOutTrans $= litObj (List nodeTransition) [litInt 0]]],
        pubFunc (typ string) getNodeLabel [] $ oneLiner $
            returnVar nodeLabel,
        pubFunc (typ $ List NodeTransT) getNodeOutTrans [] $ oneLiner $
            returnVar nodeOutTrans,
        pubFunc Void setNodeOutTrans [param trans (List NodeTransT)] $ oneLiner $
            nodeOutTrans $= Var trans,
        pubFunc (typ section) "GetNodeParent" [] $ oneLiner $
            returnVar nodeParent,
        pubFunc Void setNodeParent [param parent section] $ oneLiner $
            nodeParent $= Var parent
    ]

nodeTransModule :: Module
nodeTransModule =
    let modName = "NodeTransition"
        srcNode = "srcNode"
        src = "src"
        dstNode = "dstNode"
        dst = "dst"
        nodeTransEvents = "nodeTransEvents"
        evts = "evts"
    in pubMod modName [
        privVar node            srcNode,
        privVar node            dstNode,
        privVar (List string)   nodeTransEvents
    ] [
        pubFunc (Construct modName) modName [param src node, param dst node, param evts (List string)] [
            Block [
                srcNode $= Var src,
                dstNode $= Var dst,
                nodeTransEvents $= Var evts
            ]],
        pubFunc (typ node) "GetSrcNode" [] $ oneLiner $
            returnVar srcNode,
        pubFunc (typ node) getDstNode [] $ oneLiner $
            returnVar dstNode,
        pubFunc (typ $ List string) getNodeTransEvents [] $ oneLiner $
            returnVar nodeTransEvents
    ]

sectionModule :: Module
sectionModule =
    let modName = "Section"
        sectionLabel = "sectionLabel"
        label = "label"
        nodes = "nodes"
        ns = "ns"
        nodeTrans = "nodeTrans"
        trans = "trans"
    in pubMod modName [
        privVar string                  sectionLabel,
        privVar (List node)             nodes,
        privVar (List nodeTransition)   nodeTrans
    ] [
        pubFunc (Construct modName) modName [param label string, param ns (List node), param trans (List nodeTransition)] [
            Block [
                sectionLabel $= Var label,
                nodes $= Var ns,
                nodeTrans $= Var trans
            ]],
        pubFunc (typ string) "GetSectionLabel" [] $ oneLiner $
            returnVar sectionLabel,
        pubFunc (typ $ List node) getNodes [] $ oneLiner $
            returnVar nodes,
        pubFunc (typ $ List nodeTransition) getNodeTrans [] $ oneLiner $
            returnVar nodeTrans
    ]

sectionTransModule :: Module
sectionTransModule =
    let modName = sectTransName
        srcSection = "srcSection"
        src = "src"
        dstSection = "dstSection"
        dst = "dst"
        sectionTransEvents = "sectionTransEvents"
        evts = "evts"
    in pubMod modName [
        privVar section         srcSection,
        privVar section         dstSection,
        privVar (List string)   sectionTransEvents
    ] [
        pubFunc (Construct modName) modName [param src section, param dst section, param evts (List string)] [
            Block [
                srcSection $= Var src,
                dstSection $= Var dst,
                sectionTransEvents $= Var evts
            ]],
        pubFunc (typ section) "GetSrcSection" [] $ oneLiner $
            returnVar srcSection,
        pubFunc (typ section) "GetDstSection" [] $ oneLiner $
            returnVar dstSection,
        pubFunc (typ $ List string) "GetSectionTransEvents" [] $ oneLiner $
            returnVar sectionTransEvents
    ]

storyModule :: Module
storyModule =
    let modName = "Story"
        storyLabel = "storyLabel"
        label = "label"
        storySections = "storySections"
        sects = "sects"
        storyHead = "storyHead"
        h = "h"
        storySectionTrans = "sectionTrans"
        sectTrans = "sectTrans"
        currentState = "currentState"
        newState = "newState"
        numEvents = "numEvents"

        constructor =
            let eventCount = "eventCount"
                i = "i"
                j = "j"
            in pubFunc (Construct modName) modName [param label string, param sects (List section), param h node, param sectTrans (List sectionTransition)] [
                Block [
                    storyLabel $= Var label,
                    storySections $= Var sects,
                    storyHead $= Var h,
                    storySectionTrans $= Var sectTrans,
                    currentState $= Var h
                ],
                Block [
                    varDecDef eventCount int (litInt 0),
                    -- forEach section i (Var storySections) [
                        -- Block [
                            -- forEach nodeTransition j (Var i $. Func getNodeTrans []) $ oneLiner $
                                -- eventCount += (Var j $. Func getNodeTransEvents [] $. ListSize)
                        -- ]
                    -- ]
                    for (varDecDef i int (litInt 0)) (Var i `less` (Var storySections $. ListSize)) (plusPlus i) [
                        Block [
                            for (varDecDef j int (litInt 0)) (Var j `less` (Var storySections $. at i $. Func getNodeTrans [] $. ListSize)) (plusPlus j) $ oneLiner $
                                eventCount += (Var storySections $. at i $. Func getNodeTrans [] $. at j $. Func getNodeTransEvents [] $. ListSize)
                        ]
                    ]
                ],
                Block [numEvents $= Var eventCount]]
    in pubMod modName [
        privVar string                      storyLabel,
        privVar (List section)              storySections,
        privVar node                        storyHead,
        privVar (List sectionTransition)    storySectionTrans,
        privVar node                        currentState,
        privVar int                         numEvents
    ] [
        constructor,
        pubFunc (typ string) "GetStoryLabel" [] $ oneLiner $
            returnVar storyLabel,
        pubFunc (typ (List section)) "GetStorySections" [] $ oneLiner $
            returnVar storySections,
        pubFunc (typ node) "GetStoryHead" [] $ oneLiner $
            returnVar storyHead,
        pubFunc (typ (List sectionTransition)) "GetSectionTrans" [] $ oneLiner $
            returnVar storySectionTrans,
        pubFunc (typ node) getStoryState [] $ oneLiner $
            returnVar currentState,
        pubFunc Void setStoryState [param newState node] $ oneLiner $
            currentState $= Var newState,
        pubFunc (typ int) "GetNumEvents" [] $ oneLiner $
            returnVar numEvents
    ]

storyManagerModule :: Story -> Module
storyManagerModule s =
    let modName = "StoryManager"
        evtData = "evtData"
        findIndex = "FindIndex"
        processSnapshot = "ProcessSnapshot"
        checkEvents = "CheckEvents"

        constructor = pubFunc (Construct modName) modName [] $ concat [
            labelStatements $ storyLabel s,
            eventsStatements (storyNodes s) (storyTrans s),
            nodesStatements (storyNodes s),
            nodeTransStatements (storyNodes s) (storyTrans s),
            sectionStatements (storyNodes s),
            outGoingStatements (storyNodes s) (storyTrans s),
            headStatements (storyHead s),
            sectionTransStatements (storyTrans s),
            storyStatements,
            eventLabelsStatements (storyEvents s)]

        receiveStorySnapshotTransform =
            let newState = "newState"
            in pubFunc (typ node) "ReceiveStorySnapshot" [param evtData (List bool)] [
                Block [
                    varDecDef newState node (FuncApp processSnapshot [Var evtData]),
                    returnVar newState
                ]
            ]

        findIndexTransform =
            let pos = "pos"
                evtLab = "evtLab"
            in pubFunc (typ int) findIndex [param evtLab string] [
                Block [
                    varDecDef pos int (Var eventLabels $. (IndexOf $ Var evtLab)),
                    returnVar pos
                ]
            ]

        processSnapshotTransform =
            let i = "i"
                oldState = "oldState"
                result = "result"
                trans = "trans"
                storyGetState = Var managerStoryVar $. Func getStoryState []
            in privFunc (typ node) processSnapshot [param evtData (List bool)] [
                Block [
                    varDecDef oldState node storyGetState,
                    varDecDef result node (Var oldState),
                    varDecDef trans (List nodeTransition) (Var oldState $. Func getNodeOutTrans [])
                ],
                Block [
                    -- forEach nodeTransition i (Var trans) [
                        -- Block [
                            -- ifCond [(
                                -- FuncApp checkEvents [Var i $. Func getNodeTransEvents [], Var evtData], [
                                    -- Block [ValState $ Var managerStoryVar $. Func setStoryState [Var i $. Func getDstNode []],
                                    -- break]]
                            -- )] noElse
                        -- ]
                    -- ]
                    for (varDecDef i int (litInt 0)) (Var i `less` (Var trans $. ListSize)) (plusPlus i) [
                        Block [
                            ifCond [(
                                FuncApp checkEvents [Var trans $. at i $. Func getNodeTransEvents [], Var evtData], [
                                    Block [ValState $ Var managerStoryVar $. Func setStoryState [Var trans $. at i $. Func getDstNode []],
                                    break]]
                            )] noElse
                        ]
                    ]
                ],
                Block [
                    ifCond [(
                        binExpr (Var managerStoryVar $. Func getStoryState [] $. Func getNodeLabel []) NotEqual (Var oldState $. Func getNodeLabel []), [
                            Block [result $= storyGetState]]
                        )] noElse
                ],
                Block [returnVar result]
            ]

        checkEventsTransform =
            let result = "result"
                i = "i"
                evtLabels = "evtLabels"
            in privFunc (typ bool) checkEvents [param evtLabels (List string), param evtData (List bool)] [
                Block [varDecDef result bool true],
                Block [
                    -- forEach string i (Var evtLabels) [
                        -- Block [
                            -- ifCond [(
                                -- negation $ Var evtData $. ListAccess (FuncApp findIndex [Var i]), [
                                    -- Block [
                                        -- result $= false,
                                        -- break
                                    -- ]
                                -- ]
                            -- )] noElse
                        -- ]
                    -- ]
                    for (varDecDef i int (litInt 0)) (Var i `less` (Var evtLabels $. ListSize)) (plusPlus i) [
                        Block [
                            ifCond [(
                                negation $ Var evtData $. ListAccess (FuncApp findIndex [Var evtLabels $. at i]), [
                                    Block [
                                        result $= false,
                                        break
                                    ]
                                ]
                            )] noElse
                        ]
                    ]
                ],
                Block [returnVar result]]

    in pubMod modName [
        privVar story           managerStoryVar,
        privVar (List string)   eventLabels
    ] [
        constructor,
        receiveStorySnapshotTransform,
        pubFunc (typ story) "GetStory" [] $ oneLiner $
            returnVar managerStoryVar,
        pubFunc (typ (List string)) "GetEventLabels" [] $ oneLiner $
            returnVar eventLabels,
        findIndexTransform,
        processSnapshotTransform,
        checkEventsTransform
    ]

labelStatements :: Label -> Body
labelStatements l =
    let commentLabel = "Label of the " ++ storyName
        label = "label"
    in addComments commentLabel [Block [varDecDef label string (litString l)]]

eventsStatements :: [Section] -> [SectionTransition] -> Body
eventsStatements sects ts =
    let commentLabel = "Events"
        sectTransEvts = concatMap (\t -> eventsStatements' (sectionTransPreNode t) (sectionTransPostNode t) (sectionTransEvents t)) ts
    in addComments commentLabel
        (eventsSectionsStatements sects ++ [Block $ (comment $ "Events for " ++ sectTransName ++ "s") : sectTransEvts])

eventsSectionsStatements :: [Section] -> Body
eventsSectionsStatements sects =
    let sectEvents = map (\s -> Block $ sectionLabelComment s : transEvents s) sects
        transEvents s = concatMap (\t -> eventsStatements' (nodeTransPreNode t) (nodeTransPostNode t) (nodeTransEvents t)) (sectionTrans s)
    in sectEvents

eventsStatements' :: Node -> Node -> [Event] -> [Statement]
eventsStatements' preN postN evts =
    let assignedEvts = zipWith (\e i -> addElem evtsName (litString $ eventLabel e) i) evts indices
        evtsName = evtsPrefix ++ (makeVarNameValid $ nodeLabel preN) ++ toLabel ++ (makeVarNameValid $ nodeLabel postN)
        indices = [0..length evts - 1]
    in listDec evtsName string (length evts) : assignedEvts

nodesStatements :: [Section] -> Body
nodesStatements ns =
    let commentLabel = nodeName ++ "s"
        sectNodes = concatMap (\n -> prefixFirstBlock (sectionLabelComment n) (nodesStatements' n)) ns
    in addComments commentLabel sectNodes

nodesStatements' :: Section -> Body
nodesStatements' (Section lab ns _) =
    let names = map (\n -> nodePrefix ++ (makeVarNameValid $ nodeLabel n)) ns
        nodesDec = Block $ zipWith (\n nam -> objDecDef nam node (litObj node [litString $ nodeLabel n])) ns names
        assignedNodes = zipWith (\n i -> addElem nodesName (Var n) i) names indices
        nodesName = nodesPrefix ++ makeVarNameValid lab
        indices = [0..length ns - 1]
    in nodesDec : [Block $ listDec nodesName node (length ns) : assignedNodes]

nodeTransStatements :: [Section] -> [SectionTransition] -> Body
nodeTransStatements ns ts =
    let commentLabel = nodeTransName ++ "s"
        allSects = concatMap (\n -> prefixFirstBlock (sectionLabelComment n) (nodeTransStatements' n (sectNodeTrans n))) ns
        sectNodeTrans n = concatMap (\t -> if sectionTransPreNode t `elem` sectionNodes n then [t] else []) ts
    in addComments commentLabel allSects

nodeTransStatements' :: Section -> [SectionTransition] -> Body
nodeTransStatements' (Section lab _ sectionTs) storyTs =
    let newTs = map (\t -> NodeTrans (sectionTransLabel t) (sectionTransPreNode t) (sectionTransPostNode t) (sectionTransEvents t)) storyTs
        ts = sectionTs ++ newTs
        preNs = map (\t -> (makeVarNameValid $ nodeLabel $ nodeTransPreNode t)) ts
        postNs = map (\t -> (makeVarNameValid $ nodeLabel $ nodeTransPostNode t)) ts
        names = zipWith (\n1 n2 -> n1 ++ toLabel ++ n2) preNs postNs
        transNames = map (\n -> transPrefix ++ n) names
        eventNames = map (\n -> evtsPrefix ++ n) names
        transArgs n1 n2 eName = litObj nodeTransition [Var $ nodePrefix ++ n1, Var $ nodePrefix ++ n2, Var eName]
        nodeTransDec = Block $ zipWith4 (\tName n1 n2 eName ->
            objDecDef tName nodeTransition (transArgs n1 n2 eName)) transNames preNs postNs eventNames
        nodeTransLabel = transPrefix ++ makeVarNameValid lab
        indices = [0..length ts - 1]
        assignedTrans = zipWith (\n i -> addElem nodeTransLabel (Var n) i) transNames indices
    in nodeTransDec : [Block $ listDec nodeTransLabel nodeTransition (length ts) : assignedTrans]

sectionStatements :: [Section] -> Body
sectionStatements ns =
    let commentLabel = sectName ++ "s"
        privVarName = makeVarNameValid . sectionLabel
        sectArgs n = litObj section [litString $ sectionLabel n, Var $ nodesPrefix ++ privVarName n, Var $ transPrefix ++ privVarName n]
        sectDecs = map (\n -> objDecDef (sectPrefix ++ privVarName n) section (sectArgs n)) ns
        assignedSects = Block $ zipWith (\n i -> addElem sections (Var $ sectPrefix ++ privVarName n) i) ns indices
        indices = [0..length ns - 1]
    in addComments commentLabel (Block (listDec sections section (length ns) : sectDecs) : [assignedSects])

outGoingStatements :: [Section] -> [SectionTransition] -> Body
outGoingStatements ns ts =
    let commentLabel = "Outgoing " ++ nodeTransName ++ "s and parent " ++ sectName ++ " of " ++ nodeName ++ "s"
        nodeTrans = "nodeTrans"
        nodeTransPrefix = nodeTrans ++ privVarSep
        indices = [0..length ns - 1]
        storyTs s = concatMap (\t ->
            if sectionTransPreNode t `elem` sectionNodes s
                then [NodeTrans (sectionTransLabel t) (sectionTransPreNode t) (sectionTransPostNode t) (sectionTransEvents t)] else []) ts
        allTrans s = sectionTrans s ++ storyTs s
        nodesWithTrans s = concatMap (\n -> if length (outTrans n (allTrans s)) > 0 then [n] else []) (sectionNodes s)
        indicesTrans s = [0..length (nodesWithTrans s) - 1]

        allSects = concatMap (\n -> Block (sectionLabelComment n : outGoingStatements' n ts) :
            [Block $ listDec (transName n) (List nodeTransition) (length $ nodesWithTrans n) : assignedTrans n]) ns
        assignedTrans n = zipWith (\node i -> addElem (transName n) (assignedElem node) i) (nodesWithTrans n) (indicesTrans n)
        transName n = nodeTransPrefix ++ (makeVarNameValid $ sectionLabel n)
        assignedElem node = Var $ outTransPrefix ++ (makeVarNameValid $ nodeLabel node)
        assignedNodeTrans = zipWith (\n i -> addElem nodeTrans (Var $ transName n) i) ns indices

        allSectTrans = Block $
            comment ("All " ++ sectName ++ "s " ++ nodeTransName ++ " data") :
            listDec nodeTrans (List $ List nodeTransition) (length ns) :
            assignedNodeTrans


        outGoingLoop =
            let i = "i"
                j = "j"
            in Block [
                for (varDecDef i int (litInt 0)) (Var i `less` (Var nodeTrans $. ListSize)) (plusPlus i) [
                    Block [
                        for (varDecDef j int (litInt 0)) (Var j `less` (Var nodeTrans $. at i $. ListSize)) (plusPlus j) [
                            Block [
                                ValState $ Var sections $. at i $. Func getNodes [] $. at j $. Func setNodeOutTrans [Var nodeTrans $. at i $. at j]
                            ]
                        ]
                    ]
                ]]

        parentLoop =
            let i = "i"
                j = "j"
            in Block [
                -- forEach section i (Var sections) [
                    -- Block [
                        -- forEach node j (Var i $. Func getNodes []) $ oneLiner $
                            -- ValState $ Var j $. Func setNodeParent [Var i]
                    -- ]
                -- ]
                for (varDecDef i int (litInt 0)) (Var i `less` (Var nodeTrans $. ListSize)) (plusPlus i) [
                    Block [
                        for (varDecDef j int (litInt 0)) (Var j `less` (Var nodeTrans $. at i $. ListSize)) (plusPlus j) $ oneLiner $
                            ValState $ Var sections $. at i $. Func getNodes [] $. at j $. Func setNodeParent [Var sections $. at i]
                    ]
                ]
            ]

    in addComments commentLabel (allSects ++ allSectTrans : outGoingLoop : [parentLoop])

outGoingStatements' :: Section -> [SectionTransition] -> [Statement]
outGoingStatements' (Section _ ns sectionTs) storyTs =
    let newTs = map (\t -> NodeTrans (sectionTransLabel t) (sectionTransPreNode t) (sectionTransPostNode t) (sectionTransEvents t)) storyTs
        ts = sectionTs ++ newTs
        sName = makeVarNameValid . nodeLabel
        assignedElem n t = Var $ transPrefix ++ sName n ++ toLabel ++ (makeVarNameValid $ nodeLabel $ nodeTransPostNode t)
        assignedTrans n = zipWith (\t i -> addElem (outTransPrefix ++ sName n) (assignedElem n t) i) (outTrans n ts) (tIndices n)
        tIndices n = [0..length (outTrans n ts) - 1]
        nodeOutTrans = concatMap (\n ->
            case length (outTrans n ts) of
                0 -> []
                _ -> concat [
                    [listDec (outTransPrefix ++ sName n) nodeTransition (length $ outTrans n ts)],
                    assignedTrans n]) ns
    in nodeOutTrans

outTrans :: Node -> [NodeTransition] -> [NodeTransition]
outTrans n = concatMap (\t -> if nodeTransPreNode t == n then [t] else [])

headStatements :: Node -> Body
headStatements n =
    let commentLabel = "Starting " ++ nodeName ++ " for the " ++ storyName
        headNode = [Block [objDecDef headVar node (Var $ nodePrefix ++ makeVarNameValid (nodeLabel n))]]
    in addComments commentLabel headNode

sectionTransStatements :: [SectionTransition] -> Body
sectionTransStatements ts =
    let commentLabel = sectTransName ++ "s"
        assignedTrans = zipWith (\t i -> addElem sectionTransVar (assignedElem t) i) ts indices
        sectLabel = makeVarNameValid . sectionLabel
        sectNodeLabel = makeVarNameValid . nodeLabel
        indices = [0..length ts - 1]
        preSection t = sectPrefix ++ (sectLabel $ sectionTransSectionPreNode t)
        postSection t = sectPrefix ++ (sectLabel $ sectionTransSectionPostNode t)
        evts t = evtsPrefix ++ (sectNodeLabel $ sectionTransPreNode t) ++ toLabel ++ (sectNodeLabel $ sectionTransPostNode t)
        assignedElem t = litObj sectionTransition [Var $ preSection t, Var $ postSection t, Var $ evts t]
    in addComments commentLabel [Block $ listDec sectionTransVar sectionTransition (length ts) : assignedTrans]

storyStatements :: Body
storyStatements =
    let commentLabel = storyName
        story = [Block [managerStoryVar $= (litObj StoryT params)]]
        params = [Var $ label, Var $ sections, Var $ headVar, Var $ sectionTransVar]
    in addComments commentLabel story

eventLabelsStatements :: [Event] -> Body
eventLabelsStatements es =
    let literals = map (LitStr . eventLabel) es
        evtLabels = "evtLabels"
        eventsArray = "eventsArray"
    in [
        Block [
            comment "Event labels data",
            listDecLiterals evtLabels eventsArray string literals,
            eventLabels $= Var evtLabels
        ]
    ]
