-- | The structure for a class of renderers is defined here.
module SAGA.CodeGeneration.LanguageRenderer (
    -- * Some Neccessary Data Types for Rendering
    StatementLocation(..), DecDef(..), FileType(..),

    -- * Generation Language Structuring
    Config(..), LanguageRenderer(..),

    -- * Language Parametric Functions
    bodyDoc, callFuncParamList, fileCode, funcAppDoc, funcListDoc,
    paramListDoc, stateListDoc, statementDoc, transTypeDoc, valueDoc,

    -- * Default Functions
    endStatement, litToValues, modDecDoc, modDecListDoc, namespace, new, scopeDoc
    ) where

import SAGA.Code (Code(..))
import SAGA.CodeGeneration.AbstractCode
import SAGA.StoryManager.DataTypes (Label)
import SAGA.StoryManager.Printing (blank,oneTab,oneTabbed,doubleQuotedText)

import Data.Maybe
import List (intersperse)
import Text.PrettyPrint.HughesPJ

data StatementLocation = Loop | NoLoop
data DecDef = Dec | Def
data FileType = Header | Source

-- | Configuration record (explicit dictionary) for a language
data Config = Config { ext :: Label, objAccess :: Doc, listN :: Doc,
    stateType :: StateType -> DecDef -> Doc,
    include :: Label -> Doc, package :: Label -> Doc,
    top :: Maybe FileType -> Label -> Doc,
    bottom :: Maybe FileType -> Doc,
    includeScope :: Scope -> Doc,
    iterForEachLabel :: Doc,
    iterInLabel :: Doc,
    listObj :: Doc
    }

-- | A rendering type class that will have instances for each desired generation language
class LanguageRenderer a where
    config :: a -> Config

    body :: a -> Maybe FileType -> Label -> [Module] -> Doc

    declaration :: a -> Declaration -> Doc

    fileName :: a -> Label -> [Label] -> String
    fileName _ _ = head

    funcDoc :: a -> Function -> Doc

    iteration :: a -> Iteration -> Doc
    iteration l (For initv guard update b) = vcat [
        text "for" <+> parens (statementDoc l Loop initv <> semi <+> valueDoc l guard <> semi <+> statementDoc l Loop update) <+> lbrace,
        oneTab $ bodyDoc l b,
        rbrace]
    iteration l (ForEach t var listVar b) = vcat [
        iterForEachLabel c <+> parens ((stateType c) t Dec <+> text var <+> iterInLabel c <+> valueDoc l listVar) <+> lbrace,
        oneTab $ bodyDoc l b,
        rbrace]
        where c = config l

    litDoc :: a -> Literal -> Doc
    litDoc _ (LitBool True) = text "true"
    litDoc _ (LitBool False) = text "false"
    litDoc _ (LitInt v) = int v
    litDoc _ (LitFloat v) = float v
    litDoc _ (LitChar v) = quotes $ char v
    litDoc _ (LitStr v) = doubleQuotedText v

    moduleDoc :: a -> Maybe FileType -> Label -> Module -> Doc
    moduleDoc l f _ (Mod n s vs fs) = vcat [
        scopeDoc s <+> text "class" <+> text n <+> lbrace,
        oneTabbed [
            transListDoc l f n fs,
            blank,
            stateListDoc l vs],
        rbrace]

    objAccessDoc :: a -> Value -> Function -> Doc
    objAccessDoc l v f = valueDoc l v <> funcDoc l f

    renderCode :: a -> AbstractCode -> Code
    renderCode l (AbsCode p) = Code [
        fileCode l p [nodeName] Nothing e,
        fileCode l p [nodeTransName] Nothing e,
        fileCode l p [sectName] Nothing e,
        fileCode l p [sectTransName] Nothing e,
        fileCode l p [storyName] Nothing e,
        fileCode l p [storyManagerName] Nothing e]
        where c = config l
              e = ext c

    transDoc :: a -> Maybe FileType -> Label -> Transformation -> Doc
    transDoc l _ _ (Transform n s t ps b) = vcat [
        scopeDoc s <+> transTypeDoc l t <+> text n <> parens (paramListDoc l ps) <+> lbrace,
        oneTab $ bodyDoc l b,
        rbrace]

    transListDoc :: a -> Maybe FileType -> Label -> [Transformation] -> Doc
    transListDoc l f m fs = vimap blank (transDoc l f m) fs

----------------------------------
-- CFamily Parametric Functions --
----------------------------------

assign :: LanguageRenderer a => a -> Assignment -> Doc
assign l (Assign v1 v2) = valueDoc l v1 <+> equals <+> valueDoc l v2
assign l (PlusEquals n v) = text n <+> text "+=" <+> valueDoc l v
assign _ (PlusPlus n) = text n <> text "++"

blockDoc :: LanguageRenderer a => a -> Block -> Doc
blockDoc l (Block ss) = vmap (statementDoc l NoLoop) ss

bodyDoc :: LanguageRenderer a => a -> Body -> Doc
bodyDoc l bs = vimap blank (blockDoc l) bs

callFuncParamList :: LanguageRenderer a => a -> [Value] -> Doc
callFuncParamList l vs = hcat $ intersperse (text ", ") (map (valueDoc l) vs)

conditional :: LanguageRenderer a => a -> Conditional -> Doc
conditional l (If (t:ts) elseBody) =
    let ifBrace = if null elseBody && null ts then rbrace else empty
        ifSect (v, b) = vcat [
            text "if" <+> parens (valueDoc l v) <+> lbrace,
            oneTab $ bodyDoc l b,
            ifBrace]
        elseIfSect (v, b) = vcat [
            rbrace <+> text "else if" <+> parens (valueDoc l v) <+> lbrace,
            oneTab $ bodyDoc l b,
            rbrace]
        elseSect = if null elseBody then empty else vcat [
            rbrace <+> text "else" <+> lbrace,
            oneTab $ bodyDoc l elseBody,
            rbrace]
    in vcat [
        ifSect t,
        vcat $ map elseIfSect ts,
        elseSect]
conditional _ (If [] _) = error "If with no body encountered"
conditional _ (Switch _ _) = error "Switch not implemented"

exprDoc :: LanguageRenderer a => a -> Expression -> Doc
exprDoc l (UnaryExpr r v) = unRelDoc r <> parens (valueDoc l v)
exprDoc l (BinaryExpr v1 r v2) = valueDoc l v1 <+> binRelDoc r <+> valueDoc l v2

fileCode :: LanguageRenderer a => a -> Package -> [Label] -> Maybe FileType -> Label -> (FilePath, Doc)
fileCode l (Pack p ms) ns f e = (fileName l p ns ++ e, fileDoc l c f p (map (modWithName ms) ns))
    where c = config l

fileDoc :: LanguageRenderer a => a -> Config -> Maybe FileType -> Label -> [Module] -> Doc
fileDoc l c f p ms = vcat $ intersperse blank [
    top c f p,
    body l f p ms,
    bottom c f]

funcAppDoc :: LanguageRenderer a => a -> Label -> [Value] -> Doc
funcAppDoc l n vs = text n <> parens (callFuncParamList l vs)

funcListDoc :: LanguageRenderer a => a -> [Function] -> Doc
funcListDoc l fs = hcat $ map (funcDoc l) fs

paramDoc :: LanguageRenderer a => a -> Parameter -> Doc
paramDoc l (StateParam n t) = (stateType c) t Dec <+> text n
    where c = config l
paramDoc _ (FuncParam _ _ _) = error "FuncParam not yet rendered"

paramListDoc :: LanguageRenderer a => a -> [Parameter] -> Doc
paramListDoc l ps = hcat $ intersperse (text ", ") (map (paramDoc l) ps)

ret :: LanguageRenderer a => a -> Return -> Doc
ret l (Ret v) = text "return" <+> valueDoc l v

stateDoc :: LanguageRenderer a => a -> State -> Doc
stateDoc l (State n s t) = includeScope c s <+> (stateType c) t Dec <+> text n <> endStatement
    where c = config l

stateListDoc :: LanguageRenderer a => a -> [State] -> Doc
stateListDoc l = vmap (stateDoc l)

statementDoc :: LanguageRenderer a => a -> StatementLocation -> Statement -> Doc
statementDoc l loc (AssignState s) = assign l s <> end loc
statementDoc l loc (DeclState s) = declaration l s <> end loc
statementDoc l _ (CondState s) = conditional l s
statementDoc l _ (IterState s) = iteration l s
statementDoc _ loc (JumpState s) = jump s <> end loc
statementDoc l loc (RetState s) = ret l s <> end loc
statementDoc l loc (ValState s) = valueDoc l s <> end loc
statementDoc _ _ (CommentState s) = comment s

transTypeDoc :: LanguageRenderer a => a -> TransType -> Doc
transTypeDoc l (TState t) = stateType c t Dec
    where c = config l
transTypeDoc _ Void = text "void"
transTypeDoc _ (Construct _) = empty

valueDoc :: LanguageRenderer a => a -> Value -> Doc
valueDoc l (Lit v) = litDoc l v
valueDoc _ (Var v) = text v
valueDoc l (FuncApp n vs) = funcAppDoc l n vs
valueDoc l (ObjAccess v f) = objAccessDoc l v f
valueDoc l (Expr v) = exprDoc l v
valueDoc l (StateObj t@(List _) vs) = listObj c <> (stateType c) t Def <> parens (callFuncParamList l vs)
    where c = config l
valueDoc l (StateObj t vs) = new <+> (stateType c) t Def <> parens (callFuncParamList l vs)
    where c = config l

-----------------------------------------------
-- 'Default' functions used in the renderers --
-----------------------------------------------

endStatement :: Doc
endStatement = semi

litToValues :: [Literal] -> [Value]
litToValues ls = map Lit ls

modDecDoc :: Module -> Doc
modDecDoc (Mod n _ _ _) = text "class" <+> text n <> endStatement

modDecListDoc :: [Module] -> Doc
modDecListDoc = vmap modDecDoc

-- for 'packages' which are namespaces
namespace :: Label -> Doc
namespace n = text "namespace" <+> text n

new :: Doc
new = text "new"

scopeDoc :: Scope -> Doc
scopeDoc Private = text "private"
scopeDoc Public = text "public"

----------------------
-- Helper Functions --
----------------------

binRelDoc :: BinaryRelation -> Doc
binRelDoc Equal = text "=="
binRelDoc NotEqual = text "!="
binRelDoc Greater = text ">"
binRelDoc GreaterEqual = text ">="
binRelDoc Less = text "<"
binRelDoc LessEqual = text "<="

comment :: Comment -> Doc
comment (Comment c) = text "//" <+> text c
comment (CommentDelimit c len) =
    let com = "//" ++ " " ++ c ++ " "
    in text com <> text (dashes com len)

dashes :: String -> Int -> String
dashes s l = take (l - length s) (repeat '-')

end :: StatementLocation -> Doc
end Loop = empty
end NoLoop = endStatement

jump :: Jump -> Doc
jump Break = text "break"
jump Continue = text "continue"

modWithName :: [Module] -> Label -> Module
modWithName (m:ms) n = if moduleName m == n then m else modWithName ms n
modWithName [] _ = error "uhoh, module was not found"

unRelDoc :: UnaryRelation -> Doc
unRelDoc Negation = text "!"

vmap :: (a -> Doc) -> [a] -> Doc
vmap f l = vcat $ map f l

vimap :: Doc -> (a -> Doc) -> [a] -> Doc
vimap c f l = vcat $ intersperse c (map f l)
