-- | The logic to render C# code from an 'AbstractCode' is contained in this module
module SAGA.CodeGeneration.LanguageRenderer.CSharpRenderer (
    -- * C# Rendering Instance
    CSharpRenderer(..)
    ) where

import SAGA.CodeGeneration.AbstractCode
import SAGA.CodeGeneration.LanguageRenderer
import SAGA.StoryManager.Printing (angles,blank,squares,oneTab)

import List (intersperse)
import Text.PrettyPrint.HughesPJ

data CSharpRenderer = CSharpRenderer

csharpConfig :: Config
csharpConfig = Config {ext = ".cs", objAccess = text ".", listN = text "List",
    iterForEachLabel = text "foreach", iterInLabel = text "in", listObj = new <+> text " ",
    include = incl, package = namespace, top = cstop, bottom = \_ -> empty,
    stateType = csstateType, includeScope = scopeDoc }

-- convenience
dot, list :: Doc
dot = objAccess csharpConfig
list = listN csharpConfig

incl :: String -> Doc
incl n = text "using" <+> text n

cstop :: a -> b -> Doc
cstop _ _ = vcat [
    incl "System" <> endStatement,
    incl "System.Collections.Generic" <> endStatement]

csstateType :: StateType -> DecDef -> Doc
csstateType (List t@(List _)) _ = list <> angles (space <> csstateType t Dec <> space)
csstateType (List t) _ = list <> angles (csstateType t Dec)
csstateType (Base Boolean) _ = text "Boolean"
csstateType (Base Integer) _ = text "int"
csstateType (Base Float) _ = text "float"
csstateType (Base Character) _ = text "char"
csstateType (Base String) _ = text "string"
csstateType NodeT _ = text nodeName
csstateType NodeTransT _ = text nodeTransName
csstateType SectT _ = text sectName
csstateType SectTransT _ = text sectTransName
csstateType StoryT _ = text storyName
csstateType StoryManT _ = text storyManagerName

instance LanguageRenderer CSharpRenderer where
    config _ = csharpConfig

    body l _ p ms = vcat [
        namespace p <+> lbrace,
        oneTab $ vcat $ intersperse blank (map (moduleDoc l Nothing p) ms),
        rbrace]

    declaration _ (VarDec n t) = csstateType t Dec <+> text n
    declaration _ (ListDec n t s) = csstateType (List t) Dec <+> text n <+> equals <+> new <+> csstateType (List t) Dec <> parens (int s)
    declaration l (ListDecLiterals n _ t vs) = csstateType (List t) Dec <+> text n <+> equals <+> new <+> csstateType (List t) Dec <+> braces (callFuncParamList l (litToValues vs))
    declaration l (VarDecDef n t v) = csstateType t Dec <+> text n <+> equals <+> valueDoc l v
    declaration l (ObjDecDef n t v) = csstateType t Dec <+> text n <+> equals <+> valueDoc l v

    funcDoc l (Func n vs) = dot <> funcAppDoc l n vs
    funcDoc l (IndexOf var) = dot <> text "IndexOf" <> parens (valueDoc l var)
    funcDoc _ ListSize = dot <> text "Count"
    funcDoc l (ListAccess i) = squares $ valueDoc l i
    funcDoc l (ListAdd v i) = dot <> text "Insert" <> parens (int i <> comma <+> valueDoc l v)
    funcDoc _ (IterBegin) = error "IterBegin in CSharp?"
    funcDoc _ (IterEnd) = error "IterEnd in CSharp?"
