-- | The logic to render C++ code from an 'AbstractCode' is contained in this module
module SAGA.CodeGeneration.LanguageRenderer.CppRenderer (
    -- * C++ Rendering Instance
    CppRenderer(..)
    ) where

import SAGA.Code (Code(..))
import SAGA.CodeGeneration.AbstractCode
import SAGA.CodeGeneration.LanguageRenderer
import SAGA.StoryManager.Printing (angles,blank,oneTab,oneTabbed,doubleQuotedText)
import SAGA.StoryManager.DataTypes

import List (intersperse)
import Text.PrettyPrint.HughesPJ hiding (Str)


data CppRenderer = CppRenderer

cppConfig :: Config
cppConfig = Config {ext=".cpp", objAccess = text ".", listN = text "vector",
    iterForEachLabel = empty, iterInLabel = empty, listObj = empty,
    include = incl, package = namespace, top = cpptop, bottom = cppbottom,
    stateType = cppstateType, includeScope = \_ -> empty}

cppHeaderExt :: String
cppHeaderExt = ".hpp"
ptrAccess :: Doc
ptrAccess = text "->"

-- for convenience
dot, list :: Doc
dot = objAccess cppConfig
list = listN cppConfig

-- pull these out of the class
incl :: String -> Doc
incl n = text "#include" <+> text n

cpptop (Just Header) p = vcat [
    text "#ifndef" <+> text p <> text "_h",
    text "#define" <+> text p <> text "_h",
    blank,
    incl "<string>",
    incl "<vector>",
    blank,
    usingNameSpace "std" (Just "string"),
    usingNameSpace "std" (Just "vector")]
cpptop (Just Source) p = vcat [
    incl ("\"" ++ p ++ cppHeaderExt ++ "\""),
    incl "<algorithm>",
    incl "<string>",
    incl "<vector>",
    blank,
    usingNameSpace p Nothing,
    usingNameSpace "std" (Just "string"),
    usingNameSpace "std" (Just "vector")]
cpptop Nothing _ = error "cpp insists on either Source or Header"

cppbottom (Just Header) = text "#endif"
cppbottom (Just Source) = empty
cppbottom (Nothing)     = error "should have failed on cpptop already!"

cppstateType :: StateType -> DecDef -> Doc
cppstateType (List t@(List _)) _ = list <> angles (space <> cppstateType t Dec <> space)
cppstateType (List t) _ = list <> angles (cppstateType t Dec)
cppstateType (Base Boolean) _ = text "bool"
cppstateType (Base Integer) _ = text "int"
cppstateType (Base Float) _ = text "float"
cppstateType (Base Character) _ = text "char"
cppstateType (Base String) _ = text "string"
cppstateType NodeT Dec = text nodeName <> text "*"
cppstateType NodeTransT Dec = text nodeTransName <> text "*"
cppstateType SectT Dec = text sectName <> text "*"
cppstateType SectTransT Dec = text sectTransName <> text "*"
cppstateType StoryT Dec = text storyName <> text "*"
cppstateType StoryManT Dec = text storyManagerName <> text "*"
cppstateType NodeT _ = text nodeName
cppstateType NodeTransT _ = text nodeTransName
cppstateType SectT _ = text sectName
cppstateType SectTransT _ = text sectTransName
cppstateType StoryT _ = text storyName
cppstateType StoryManT _ = text storyManagerName


instance LanguageRenderer CppRenderer where
    config _ = cppConfig

    body l f@(Just Header) p ms = vcat [
        namespace p <+> lbrace,
        oneTabbed [
            modDecListDoc ms,
            blank,
            vcat $ intersperse blank (map (moduleDoc l f p) ms)],
        rbrace]
    body l f@(Just Source) p ms = vcat $ intersperse blank (map (moduleDoc l f p) ms)
    body _ Nothing _ _ = error "CPP needs Header/Source"

    declaration _ (VarDec n t) = cppstateType t Dec <+> text n
    declaration _ (ListDec n t s) = cppstateType (List t) Dec <+> text n <> parens (int s)
    declaration l (ListDecLiterals n temp t vs) = vcat [
        cppstateType t Dec <+> text temp <> text "[]" <+> equals <+> braces (callFuncParamList l (litToValues vs)) <> endStatement,
        cppstateType (List t) Dec <+> text n <> parens(text temp <> comma <+> text temp <+> text "+" <+> text "sizeof" <> parens (text temp) <+> text "/" <+> text "sizeof" <> parens (text temp <> text "[0]"))]
    declaration l (VarDecDef n t v) = cppstateType t Dec <+> text n <+> equals <+> valueDoc l v
    declaration l (ObjDecDef n t v) = cppstateType t Dec <+> text n <+> equals <+> valueDoc l v

    fileName _ p _ = p

    funcDoc l (Func n vs) = ptrAccess <> funcAppDoc l n vs
    funcDoc l (IndexOf var) = funcAppDoc l "find" [Var eventLabels $. IterBegin, Var eventLabels $. IterEnd, var] <+> text "-" <+> valueDoc l (Var eventLabels $. IterBegin)
    funcDoc _ ListSize = dot <> text "size()"
    funcDoc l (ListAccess i) = dot <> text "at" <> parens (valueDoc l i)
    funcDoc l (ListAdd v i) = dot <> text "at" <> parens (int i) <+> equals <+> valueDoc l v
    funcDoc l (IterBegin) = dot <> funcAppDoc l "begin" []
    funcDoc l (IterEnd) = dot <> funcAppDoc l "end" []

    iteration l (For init guard update b) = vcat [
        text "for" <+> parens (statementDoc l Loop init <> semi <+> valueDoc l guard <> semi <+> statementDoc l Loop update) <+> lbrace,
        oneTab $ bodyDoc l b,
        rbrace]
    iteration l (ForEach _ oldV listVar bs) =
        let newV = listVar $. at oldV
            init = varDecDef oldV (Base Integer) (litInt 0)
            guard = Var oldV `less` (listVar $. ListSize)
            update = plusPlus oldV
            newBody = forEachBodyToFor (Var oldV) newV bs
        in iteration l (For init guard update newBody)
    -- iteration l i@(ForEach _ oldV listVar _) = iteration l (fixIter (Var oldV) newV i)
        -- where newV = listVar $. at oldV

    litDoc _ (LitBool True) = text "true"
    litDoc _ (LitBool False) = text "false"
    litDoc _ (LitInt v) = int v
    litDoc _ (LitFloat v) = float v
    litDoc _ (LitChar v) = quotes $ char v
    litDoc _ (LitStr v) = doubleQuotedText v
    -- litDoc l (StateObj t@(List _) vs) = cppstateType t Def <> parens (callFuncParamList l vs)
    -- litDoc l (StateObj t vs) = text "new" <+> cppstateType t Def <> parens (callFuncParamList l vs)

    moduleDoc l f@(Just Header) _ (Mod n _ vs fs) =
        let pubFuncs = concatMap (\f@(Transform _ s _ _ _) -> if s == Public then [f] else []) fs
            pubVars = concatMap (\v@(State _ s _) -> if s == Public then [v] else []) vs
            privFuncs = concatMap (\f@(Transform _ s _ _ _) -> if s == Private then [f] else []) fs
            privVars = concatMap (\v@(State _ s _) -> if s == Private then [v] else []) vs
            pubBlank = if null pubVars then empty else blank
            privBlank = if null privFuncs then empty else blank
        in vcat [
            text "class" <+> text n <+> lbrace,
            oneTabbed [
                scopeDoc Public <> colon,
                oneTabbed [
                    stateListDoc l pubVars,
                    pubBlank,
                    transListDoc l f n pubFuncs],
                blank,
                scopeDoc Private <> colon,
                oneTabbed [
                    stateListDoc l privVars,
                    privBlank,
                    transListDoc l f n privFuncs]],
            rbrace <> endStatement]
    moduleDoc l f@(Just Source) _ (Mod n _ _ fs) = transListDoc l f n fs

    objAccessDoc l _ f@(IndexOf _) = funcDoc l f
    objAccessDoc l v f = valueDoc l v <> funcDoc l f

    renderCode l (AbsCode p) = Code [
        fileCode l p [nodeName,nodeTransName,sectName,sectTransName,storyName,storyManagerName] (Just Header) cppHeaderExt,
        fileCode l p [nodeName,nodeTransName,sectName,sectTransName,storyName,storyManagerName] (Just Source) (ext c)]
        where c = config l

    transDoc l (Just Header) _ (Transform n _ t ps _) = transTypeDoc l t <+> text n <> parens (paramListDoc l ps) <> endStatement
    transDoc l (Just Source) m (Transform n _ t ps b) = vcat [
        transTypeDoc l t <+> text m <> colon <> colon <> text n <> parens (paramListDoc l ps) <+> lbrace,
        oneTab $ bodyDoc l b,
        rbrace]

    transListDoc l f@(Just Header) m fs = vcat $ map (transDoc l f m) fs
    transListDoc l f m fs = vcat $ intersperse blank (map (transDoc l f m) fs)

----------------------
-- Helper Functions --
----------------------

-- <<<<<<< .mine
-- Must look for "oldV" in the body, and replace all "oldV" with "newV" (someContainerVariable.at(oldV))
forEachBodyToFor :: Value -> Value -> Body -> Body
forEachBodyToFor oldV newV bs = map (forEachBlockToFor oldV newV) bs
-- =======
-- forEachBodyToFor :: Label -> Body -> Body
-- forEachBodyToFor v bs = map (forEachBlockToFor v) bs
-- >>>>>>> .r263

-- <<<<<<< .mine
forEachBlockToFor :: Value -> Value -> Block -> Block
forEachBlockToFor oldV newV (Block ss) = Block $ map (forEachStatementToFor oldV newV) ss
-- =======
-- forEachBlockToFor :: Label -> Block -> Block
-- forEachBlockToFor v (Block ss) = Block $ map (forEachStatementToFor v) ss
-- >>>>>>> .r263

-- <<<<<<< .mine
forEachStatementToFor :: Value -> Value -> Statement -> Statement
forEachStatementToFor oldV newV (AssignState a) = AssignState $ fixAssign oldV newV a
forEachStatementToFor oldV newV (DeclState d) = DeclState $ fixDecl oldV newV d
forEachStatementToFor oldV newV (CondState c) = CondState $ fixCond oldV newV c
forEachStatementToFor oldV newV (IterState i) = IterState $ fixIter oldV newV i
forEachStatementToFor oldV newV s@(JumpState j) = s
forEachStatementToFor oldV newV (RetState r) = RetState $ fixRet oldV newV r
forEachStatementToFor oldV newV (ValState val) = ValState $ fixVal oldV newV val
forEachStatementToFor oldV newV s@(CommentState c) = s
-- =======
-- forEachStatementToFor :: Label -> Statement -> Statement
-- forEachStatementToFor v (AssignState a) = AssignState $ fixAssign v a
-- forEachStatementToFor v (DeclState d) = DeclState $ fixDecl v d
-- forEachStatementToFor v (CondState c) = CondState $ fixCond v c
-- forEachStatementToFor v (IterState i) = IterState $ fixIter v i
-- forEachStatementToFor _ s@(JumpState _) = s
-- forEachStatementToFor v (RetState r) = RetState $ fixRet v r
-- forEachStatementToFor v (ValState val) = ValState $ fixVal v val
-- forEachStatementToFor _ s@(CommentState _) = s
-- >>>>>>> .r263

-- <<<<<<< .mine
fixAssign :: Value -> Value -> Assignment -> Assignment
fixAssign oldV newV (Assign v1 v2) | oldV == v1 = Assign newV (fixVal oldV newV v2)
fixAssign oldV newV (Assign v1 v2) | oldV == v2 = Assign (fixVal oldV newV v1) newV
fixAssign oldV newV (Assign v1 v2)              = Assign (fixVal oldV newV v1) (fixVal oldV newV v2)
fixAssign oldV newV (PlusEquals n val) | oldV == val = PlusEquals n newV
fixAssign oldV newV (PlusEquals n val)               = PlusEquals n (fixVal oldV newV val)
fixAssign _ _ a@(PlusPlus _) = a
-- =======
-- fixAssign :: Label -> Assignment -> Assignment
-- fixAssign v (Assign v1 v2) | v == v1 = Assign v v2
-- fixAssign v (Assign v1 v2)           = Assign v1 (fixVal v v2)
-- >>>>>>> .r263

-- <<<<<<< .mine
fixDecl :: Value -> Value -> Declaration -> Declaration
fixDecl _ _ d@(VarDec _ _) = d
fixDecl _ _ d@(ListDec _ _ _) = d
fixDecl oldV newV (ListDecLiterals n temp t ls) = ListDecLiterals n temp t (map (fixLit oldV newV) ls)
fixDecl oldV newV (VarDecDef n t val) | oldV == val = VarDecDef n t newV
fixDecl oldV newV (VarDecDef n t val)               = VarDecDef n t (fixVal oldV newV val)
fixDecl oldV newV (ObjDecDef n t val) | oldV == val = ObjDecDef n t newV
fixDecl oldV newV (ObjDecDef n t val)               = ObjDecDef n t (fixVal oldV newV val)
-- =======
-- -- fixAssign v (PlusEquals n val) | (Var v) == val = PlusEquals n v
-- fixAssign _ a@(PlusEquals _ _)                  = a

-- fixAssign _ a@(PlusPlus _) = a

-- fixDecl :: Label -> Declaration -> Declaration
-- fixDecl _ d@(VarDec _ _) = d
-- fixDecl _ d@(ListDec _ _ _) = d
-- fixDecl _ (ListDecLiterals n temp t ls) = ListDecLiterals n temp t ls
-- -- fixDecl v (VarDecDef n t val) | v == val = VarDecDef n t v
-- fixDecl v (VarDecDef n t val)            = VarDecDef n t (fixVal v val)
-- -- fixDecl v (ObjDecDef n t val) | v == val = ObjDecDef n t v
-- fixDecl v (ObjDecDef n t val)            = ObjDecDef n t (fixVal v val)
-- >>>>>>> .r263

-- <<<<<<< .mine
fixCond :: Value -> Value -> Conditional -> Conditional
fixCond oldV newV (If [(val, b)] elseBody)
    | oldV == val = If [(newV, forEachBodyToFor oldV newV b)] (forEachBodyToFor oldV newV elseBody)
fixCond oldV newV (If [(val, b)] elseBody) =
    If [(fixVal oldV newV val, forEachBodyToFor oldV newV b)] (forEachBodyToFor oldV newV elseBody)
-- fixCond oldV c@() =
-- =======
-- fixCond :: Label -> Conditional -> Conditional
-- -- fixCond v (If [(val, b)] elseBody)
-- --     | v == val = If [(v, forEachBodyToFor v b)] (forEachBodyToFor v elseBody)
-- fixCond v (If [(val, b)] elseBody) =
    -- If [(fixVal v val, forEachBodyToFor v b)] (forEachBodyToFor v elseBody)
-- -- fixCond v c@() =
-- >>>>>>> .r263

-- <<<<<<< .mine
fixIter :: Value -> Value -> Iteration -> Iteration
fixIter oldV newV (For s1 val s2 b)
   | oldV == val = For (forEachStatementToFor oldV newV s1) newV (forEachStatementToFor oldV newV s2) (forEachBodyToFor oldV newV b)
fixIter oldV newV (For s1 val s2 b) =
    For (forEachStatementToFor oldV newV s1) (fixVal oldV newV val) (forEachStatementToFor oldV newV s2) (forEachBodyToFor oldV newV b)
-- -- fixIter oldV newV (ForEach t v1 v2 b)
    -- -- | oldV == v1 = ForEach t newV (fixVal oldV newV v2) (forEachBodyToFor oldV newV b)
-- -- fixIter oldV newV (ForEach t v1 v2 b)
    -- -- | oldV == v2 = ForEach t (fixVal oldV newV v1) newV (forEachBodyToFor oldV newV b)
-- -- fixIter oldV newV (ForEach t v1 v2 b) =
    -- -- ForEach t (fixVal oldV newV (Var v1)) (fixVal oldV newV v2) (forEachBodyToFor oldV newV b)


-- fixIter oldV@(Var old) newV (ForEach _ v listVar b) | old == v =
    -- let init = varDecDef old (Base Integer) (litInt 0)
        -- guard = oldV `less` ((fixVal oldV newV listVar) $. ListSize)
        -- update = plusPlus old
        -- newBody = forEachBodyToFor oldV newV b
    -- in For init guard update newBody
-- fixIter oldV newV (ForEach _ v listVar b) | oldV == listVar =
    -- let init = varDecDef v (Base Integer) (litInt 0)
        -- guard = Var v `less` (oldV $. ListSize)
        -- update = plusPlus v
        -- newBody = forEachBodyToFor oldV newV b
    -- in For init guard update newBody
-- fixIter oldV newV (ForEach _ v listVar b) =
    -- let init = varDecDef v (Base Integer) (litInt 0)
        -- guard = Var v `less` ((fixVal oldV newV listVar) $. ListSize)
        -- update = plusPlus v
        -- newBody = forEachBodyToFor oldV newV b
    -- in For init guard update newBody

-- =======
-- fixIter :: Label -> Iteration -> Iteration
-- -- fixIter v (For s1 val s2 b)
-- --    | v == val = For (forEachStatementToFor v s1) v (forEachStatementToFor v s2) (forEachBodyToFor v b)
-- -- fixIter v (For s1 val s2 b) =
    -- -- For (forEachStatementToFor v s1) (fixVal v val) (forEachStatementToFor v s2) (forEachBodyToFor v b)
-- -- fixIter v (ForEach t v1 v2 b)
    -- -- | v == v1 = ForEach t v (fixVal v v2) (forEachBodyToFor v b)
-- -- fixIter v (ForEach t v1 v2 b)
-- --     | v == v2 = ForEach t (fixVal v v1) v (forEachBodyToFor v b)
-- -- fixIter v (ForEach t v1 v2 b) =
    -- -- ForEach t v1 (fixVal v v2) (forEachBodyToFor v b)
-- >>>>>>> .r263

-- <<<<<<< .mine
fixRet :: Value -> Value -> Return -> Return
fixRet oldV newV (Ret val) | oldV == val = Ret newV
fixRet oldV newV (Ret val)               = Ret $ fixVal oldV newV val
-- =======
-- fixRet :: Label -> Return -> Return
-- -- fixRet v (Ret val) | v == val = Ret v
-- fixRet v (Ret val)            = Ret $ fixVal v val
-- >>>>>>> .r263

-- <<<<<<< .mine
fixVal :: Value -> Value -> Value -> Value
fixVal oldV newV (Lit l) = Lit $ fixLit oldV newV l
fixVal oldV newV v@(Var _) | oldV == v = newV
fixVal oldV newV v@(Var _)             = v
fixVal oldV newV (FuncApp n vs) = FuncApp n (map (fixVal oldV newV) vs)
fixVal oldV newV (ObjAccess val f) | oldV == val = ObjAccess newV (fixFunc oldV newV f)
fixVal oldV newV (ObjAccess val f)               = ObjAccess (fixVal oldV newV val) (fixFunc oldV newV f)
fixVal oldV newV (StateObj t vs) = StateObj t (map (fixVal oldV newV) vs)
fixVal oldV newV (Expr e) = Expr $ fixExpr oldV newV e
-- =======
-- fixVal :: Label -> Value -> Value
-- fixVal _ l@(Lit _) = l
-- fixVal _ oldV@(Var _) = oldV
-- fixVal v (FuncApp n vs) = FuncApp n (map (fixVal v) vs)
-- -- fixVal v (ObjAccess val f) | v == val = ObjAccess v (fixFunc v f)
-- fixVal v (ObjAccess val f)               = ObjAccess val (fixFunc v f)
-- fixVal v (Expr e) = Expr $ fixExpr v e
-- >>>>>>> .r263

-- <<<<<<< .mine
fixFunc :: Value -> Value -> Function -> Function
fixFunc oldV newV (Func n vs) = Func n (map (fixVal oldV newV) vs)
fixFunc oldV newV (IndexOf val) | oldV == val = IndexOf newV
fixFunc oldV newV (IndexOf val)               = IndexOf (fixVal oldV newV val)
fixFunc _ _ f@(ListSize) = f
fixFunc oldV newV (ListAccess val) | oldV == val = ListAccess newV
fixFunc oldV newV (ListAccess val)               = ListAccess $ fixVal oldV newV val
fixFunc oldV newV (ListAdd val i) | oldV == val = ListAdd newV i
fixFunc oldV newV (ListAdd val i)               = ListAdd (fixVal oldV newV val) i
fixFunc _ _ f@(IterBegin) = f
fixFunc _ _ f@(IterEnd) = f
-- =======
-- fixFunc :: Label -> Function -> Function
-- fixFunc v (Func n vs) = Func n (map (fixVal v) vs)
-- -- fixFunc v (IndexOf val) | v == val = IndexOf v
-- fixFunc v (IndexOf val)            = IndexOf (fixVal v val)
-- fixFunc _ f@(ListSize) = f
-- -- fixFunc v (ListAccess val) | v == val = ListAccess v
-- fixFunc v (ListAccess val)            = ListAccess $ fixVal v val
-- -- fixFunc v (ListAdd val i) | v == val = ListAdd v i
-- fixFunc v (ListAdd val i)            = ListAdd (fixVal v val) i
-- fixFunc _ f@(IterBegin) = f
-- fixFunc _ f@(IterEnd) = f
-- >>>>>>> .r263

-- <<<<<<< .mine
fixExpr :: Value -> Value -> Expression -> Expression
fixExpr oldV newV (UnaryExpr r val) | oldV == val = UnaryExpr r newV
fixExpr oldV newV (UnaryExpr r val)               = UnaryExpr r (fixVal oldV newV val)
fixExpr oldV newV (BinaryExpr v1 r v2)
    | oldV == v1 = BinaryExpr newV r (fixVal oldV newV v2)
fixExpr oldV newV (BinaryExpr v1 r v2)
    | oldV == v2 = BinaryExpr (fixVal oldV newV v1) r newV
fixExpr oldV newV (BinaryExpr v1 r v2) =
                BinaryExpr (fixVal oldV newV v1) r (fixVal oldV newV v2)
-- =======
-- fixExpr :: Label -> Expression -> Expression
-- -- fixExpr v (UnaryExpr r val) | v == val = UnaryExpr r v
-- fixExpr v (UnaryExpr r val)            = UnaryExpr r (fixVal v val)
-- -- fixExpr v (BinaryExpr v1 r v2)
-- --     | v == v1 = BinaryExpr v r (fixVal v v2)
-- -- fixExpr v (BinaryExpr v1 r v2)
-- --     | v == v2 = BinaryExpr (fixVal v v1) r v2
-- fixExpr v (BinaryExpr v1 r v2) =
                -- BinaryExpr (fixVal v v1) r (fixVal v v2)
-- >>>>>>> .r263

-- <<<<<<< .mine
fixLit :: Value -> Value -> Literal -> Literal
fixLit _ _ l@(LitBool _) = l
fixLit _ _ l@(LitInt _) = l
fixLit _ _ l@(LitFloat _) = l
fixLit _ _ l@(LitChar _) = l
fixLit _ _ l@(LitStr _) = l
-- fixLit oldV newV (StateObj t vs) = StateObj t (map (fixVal oldV newV) vs)
-- =======
-- -- fixLit v (StateObj t vs) = StateObj t (map (fixVal v) vs)
-- >>>>>>> .r263

usingNameSpace :: String -> Maybe String -> Doc
usingNameSpace n (Just m) = text "using" <+> text n <> colon <> colon <> text m <> endStatement
usingNameSpace n Nothing = text "using namespace" <+> text n <> endStatement
