-- | The logic to render Java code from an 'AbstractCode' is contained in this module
module SAGA.CodeGeneration.LanguageRenderer.JavaRenderer (
    -- * Java Rendering Instance
    JavaRenderer(..)
    ) where

import SAGA.CodeGeneration.AbstractCode
import SAGA.CodeGeneration.LanguageRenderer
import SAGA.StoryManager.Printing (angles,blank)

import List (intersperse)
import Text.PrettyPrint.HughesPJ

data JavaRenderer = JavaRenderer

javaConfig :: Config
javaConfig = Config {ext = ".java", objAccess = text ".",
    listN = text "Vector",
    include = incl,
    package = pack,
    top = jtop,
    bottom = \_ -> empty,
    stateType = jstateType,
    includeScope = scopeDoc,
    iterForEachLabel = text "for",
    iterInLabel = colon,
    listObj = new <+> text " "
}

-- convenience
dot = objAccess javaConfig
list = listN javaConfig

-- short names, packaged up above (and used below)
incl n = text "import" <+> text n
pack n = text "package" <+> text n

jtop _ p = vcat [
    pack p <> endStatement,
    blank,
    incl "java.util.Arrays" <> endStatement,
    incl "java.util.Vector" <> endStatement]

jstateType :: StateType -> DecDef -> Doc
jstateType (List t@(List _)) _ = list <> angles (space <> jstateType t Dec <> space)
jstateType (List t) _ = list <> angles (jstateType t Dec)
jstateType (Base Boolean) _ = text "Boolean"
jstateType (Base Integer) _ = text "int"
jstateType (Base Float) _ = text "float"
jstateType (Base Character) _ = text "char"
jstateType (Base String) _ = text "String"
jstateType NodeT _ = text nodeName
jstateType NodeTransT _ = text nodeTransName
jstateType SectT _ = text sectName
jstateType SectTransT _ = text sectTransName
jstateType StoryT _ = text storyName
jstateType StoryManT _ = text storyManagerName

instance LanguageRenderer JavaRenderer where
    config _ = javaConfig

    body l _ p ms = vcat $ intersperse blank (map (moduleDoc l Nothing p) ms)

    declaration _ (VarDec n t) = jstateType t Dec <+> text n
    declaration _ (ListDec n t s) = jstateType (List t) Dec <+> text n <+> equals <+> new <+> jstateType (List t) Dec <> parens (int s)
    declaration l (ListDecLiterals n _ t vs) = jstateType (List t) Dec <+> text n <+> equals <+> new <+> jstateType (List t) Dec <> parens (text "Arrays.asList" <> parens (callFuncParamList l (litToValues vs)))
    declaration l (VarDecDef n t v) = jstateType t Dec <+> text n <+> equals <+> valueDoc l v
    declaration l (ObjDecDef n t v) = jstateType t Dec <+> text n <+> equals <+> valueDoc l v

    funcDoc l (Func n vs) = dot <> funcAppDoc l n vs
    funcDoc l (IndexOf var) = dot <> text "indexOf" <> parens (valueDoc l var)
    funcDoc _ ListSize = dot <> text "size" <> parens empty
    funcDoc l (ListAccess i) = dot <> text "get" <> parens (valueDoc l i)
    funcDoc l (ListAdd v i) = dot <> text "add" <> parens (int i <> comma <+> valueDoc l v)
    funcDoc _ (IterBegin) = error "IterBegin in Java"
    funcDoc _ (IterEnd) = error "IterEnd in Java"
