-- | The parser that processes the configuration file specified as a command-line argument is defined here
module SAGA.Parsers.ConfigParser (
    -- * Configuration for Code Generation
    Configuration(..),

    -- * Language Labels Used in the Configuration File
    cSharpLabel, cppLabel, javaLabel,

    -- * Processing the Configuration File
    readConfig
    ) where

import SAGA.StoryManager.DataTypes

import Text.ParserCombinators.Parsec
import Text.ParserCombinators.Parsec.Language
import qualified Text.Parsec.Token as P


-- | Data type that defines a configuration for generating code
data Configuration = Config String

cSharpLabel = "C#"
cppLabel = "C++"
javaLabel = "Java"

--------------------------------
-- Config Language Definition --
--------------------------------

configDef :: LanguageDef st
configDef = emptyDef
            { commentStart      = "/*"
            , commentEnd        = "*/"
            , commentLine       = "//"
            , nestedComments    = True
            , identStart        = noneOf ", \t\n\r\f\v\xa0"
            , identLetter	    = noneOf ", \t\n\r\f\v\xa0"
            , opStart	        = opStart emptyDef
            , opLetter	        = opLetter emptyDef
            , reservedOpNames   = ["="]
            , reservedNames     = ["Generation", "Language"]
            , caseSensitive     = False
            }

configLexer :: P.TokenParser st
configLexer = P.makeTokenParser configDef

commaSep1   = P.commaSep1 configLexer
identifier  = P.identifier configLexer
reserved    = P.reserved configLexer
reservedOp  = P.reservedOp configLexer
whiteSpace  = P.whiteSpace configLexer

-----------------------
-- Parsing Functions --
-----------------------

parseConfig :: Parser Configuration
parseConfig = do
    whiteSpace
    lang <- parseLang
    return $ Config lang

parseLang :: Parser String
parseLang = do
    reserved "Generation"
    reserved "Language"
    reservedOp "="
    identifier

{-|
Parses the configuration file and returns a 'Configuration', or a 'ParseError' if the configuration file was not well-formed

'readConfig' takes the name of the file and the contents of the file
-}
readConfig :: String -> String -> Either ParseError Configuration
readConfig name input = case parse parseConfig name input of
    Left err        -> Left err
    Right config    -> Right config
