-- | The parser that processes the story description file specified as a command-line argument is defined here
module SAGA.Parsers.StoryParser (
    -- * Processing the Story Description File
    readStory
    ) where

import SAGA.StoryManager.DataTypes
import SAGA.StoryManager.Helper (makeLiteralNameValid)

import List (intersperse,nub)
import Text.ParserCombinators.Parsec
import Text.ParserCombinators.Parsec.Language
import qualified Text.Parsec.Token as P

----------------------
-- Data Definitions --
----------------------

data TransDescription = TransDescrip
    {preNodes :: [Label],
     postNode :: Label,
     transDescEvents :: [Label]
    }
    deriving Show

data SectDescription = SectDescrip
    {sectLabel :: Label,
     sectTrans :: [TransDescription]
    }
    deriving Show

data StoryDescription = StoryDescrip Label Label [SectDescription] [TransDescription] [Label]
    deriving Show

-------------------------------
-- Story Language Definition --
-------------------------------

storyDef :: LanguageDef st
storyDef = emptyDef
            { commentStart      = "/*"
            , commentEnd        = "*/"
            , commentLine       = "//"
            , nestedComments    = True
            , identStart        = noneOf ", \t\n\r\f\v\xa0"
            , identLetter	    = noneOf ", \t\n\r\f\v\xa0"
            , opStart	        = opStart emptyDef
            , opLetter	        = opLetter emptyDef
            , reservedOpNames   = []
            , reservedNames     = ["STORY", "INITIAL", "SECTION", "WHERE",
                                   "{", "}", "GOES", "WHEN", "OR", "AND"]
            , caseSensitive     = True
            }

storyLexer :: P.TokenParser st
storyLexer = P.makeTokenParser storyDef

commaSep    = P.commaSep storyLexer
commaSep1   = P.commaSep1 storyLexer
identifier  = P.identifier storyLexer
reserved    = P.reserved storyLexer
whiteSpace  = P.whiteSpace storyLexer

-----------------------
-- Parsing Functions --
-----------------------

-- The main parser for a Story description
parseStory :: Parser StoryDescription
parseStory = do
    whiteSpace
    name <- parseStoryName
    start <- parseInitial
    secs <- parseSectionList
    trans <- parseStoryTrans
    let input = getEventLabels secs trans
    -- parseInput
    -- parseOutput
    return $ StoryDescrip name start secs trans input

parseStoryName :: Parser Label
parseStoryName = reserved "STORY" >> parseLabel

parseInitial :: Parser Label
parseInitial = reserved "INITIAL" >> parseLabel

parseSectionList :: Parser [SectDescription]
parseSectionList = sepBy1 parseSection spaces

parseSection :: Parser SectDescription
parseSection = do
    reserved "SECTION"
    name <- parseLabel
    reserved "{"
    trans <- parseTransList
    reserved "}"
    return $ SectDescrip name trans

parseTransList :: Parser [TransDescription]
parseTransList = commaSep parseTrans

parseTrans :: Parser TransDescription
parseTrans = do
    preNodes <- parsePreLabels
    reserved "GOES"
    postNode <- parseLabel
    reserved "WHEN"
    events <- parseEventLabels
    return $ TransDescrip preNodes postNode events

parsePreLabels :: Parser [Label]
parsePreLabels = sepBy1 parseLabel (reserved "OR")

parseEventLabels :: Parser [Label]
parseEventLabels = sepBy1 parseLabel (reserved "AND")

parseStoryTrans :: Parser [TransDescription]
parseStoryTrans = reserved "WHERE" >> parseTransList

parseLabel :: Parser Label
parseLabel = do
    labs <- sepBy1 identifier spaces
    return $ concat $ intersperse " " labs

getEventLabels :: [SectDescription] -> [TransDescription] -> [Label]
getEventLabels sects trans = nub $ concatMap transDescEvents (concatMap sectTrans sects) ++ concatMap transDescEvents trans

--------------------------
-- Evaluation Functions --
--------------------------

{-|
Parses the story description file and returns a 'Story', or a 'ParseError' if the
story file was not well-formed

'readStory' takes the name of the file and the contents of the file
-}
readStory :: String -> String -> Either ParseError Story
readStory name input =
    case parse parseStory name input of
        Left err -> Left err
        Right descrip -> Right $ prepareStory . makeStory $ descrip

prepareStory :: Story -> Story
prepareStory (Story l ss h ts es) =
    let fixLabel = makeLiteralNameValid
        fixNode = createNode . fixLabel . nodeLabel
        fixNodes = map fixNode
        fixEvent = createEvent . fixLabel . eventLabel
        fixEvents = map fixEvent
        fixNodeTran t = createNodeTransition (fixLabel $ nodeTransLabel t) (fixNode $ nodeTransPreNode t) (fixNode $ nodeTransPostNode t) (fixEvents $ nodeTransEvents t)
        fixNodeTrans = map fixNodeTran
        fixSect s = createSection (fixLabel $ sectionLabel s) (fixNodes $ sectionNodes s) (fixNodeTrans $ sectionTrans s)
        fixSectTran t = createSectionTransition (fixLabel $ sectionTransLabel t) (fixSect $ sectionTransSectionPreNode t) (fixSect $ sectionTransSectionPostNode t) (fixNode $ sectionTransPreNode t) (fixNode $ sectionTransPostNode t) (fixEvents $ sectionTransEvents t)
    in createStory (fixLabel l) (map fixSect ss) (fixNode h) (map fixSectTran ts) (fixEvents es)

makeStory :: StoryDescription -> Story
makeStory (StoryDescrip name initial sects trans events) =
    let nodes = map makeStoryNode sects
        start = createNode initial
        ts = concatMap (makeStoryTrans nodes) trans
        evts = map createEvent events
    in createStory name nodes start ts evts

makeStoryNode :: SectDescription -> Section
makeStoryNode (SectDescrip name ts) =
    let nodes = nub $ map createNode (concatMap preNodes ts ++ map postNode ts)
        trans = makeTrans ts
    in createSection name nodes trans

makeTrans :: [TransDescription] -> [NodeTransition]
makeTrans = concatMap (\(TransDescrip ns postN evts) ->
    map (\n -> createNodeTransitionAutoLabel (createNode n) (createNode postN) (map createEvent evts)) ns)

makeStoryTrans :: [Section] -> TransDescription -> [SectionTransition]
makeStoryTrans storyNodes (TransDescrip ns post evts) = map (\n ->
    let preN = createNode n
        postN = createNode post
    in createSectionTransitionAutoLabel (findParent preN storyNodes) (findParent postN storyNodes) preN postN (map createEvent evts)) ns

findParent :: Node -> [Section] -> Section
findParent node [] = error $ "SAGA.findParent: " ++ nodeLabel node ++ "'s containing Section was not found"
findParent node (n:ns) =
    if (node `elem` sectionNodes n)
        then n
        else findParent node ns
