module SAGA.StoryManager.DotOutput (dotFileFromStory) where

import SAGA.StoryManager.DataTypes
import SAGA.StoryManager.Helper (makeVarNameValid)
import SAGA.StoryManager.Printing (blank,doubleQuoted,doubleQuotedText,oneTabbed,vertical,verticalNewLine)

import List (intersperse)
import System.IO
import Text.PrettyPrint.HughesPJ


labelText :: Doc
labelText = text "label ="

arrow :: Doc
arrow = text "->"

-- | Creates an ASCII text file with .dot extension. The file describes the Story in the DOT 
--   language, which when ran through a DOT layout engine (dot, neato, twopi) produces a 
--   visual graph of the Story's flow
dotFileFromStory :: Story -> IO ()
dotFileFromStory s = do
    h <- openFile (makeVarNameValid (storyLabel s) ++ ".dot") WriteMode
    hPutStrLn h (render $ dotTextFromStory s)
    hClose h

dotTextFromStory :: Story -> Doc
dotTextFromStory s = vcat [
    text "digraph" <+> doubleQuoted storyLabel s,
    lbrace,
    oneTabbed [
        labelText <+> doubleQuoted storyLabel s,
        blank,
        dotTextFromSections $ storyNodes s,
        blank,
        dotTextFromStoryTransitions $ storyTrans s],
    rbrace]

dotTextFromSection :: Section -> Doc
dotTextFromSection s = vcat [
    text "subgraph" <+> (doubleQuotedText $ "cluster " ++ sectionLabel s),
    lbrace,
    oneTabbed [
        labelText <+> doubleQuoted sectionLabel s,
        blank,
        dotTextFromSectionTransitions $ sectionTrans s],
    rbrace]

dotTextFromSections :: [Section] -> Doc
dotTextFromSections = verticalNewLine dotTextFromSection

dotTextFromStoryTransition :: SectionTransition -> Doc
dotTextFromStoryTransition t = 
    let condPrefix = concat $ intersperse " AND " $ map eventLabel $ sectionTransEvents t
        preL = doubleQuoted nodeLabel $ sectionTransPreNode t
        postL = doubleQuoted nodeLabel $ sectionTransPostNode t
        eventL = doubleQuotedText condPrefix
    in preL <+> arrow <+> postL <+> (brackets $ labelText <+> eventL)

dotTextFromStoryTransitions :: [SectionTransition] -> Doc
dotTextFromStoryTransitions = vertical dotTextFromStoryTransition

dotTextFromSectionTransition :: NodeTransition -> Doc
dotTextFromSectionTransition t = 
    let condPrefix = concat $ intersperse " AND " $ map eventLabel $ nodeTransEvents t
        preL = doubleQuoted nodeLabel $ nodeTransPreNode t
        postL = doubleQuoted nodeLabel $ nodeTransPostNode t
        eventL = doubleQuotedText condPrefix
    in preL <+> arrow <+> postL <+> (brackets $ labelText <+> eventL)

dotTextFromSectionTransitions :: [NodeTransition] -> Doc
dotTextFromSectionTransitions = vertical dotTextFromSectionTransition
