open Staged
open Basetypes

module Sign =
struct
  type t = Pos | Zero | Neg | PosOrZero | NegOrZero
  let eq = (=)
  let pos = eq Pos
  let zero = eq Zero
  let neg = eq Neg
  let pos_or_zero x = eq PosOrZero
  let neg_or_zero x = eq NegOrZero
  let bind a b c d e = function Pos -> a | Zero -> b | Neg -> c
    | PosOrZero -> d | NegOrZero -> e
end


(* **** ORIENTATION & SIDENESS FOR GEOMETRIC PREDICATES **** *)
(* @TODO move it to a seperate file *)
(* @TODO Convert into Open/Closed ball with only In/Out *)

(* +ve -> ccw;  -ve -> cw
   +ve -> below; -ve -> above *)

module Orientation =
struct
  type t = Ccw | Col | Cw | CcwOrCol | CwOrCol
  let eq = (=)
  let ccw = eq Ccw
  let col = eq Col
  let cw  = eq Cw
  let ccw_or_col = eq CcwOrCol
  let cw_or_col = eq CwOrCol
  let bind a b c d e = function Ccw -> a | Col -> b | Cw -> c
    | CcwOrCol -> d | CwOrCol -> e
  let of_sign = Sign.bind Ccw Col Cw CcwOrCol CwOrCol
  let to_sign = bind Sign.Pos Sign.Zero Sign.Neg Sign.PosOrZero Sign.NegOrZero
end

(* +ve -> inside; -ve -> outside*)

module Side =
struct
  type t = In | On | Out | InOrOn | OutOrOn
  let eq = (=)
  let in_  = eq In
  let on  = eq On
  let out = eq Out
  let in_or_on = eq InOrOn
  let in_or_out = eq OutOrOn
  let bind a b c d e = function In -> a | On -> b | Out -> c
    | InOrOn -> d | OutOrOn -> e
  let of_sign = Sign.bind In On Out InOrOn OutOrOn
  let to_sign = bind Sign.Pos Sign.Zero Sign.Neg Sign.PosOrZero Sign.NegOrZero
end

(* *************************** *)

(* The types here are very carefully built.  There is a lot of 
 * redundancy [because there are no good signature building tools, 
 * include just won't work].  The fact that the names of types are the
 * same is *on purpose*, not an accident, to get subtyping to work *)

(* We have 2 copies of everything.  Once for the base case, and another
 * for the lifted case, aka staged. Roughly speaking, the longer we
 * can stay with 'base' the better (as it is more precise), but
 * if that fails, then staged will do fine. *)

(* Base version *)
(* will be filled-in as needed *)

(* Staged version *)

module type SET =
sig
  type n
  type 'a ns = ('a, n) staged
  (* some convenient short hands *)
  type 'a bin = 'a ns -> 'a ns -> 'a ns
  type 'a rel = 'a ns -> 'a ns -> 'a Bool.b
  val to_string_b : ('a, n, string) unary
  val to_string_s : 'a ns -> ('a, string) staged
  val eq_b : ('a, n, n, bool) binary
  val eq_s : 'a rel
  val eq_tol : 'a rel
  val neq_b : ('a, n, n, bool) binary
  val neq_s : 'a rel
  (* coercion from integer *)
  val of_int : int -> n
end

module type ORDER =
sig
  type t
  type 'a t_s
  val bot : t (* always NOW *)
  val top : t
  (* same convenient short-hands *)
  type 'a bin = 'a t_s -> 'a t_s -> 'a t_s
  type 'a rel = 'a t_s -> 'a t_s -> 'a Bool.b
  val eq : 'a rel
  val neq : 'a rel
  (* 0: equal ; +ve: > ; -ve: < *)
  val compare_b : ('a, t, t, int) binary
  val compare_s : 'a t_s -> 'a t_s -> ('a, int) staged
  val lt_b : ('a, t, t, bool) binary
  val lt_s : 'a rel
  val le_b : ('a, t, t, bool) binary
  val le_s : 'a rel
  val gt_b : ('a, t, t, bool) binary
  val gt_s : 'a rel
  val ge_b : ('a, t, t, bool) binary
  val ge_s : 'a rel
  val min_b : ('a, t, t, t) binary
  val min_s : 'a bin
  val max_b : ('a, t, t, t) binary
  val max_s : 'a bin
end

module type ORDERED_SET =
sig
  type n
  type 'a ns = ('a, n) staged
  include ORDER with type t = n
                 and type 'a t_s = 'a ns
  val to_string_b : ('a, n, string) unary
  val to_string_s : 'a ns -> ('a, string) staged
  val eq_b : ('a, n, n, bool) binary
  val eq_s : 'a rel
  val eq_tol : 'a rel
  val neq_b : ('a, n, n, bool) binary
  val neq_s : 'a rel
  (* coercion from integer *)
  val of_int : int -> n
end

module type MONOID =
sig
  include SET
  val zero : n
  val plus_b : ('a, n) monoid
  val plus_s : 'a bin
end

module type NORMED_SET =
sig
  include SET
  module R : MONOID
  val norm_b : ('a, n, R.n) unary
  val norm_s : 'a ns -> 'a R.ns
 
end

module type RING =
sig
  include SET (* include here is safe *)
  val zero : n
  val one : n
  val negone : n
  val two : n
  val add_b : ('a, n) monoid
  val add_s : 'a bin
  val sub_b : ('a, n, n, n) binary
  val sub_s : 'a bin
  val neg_b : ('a, n, n) unary
  val neg_s : 'a ns -> 'a ns
  val mul_b : ('a, n) ring
  val mul_s : 'a bin
  val abs_b : ('a, n, n) unary
  val abs_s : 'a ns -> 'a ns
  val sgn : 'a ns -> (Sign.t -> 'a Bool.b)
  (* int_pow : x ^ n, where n : integer *)
  val int_pow : int -> 'a ns -> 'a ns
  val pow : 'a ns -> 'a ns -> 'a ns
end

module type FIELD =
sig
  (* also safe *)
  include RING
  val pi : n
  val inv_b : ('a, n, n) unary
  val inv_s : 'a ns -> 'a ns
  val div_b : ('a, n, n, n) binary
  val div_s : 'a bin
end

module type REALFIELD =
sig
  (* we can use only 1 include, the rest must be duplicated *)
  include FIELD
  val bot : n (* always NOW *)
  val top : n
  (* 0: equal ; +ve: > ; -ve: < *)
  val compare_s : 'a ns -> 'a ns -> ('a, int) staged
  val lt_b : ('a, n, n, bool) binary
  val lt_s : 'a rel
  val le_b : ('a, n, n, bool) binary
  val le_s : 'a rel
  val gt_b : ('a, n, n, bool) binary
  val gt_s : 'a rel
  val ge_b : ('a, n, n, bool) binary
  val ge_s : 'a rel
  val min_b : ('a, n, n, n) binary
  val min_s : 'a bin
  val max_b : ('a, n, n, n) binary
  val max_s : 'a bin

  val sqrt_s : 'a ns -> 'a ns
(*
  val pi : n
  val cos : 'a ns -> 'a ns
  val sin : 'a ns -> 'a ns
  val tan : 'a ns -> 'a ns
  val acos : 'a ns -> 'a ns
  val asin : 'a ns -> 'a ns
  val atan : 'a ns -> 'a ns *)
end
