open Algebra
open Matrix

(* Based on:
   "Staged Matrix Determinant"
   Walid Taha -- Jun 26, 2003
   http://www.metaocaml.org/examples/determinant.ml
*)

module type DETERMINANT =
sig
  module N : FIELD
  module M : MATRIX with type 'a n_s = 'a N.ns
  val eval : 'a M.m -> 'a N.ns
end

module Determinant (N : FIELD)
    (M : MATRIX with type 'a n_s = 'a N.ns) (* : DETERMINANT *)
= struct
  module N = N
  module M = M
  let make dim = Array.make_matrix dim dim N.zero
  (* >> evaluation helpers *)
  let one = N.one
  let negone = N.negone
  let add a b = N.add_s a b
  let trimul a b c = N.mul_s a (N.mul_s b c)
  (* let wrap a (x,y) = a.(x).(y) *)
  let wrap a (x,y) = M.get a x y
  let rec sum (i,j,f) =
    if i=j then f i
    else (add (f i) (sum (i+1,j,f)))
  let sub (i,j) a (x,y) = 
    a ((if x<i then x else x+1),
    (if y<j then y else y+1))
  let signexp j = Staged.of_immediate (if j mod 2 = 0 then one else negone)
  (* << evaluation helpers *)
  let eval a =
    let rec det' n a = match n with
    | 1 -> a (0,0)
    | n -> sum (0,n-1, fun j ->
      (trimul (signexp j) (a (0,j)) (det' (n-1) (sub (0,j) a))))
    in det' (M.nrows a) (wrap a)

end

module Determinant_Let (N : FIELD)
    (M : MATRIX with type 'a n_s = 'a N.ns) (* : DETERMINANT *)
= struct
  module N = N
  module M = M
  let make dim = Array.make_matrix dim dim N.zero
  (* >> evaluation helpers *)
  let one = N.one
  let negone = N.negone
  let add a b = N.add_s a b
  let trimul a b c = N.mul_s a (N.mul_s b c)
  (* let wrap a (x,y) = a.(x).(y) *)
  let wrap a (x,y) = M.get a x y
  let rec sum (i,j,f) =
    if i=j then f i
    else (add (f i) (sum (i+1,j,f)))
  let sub (i,j) a (x,y) = 
    a ((if x<i then x else x+1),
    (if y<j then y else y+1))
  let signexp j = Staged.of_immediate (if j mod 2 = 0 then one else negone)
  (* << evaluation helpers *)
  let eval a =
    let rec det' n a = match n with
    | 1 -> a (0,0)
    | n -> sum (0,n-1, fun j ->
      (trimul (signexp j) (a (0,j)) (det' (n-1) (sub (0,j) a))))
    in det' (M.nrows a) (wrap a)

end

