open Algebra
open Vector
open Point

type ('p, 'v) frame = { orig : 'p; basis : 'v list }
let create_frame o b = { orig = o; basis = b }

(* k-hyperplane in n-dimenstions has (k+1) points
   k = 0, 1, 2, 3+ for point, line, plane, hyperplane *)
module type HYPER_PLANE =
sig
  module N : REALFIELD
  module V : VECTOR with module N = N
  module P : POINT with module N = N and module V = V
  type 'a t_s
  val dim : 'a t_s -> int
  (* construction *)
  val of_points : 'a P.point_s list -> 'a t_s
  val point : 'a t_s -> int -> 'a P.point_s
  val points : 'a t_s -> 'a P.point_s list
  (* implicit equation : \sum (a_i x_i) + d *)
  val poly : 'a t_s -> 'a N.ns array
  (* vector equation : p.n + d = 0 *)
  val normal : 'a t_s -> 'a V.vector_s
  val d : 'a t_s -> 'a N.ns
  (* frame *)
  val orig : 'a t_s -> 'a P.point_s
  val basis : 'a t_s -> int -> 'a V.vector_s
  val bases : 'a t_s -> 'a V.vector_s list
  val frame : 'a t_s -> ('a P.point_s, 'a V.vector_s) frame
  (* local coordinates of a point 'on' the hyperplane *)
  val coord : 'a t_s -> 'a P.point_s -> 'a P.point_s
  val pos_vec : 'a t_s -> 'a P.point_s -> 'a V.vector_s
end

(* Line is 2D hyperplane *)
module Line (N : REALFIELD)
  (V : VECTOR with module N = N)
  (P : POINT with module N = N and module V = V)
 (* : HYPER_PLANE *) =
struct
  module N = N
  module V = V
  module P = P
  type 'a t_s = 'a P.point_s * 'a P.point_s
  let dim _ = 1
  let of_points = fun l -> (List.nth l 0), (List.nth l 1)
  let points = fun (a, b) -> [a; b]
  let point l i = List.nth (points l) i
  (* Frame *)
  let orig (p0, p1) = p0
  (* one basis vector *)
  (* r0, r1: position vector of p0, p1 *)
  (* d = normalized  (p1 - p0) *)
  let basis (p0,p1) i =
    V.direction (V.sub (P.pos_vec p1) (P.pos_vec p0))
  let bases h = [basis h 0]
  let frame h = create_frame (orig h) (bases h)
  (* normal vector (normalized) *)
  let normal h =
    Code.let_ (basis h 0) (fun b ->
     let x,y = (V.coord b 0), (V.coord b 1) in
     Code.let_ (V.of_coords [(N.neg_s y); x]) V.direction)
  (* d = -n.p0 *)
  let d h = 
    let p0 = orig h and n = normal h in
    Code.let_ n (fun n -> N.neg_s (V.dot (P.pos_vec p0) n))
  (* polynomial ax + by + c = 0
     (a, b) = n, c = -p0.n *)
  let v_to_list v = Array.to_list (Array.init V.dim (V.coord v))
  let poly h = Array.of_list ((v_to_list (normal h)) @ [(d h)])
  (* Local coord of p in the line *)
  let coord h p =
    (* find t = d . (p - orig) *)
    let d = basis h 0
    and v = V.sub (P.pos_vec p) (P.pos_vec (orig h)) in
    let t = (V.dot d v) in
    (* 1D point relative the frame *)
    P.of_list [t]
  (* local position vector of p *)
  let pos_vec h p = P.pos_vec (coord h p)
end

(* n-D plane *)
module N_plane
  (N : REALFIELD)
  (V : VECTOR with module N = N)
  (P : POINT with module N = N and module V = V)
  (* : HYPER_PLANE *) =
struct
  module N = N
  module V = V
  module P = P  
  type 'a t_s = 'a P.point_s list
  (* k-hyperplane has k+1 points *)
  let dim h = (List.length h)-1
  let of_points l = l
  let point h i = List.nth h i
  let points h = h

  (* frame *)
  let orig h = point h 0
  (* normalized bases
     p1-p0, p2-p0, p3-p0, ... *)
  let basis h i =
    if (dim h) = 0 then V.zero 1 else
    let p = point h (i+1) and o = orig h in
    V.direction (P.sub p o)
  let bases h =
    if (dim h) = 0 then [basis h 0] else
    Array.to_list (Array.init (dim h) (basis h))
  let frame h = create_frame (orig h) (bases h)

  (* normal vector (normalized) *)
  (* @TODO this is only valid for 1,2,3D;
     since gcross is not implemented yet *)
  let normal h =
    if (dim h) = 0 then V.of_coords [Staged.Now N.one]
    else if (dim h) = 1 then
      Code.let_ (basis h 0) (fun b ->
	V.of_coords [N.neg_s (V.coord b 1); (V.coord b 0)])
    else V.direction (V.bcross (basis h 0) (basis h 1))
  (* d = -n.p0 *)
  let d h = 
    let p0 = orig h and n = normal h in
    N.neg_s (V.dot (P.pos_vec p0) n)
  (* polynomial a0x0 + a1x1 + ... + d = 0
     (a0, a1, ...) = n, d = -p0.n *)
  let v_to_list v = Array.to_list (Array.init V.dim (V.coord v))
  let poly h = Array.of_list ((v_to_list (normal h)) @ [d h])
  (* baricentric coord of p *)
  (* @TODO this is a 3D only version *)
  let coord h p =
    let u = basis h 0
    and v = basis h 1
    and w = P.sub p (orig h) in
    (* denom = (u.v)^2 - (u.u)(v.v) = (u.v)^2 - 1
       because all basis normalized? *)
    let den = N.sub_s (N.int_pow 2 (V.dot u v))
	              (N.mul_s (V.dot u u) (V.dot v v))
    in
    let s = N.sub_s (N.mul_s (V.dot u v) (V.dot w v))
	            (N.mul_s (V.dot v v) (V.dot w u))
    and t = N.sub_s (N.mul_s (V.dot u v) (V.dot w u))
	            (N.mul_s (V.dot u u) (V.dot w v))
    in
    let s = N.div_s s den
    and t = N.div_s t den in
    let b0 = N.sub_s (N.sub_s (Staged.of_immediate N.one) s) t
    and b1 = s and b2 = t in
    P.of_list [b0; b1; b2]
  (* local position vector of p *)
  let pos_vec h p = P.pos_vec (coord h p)
end

module Hplane_Operations (H : HYPER_PLANE) =
struct
  (* dist (h,p) =  n.p + d = n.p - n.p0 = n (p-p0) *)
  let dist h p =
    H.V.dot (H.normal h) (H.P.sub p (H.orig h))
(*    Code.let_ (H.normal h) (fun n ->
      H.N.sub_s (H.V.dot n (H.P.pos_vec p))
                (H.V.dot n (H.P.pos_vec (H.orig h)))) *)
  (* p' = p - d(p,h).n *)
  (* @TODO The expression p-p0 is computed inside 'normal' but this
	   caller has no clue. Hence the code repetition is inevitable
           in this case. *)
  let project h p =
    Code.let_ (H.normal h) (fun n ->
      let p0 = H.orig h in
      let v = H.P.sub p p0 in
      let t = H.N.neg_s (H.V.dot n v) in
      let dis = H.V.scale n t in
      H.P.add p dis)
end

