open Staged
open Algebra
open Vector
open Point
open Sphere
open Matrix
open Determinant

(*
  insphere p_0 p_1 .. p_n-1 p:
    1. let h = hplane L(p_0) L(p_1) .. L(p_n-1)
               where L = parabolic lifting
    2. make sure oriented p_0 p_1 ... p_n-1 is pos
       otherwise swap p_0 and p_1
    3. get s = orient h p
    4. interpret s as:
        +ve -> in, -ve -> out, 0 -> on
       [basically +ve = below the lifted hplane = inside
                  -ve = above ... = outside ]

  usage: arrange a b c d such that: abc are cw and d is above, or
         abc are ccw and d is below.
*)

module Insphere (S : HSPHERE) =
struct
  module M = Matrix (S.N)
  module Det = Determinant (S.N) (M)
  (* (c0, c1, c2, .. , cn) ->
     (c0, .. , cn, c0^2 + .. + cn^2) *)
  let parab p = 
    let v = (S.P.pos_vec p) in
    S.P.V.dot v v

  let matrix ps =
    let p_row p = Array.of_list (
      (S.P.to_list p) @ [parab p] @ [Now S.N.one]) in
    let a = Array.of_list ps in
    let arr = Array.map p_row a in
    M.of_array arr  

  let inside_val s p =
    let ps = (S.points s) @ [p] in
    let m = matrix ps in
    Det.eval m

  let inside s p =
    let d = (inside_val s p) in
    S.N.sgn d

  (* point is inside, if it's below to hyperplan *)
  let in_ s p = (inside s p) Sign.Pos
  let on s p = (inside s p) Sign.Zero
  let out s p = (inside s p) Sign.Neg
  let in_or_on s p = (inside s p) Sign.PosOrZero
  let out_or_on s p = (inside s p) Sign.NegOrZero
end

