(* This is a duplicate of stuff in float.ml, comment it all out
open Staged
open Algebra

module Float_Set =
struct
  (* SET *)
  type n = float
  type 'a n_s = ('a, n) staged
  let eq x y = mk_binary { bnow = (fun x y -> x=y);
                           blater = (fun x y -> .<.~x = .~y>.);
                           zero = None; one = None } x y
  let neq x y = mk_binary { bnow = (fun x y -> x <> y);
                            blater = (fun x y -> .<.~x <> .~y>.);
                            zero = None; one = None } x y
  let to_string x = mk_unary { unow = (fun x -> string_of_float x);
                             ulater = (fun x -> .< string_of_float .~x>.) } x
end

module Float_Order (* : ORDER *) =
struct
  include Float_Set
  type t = n
  type 'a t_s = 'a n_s
  let bot = mk_const neg_infinity
  let top = mk_const infinity
  let compare x y =
    mk_binary { bnow = (fun x y -> if x > y then 1 else if x = y then 0 else -1);
                blater = (fun x y -> .< let x = .~x and y = .~y in if x > y then 1 else if x = y then 0 else -1 >.);
                zero = None; one = None } x y
  let lt x y = mk_binary { bnow = (fun x y -> x < y); blater = (fun x y -> .<.~x < .~y>.); zero = None; one = None } x y
  let le x y = mk_binary { bnow = (fun x y -> x <= y); blater = (fun x y -> .<.~x <= .~y>.); zero = None; one = None } x y
  let gt x y = mk_binary { bnow = (fun x y -> x > y); blater = (fun x y -> .<.~x > .~y>.); zero = None; one = None } x y
  let ge x y = mk_binary { bnow = (fun x y -> x >= y); blater = (fun x y -> .<.~x >= .~y>.); zero = None; one = None } x y
  let min x y = mk_binary { bnow = (fun x y -> min x y);
    blater = (fun x y -> .<min .~x .~y>.); zero = None; one = None } x y
  let max x y = mk_binary { bnow = (fun x y -> max x y);
    blater = (fun x y -> .<max .~x .~y>.); zero = None; one = None } x y
end

(* Implementation of REALFIELD *)
module Float =
struct
  (* ORDERED_FIELD *)
  include Float_Order
  (* open Float_Set -- not needed *)
  (* RING *)
  let zero = mk_const 0.
  let one = mk_const 1.
  let add x y =
    let add_staged = { bnow = (fun x y -> x+.y); blater = (fun x y -> .<.~x +. .~y>.);
                       zero = None; one = Some 0. } in
    mk_monoid add_staged x y
  let neg x =
    let neg_staged = { unow = (fun x -> -. x); ulater = (fun x -> .< -. .~x >.) } in
    mk_unary neg_staged x
  let sub x y = add x (neg y)
  let mul x y =
    let mul_staged = { bnow = (fun x y -> x*.y); blater = (fun x y -> .<.~x *. .~y>.);
                       zero = Some 0.; one = Some 1. } in
    mk_ring mul_staged x y
  let abs x = mk_unary { unow = (fun x -> abs_float x);
                       ulater = (fun x -> .< abs_float .~x >.) } x
  let sgn x =
    let sgn_now x = if x = 0. then Zero
                    else if x > 0. then Pos
                    else Neg
    and sgn_later x = .< if .~x = 0. then Zero
                       else if .~x > 0. then Pos
                       else Neg >.
    in
    mk_unary { unow = sgn_now; ulater = sgn_later } x
  (* FIELD *)
  let inv x =
    mk_unary { unow = (fun x -> 1. /. x);
             ulater = (fun x -> .< 1. /. .~x >.) } x
  let div x y = mul x (inv y)
  (* REALFIELD *)
  let sqrt x =
    mk_unary { unow = (fun x -> sqrt x);
               ulater = (fun x -> .< sqrt .~x >.) } x
  let pi = mk_const ~-.1.
  let cos x = 
    mk_unary { unow = (fun x -> cos x);
               ulater = (fun x -> .< cos .~x >.) } x
  let sin x = 
    mk_unary { unow = (fun x -> sin x);
               ulater = (fun x -> .< sin .~x >.) } x
  let tan x = 
    mk_unary { unow = (fun x -> tan x);
               ulater = (fun x -> .< tan .~x >.) } x
  let acos x = 
    mk_unary { unow = (fun x -> acos x);
               ulater = (fun x -> .< acos .~x >.) } x
  let asin x = 
    mk_unary { unow = (fun x -> asin x);
               ulater = (fun x -> .< asin .~x >.) } x
  let atan x = 
    mk_unary { unow = (fun x -> atan x);
               ulater = (fun x -> .< atan .~x >.) } x
end

(* ***************************************** *)

(* Implementation of OFIELD *)
module Rational =
struct
  (* SET *)
  type nn = int*int
  type 'a n = ('a, int*int) staged (* not normalized *)
  let eq_now (a,b) (c,d) = a*d=b*c
  let eq_later x y = .< let a,b = .~x and c,d = .~y in a*d=b*c >.
  let eq x y = mk_binary { bnow = eq_now; blater = eq_later; zero = None; one = None } x y
  let neq x y = mk_binary { bnow=(fun x y -> x <> y);
                        blater=(fun x y -> .<.~x <> .~y>.); zero = None; one = None } x y
  (* RING *)
  let zero = mk_const (0,1)
  let one = mk_const (1,1)
  let add x y =
    let add_staged = { bnow = (fun (a,b) (c,d) -> (a*d+b*c,b*d));
                       blater = (fun x y -> .< let a,b = .~x and c,d = .~y in (a*d+b*c,b*d)>.);
                       zero = None; one = Some (0,1) } in
    mk_monoid add_staged x y
  let neg x =
    let neg_staged = { unow = (fun (a,b) -> (-a, b)); ulater = (fun x -> .< let (a,b) = .~x in (-a,b) >.) } in
    mk_unary neg_staged x
  let sub x y = add x (neg y)
  let mul x y =
    let mul_staged = { bnow = (fun (a,b) (c,d) -> (a*c, b*d));
                       blater = (fun x y -> .< let a,b = .~x and c,d = .~y in (a*c, b*d) >.);
                       zero = Some (0,1); one = Some (1,1) } in
    mk_ring mul_staged x y

  let abs x = mk_unary { unow = (fun (a,b) -> (abs a, abs b));
                         ulater = (fun x -> .< let a,b = .~x in (abs a, abs b) >.) } x
  let sgn x =
    let sgn_now (a,b) = if a = 0 then Zero
                    else if (a < 0 or b < 0) then Neg
                    else Pos
    and sgn_later x = .< let a,b = .~x in
                       if a = 0 then Zero
                       else if (a < 0 or b < 0) then Neg
                       else Pos >.
    in
    mk_unary { unow = sgn_now; ulater = sgn_later } x
  (* FIELD *)
  let inv x =
    mk_unary { unow = (fun (a,b) -> (b,a));
               ulater = (fun x -> .< let a,b = .~x in (b,a) >.) } x
  let div x y = mul x (inv y)
  (* ORDERED_FIELD *)
  let lt x y = mk_binary { bnow = (fun x y -> x < y); blater = (fun x y -> .<.~x < .~y>.);
                           zero = None; one = None } x y
  let le x y = mk_binary { bnow = (fun x y -> x <= y); blater = (fun x y -> .<.~x <= .~y>.);
                           zero = None; one = None } x y
  let gt x y = mk_binary { bnow = (fun x y -> x > y); blater = (fun x y -> .<.~x > .~y>.);
                           zero = None; one = None } x y
  let ge x y = mk_binary { bnow = (fun x y -> x >= y); blater = (fun x y -> .<.~x >= .~y>.);
                           zero = None; one = None } x y

  let lt_now (a,b) (c,d) = a*d < b*c
  let lt_later x y = .< let a,b = .~x and c,d = .~y in a*d < b*c >.
  let le_now (a,b) (c,d) = a*d <= b*c
  let le_later x y = .< let a,b = .~x and c,d = .~y in a*d <= b*c >.
  let gt_now (a,b) (c,d) = a*d > b*c
  let gt_later x y = .< let a,b = .~x and c,d = .~y in a*d > b*c >.
  let ge_now (a,b) (c,d) = a*d >= b*c
  let ge_later x y = .< let a,b = .~x and c,d = .~y in a*d >= b*c >.

  let lt x y = mk_binary { bnow = lt_now; blater = lt_later; zero = None; one = None } x y
  let le x y = mk_binary { bnow = le_now; blater = le_later; zero = None; one = None } x y
  let gt x y = mk_binary { bnow = gt_now; blater = gt_later; zero = None; one = None } x y
  let ge x y = mk_binary { bnow = ge_now; blater = ge_later; zero = None; one = None } x y
end
*)
