open Staged
open Algebra
open Matrix
open Determinant
open Vector
open Point
open Hplane

(* ***** Point orientation w.r.t. hyperplane ***** *)

(*
  Adaptive Precision Floating-Point Arithmetic and Fast Robust
  Geometric Predicates, J. R. Shewchuck, Discrete & Computational
  Geometry 18(3):305-363, October 1997.
  http://citeseerx.ist.psu.edu/viewdoc/summary?doi=10.1.1.71.8830
  
 @ARTICLE{Shewchuk97adaptiveprecision,
    author = {Jonathan Richard Shewchuk},
    title = {Adaptive precision floating-point arithmetic and fast robust geometric predicates},
    journal = {Discrete & Computational Geometry},
    year = {1997},
    volume = {18},
    pages = {305--363}}
*)

(* +ve -> left;  -ve -> right (1D)
   +ve -> ccw;  -ve -> cw (2D)
   +ve -> below; -ve -> above (3D) *)

module Orient (H : HYPER_PLANE) =
struct
  module N = H.N
  module V = H.V
  module P = H.P
  module M = Matrix (N)
  module Det = Determinant (N) (M)

  let matrix ps =
    let p_row p = Array.of_list ((P.to_list p) @ [Now N.one]) in
    let a = Array.of_list ps in
    let arr = Array.map p_row a in
    M.of_array arr

  let orient_val h p =
    let ps = (H.points h) @ [p] in
    let m = matrix ps in
    Det.eval m

  let orient h p =
    let ps = (H.points h) @ [p] in
    let m = matrix ps in
    let d = Det.eval m in
    (N.sgn d)

  (* collinear, coplanar *)
  let col h p = (orient h p) Sign.Zero
  (* on positive side *)
  let pos h p = (orient h p) Sign.Pos
  (* on negative side *)
  let neg h p = (orient h p) Sign.Neg
  let pos_or_col h p = (orient h p) Sign.PosOrZero
  let neg_or_col h p = (orient h p) Sign.NegOrZero
  (* 2D +ve = ccw;  -ve = cw *)
  let ccw = pos
  let cw = neg
  let ccw_or_col = pos_or_col
  let cw_or_col h p = neg_or_col
  (* +ve = below; -ve = above *)
  let below = pos
  let above = neg
  let on = col
  let below_or_on = pos_or_col
  let above_or_on = neg_or_col

  (* matrix  [.. x .. 1] *)
  let triple_product n a = 
    let arr =
      let a = a in
      Array.init n (fun i -> Array.init n (
                     fun j -> Staged.of_atom .<(.~a).(i).(j)>.)) in
    (for i = 0 to n-1 do
       arr.(i).(n-1) <- Now N.one
     done); arr

  (* templates for expansion of the triple_product determinant *)
  (* int -> ('a, N.n array array -> N.n) code *)
  let gen_expansion n = .< fun a -> .~(
    let arr = triple_product n (.<a>.) in
    let m = M.of_array arr in
    let d = Det.eval m in
    Staged.to_code d) >.

  let bt_orient_val h p =
    let n = P.dim+1 in
    let ps = (H.points h) @ [p] in
    let p_row p = Array.of_list ((P.to_list p) @ [Now N.one]) in
    let a = Array.of_list ps in
    let arr = Array.map p_row a in
    let arr1 = Array.map (fun a ->
      Array.map (fun b -> Staged.to_code b) a) arr in
    let arr2 = Array.map Util.codearray_to_arraycode arr1 in
    let arr3 = Util.codearray_to_arraycode arr2 in
    of_comp .< let arr = .~(arr3) in
             let det = .~(gen_expansion n) in
             det arr >.
end


