open Staged
open Basetypes
open Algebra
open Vector

module type POINT =
sig
  module N : REALFIELD
  module V : VECTOR with module N = N
  type point
  type 'a point_s = ('a, point) staged
  val dim : int
  val of_immediate : point -> 'a point_s
  val to_expr : 'a point_s -> ('a, point) code_expr
  val to_code : 'a point_s -> ('a, point) code
  val of_code : int -> ('a, point) code -> 'a point_s
  (* of_list, to_list are utilties for generation time.
   * Coercion from/to lists is extremely useful at
   * generation time for tuple-like structures. It's not an optimal
   * choice for manipulating components of a point (because coercion
   * forgets about domain & design -specific information such as number
   * of dimensions). In addition, coercion breaks the encapsulation, thus
   * risking the model integrity. However, it is still a handy tool if
   * carefully used inside other modules such as simplex etc. *)
  val of_list : 'a N.ns list -> 'a point_s
  val to_list : 'a point_s -> 'a N.ns list
  val of_array : 'a N.ns array -> 'a point_s
  val to_array : 'a point_s -> 'a N.ns array
  val eq : 'a point_s -> 'a point_s -> ('a, bool) staged
  val eq_tol : 'a point_s -> 'a point_s -> ('a, bool) staged
  val neq : 'a point_s -> 'a point_s -> ('a, bool) staged
  val orig : int -> point
  val coord : 'a point_s -> int -> 'a N.ns
  (* position vector of point *)
  val pos_vec : 'a point_s -> 'a V.vector_s
  val add : 'a point_s -> 'a V.vector_s -> 'a point_s
  val sub : 'a point_s -> 'a point_s -> 'a V.vector_s
  (* L_f(p) =  (p, f(p)) :: R^d -> R^d+1 *)
  (* val lift : ('a point_s -> 'a N.ns) -> 'a point_s -> 'a point_s *)
  val to_string : 'a point_s -> ('a, string) staged
  (* val to_string_b : ('a, point, string) unary *)
end

(* Point with ordering *)
module type ORDERED_POINT =
sig
  include POINT
  val bot : int -> point
  val top : int -> point
  val lt : 'a point_s -> 'a point_s -> 'a Bool.b
  val le : 'a point_s -> 'a point_s -> 'a Bool.b
  val gt : 'a point_s -> 'a point_s -> 'a Bool.b
  val ge : 'a point_s -> 'a point_s -> 'a Bool.b
  val min : 'a point_s -> 'a point_s -> 'a point_s  
  val max : 'a point_s -> 'a point_s -> 'a point_s  
end

(* Special version of point ordering: axis-oriented ordering.
* The extra 'int' parameter referes to the coordinate in E^n *)
module type ISO_AXIS_ORDERED_POINT =
sig
  include POINT
  val bot : int -> point
  val top : int -> point
  val lt : int -> 'a point_s -> 'a point_s -> 'a Bool.b
  val le : int -> 'a point_s -> 'a point_s -> 'a Bool.b
  val gt : int -> 'a point_s -> 'a point_s -> 'a Bool.b
  val ge : int -> 'a point_s -> 'a point_s -> 'a Bool.b
  val min : int -> 'a point_s -> 'a point_s -> 'a point_s  
  val max : int -> 'a point_s -> 'a point_s -> 'a point_s  
end

