open Staged
open Algebra
open Basetypes
open Vector
open Point
open Tuple

(* Point in E^n *)
module En_Point (N : REALFIELD)
  (V : VECTOR with module N = N)
  (T : TUPLE) (* : POINT *)
= struct
  module N = N
  module V = V
  type point = N.n T.t
  type 'a point_s = ('a, point) staged
  let to_expr = Staged.to_expr
  (* @TODO fix the order of paramters in Tuple signature *)
  let of_expr p = Staged.of_expr p
  let of_code i c = Staged.of_atom c
  let to_code = Staged.to_code
  let of_immediate = Staged.of_immediate
  let orig n = (T.init n (fun _ -> N.zero))
  let dim = T.dim
  let coord p i =
    let unow t = T.proj_n t i
    and ulater t =
      let x = (T.proj_c t i) in
      if t.a then lift_atom x.c else lift_comp x.c in
    mk_unary { unow = unow; ulater = ulater } p
  (* coercions *)
  let of_list l =
    if (List.exists is_later l)
    then Later (T.of_list_c (List.map Staged.to_expr l))
    else Now (T.of_list_n (List.map Staged.to_immediate l))
  let to_array p = Array.init dim (fun i -> coord p i)
  let to_list p = Array.to_list (to_array p)
  let of_array p = of_list (Array.to_list p)
  (* map & fold *)
  let map f v =
    let unow t = T.map_n f t
    and ulater t = (T.map_c f t) in
    mk_unary { unow = unow; ulater = ulater } v
  let mapi f v =
    let unow t = T.mapi_n f t
    and ulater t = (T.mapi_c f t) in
    mk_unary { unow = unow; ulater = ulater } v
  let map2 f v v' =
    let bnow v v' = T.map2_n f v v'
    and blater v v' = (T.map2_c f v' v) in
    mk_binary { bnow = bnow; blater = blater } v v'
  let fold f z v =
    let unow v = T.fold_n f z v
    and ulater v = (T.fold_c f z v) in
    mk_unary { unow = unow; ulater = ulater } v
  let mapfold m f z v =
    let unow v = T.mapfold_n m f z v
    and ulater v = (T.mapfold_c m f z v) in
    mk_unary { unow = unow; ulater = ulater } v
  let map2fold m f z v v' =
    let bnow v v' = T.map2fold_n m f z v v'
    and blater v v' = (T.map2fold_c m f z v v') in
    mk_binary { bnow = bnow; blater = blater } v v'
  (* equality *)
  let eq a b = map2fold N.eq_s Bool.and_s true a b
  let neq a b = Bool.not_s (eq a b)
  let eq_tol a b = map2fold N.eq_tol Bool.and_s true a b
  let neq_tol a b = Bool.not_s (eq_tol a b)
  (* operations *)
  let pos_vec p = V.of_coords (to_list p)
  let sub p0 p1 = V.sub (pos_vec p0) (pos_vec p1)
  let add p v =
    Code.let_ v (fun v ->
      mapi (fun i x -> N.add_s (V.coord v i) x) p)
  let to_string p =
    List.fold_left (fun x y -> String.concat_s x (String.concat_s (Now ", ") y)) (Now ", ")
      (List.map N.to_string_s (to_list p))
  (* let _to_string_n p =
    let l = T.to_list_n p
  let to_string_b p =
    { unow = (fun p -> T.fold_n String.concat_b.bop.bnow ", "
	(T.map_n N.to_string_b.unow p));
      ulater = (fun p -> .< T.fold_n String.concat_b.bop.bnow ", "
	  (T.map_n N.to_string_b.unow .~p) >.) } *)
end

(* There are several kinds of point orders *)

(* Ordered Point along certain axis *)
module Iso_Axis_Ordered_En_Point (N : REALFIELD)
  (V : VECTOR with module N = N)
  (T : TUPLE) (* : ISO_AXIS_ORDERED_POINT *)
= struct
  (* @TODO this include might cause troubles with OCaml 3.11? *)
  include En_Point(N)(V)(T)

  let bot n = T.init n (fun _ -> N.bot)
  let top n = T.init n (fun _ -> N.top)

  (* apply is a utility that can generate better code
   * apply i p0 p1 f == f (proj i p0 p1) *)
  let apply2 i p0 p1 f = f (coord p0 i) (coord p1 i)
  let lt i p0 p1 = apply2 i p0 p1 N.le_s
  let le i p0 p1 = apply2 i p0 p1 N.le_s
  let gt i p0 p1 = apply2 i p0 p1 N.gt_s
  let ge i p0 p1 = apply2 i p0 p1 N.ge_s
  (* min 0 p0@(x0,y0) p1@(x1,y1) == if x0 <= x1 then p0 else p1 *)
  (* 
  let min i p0 p1 = T.map2 (fun x y -> Code.ife_ (N.le_s x y) x y)
  Code.ife_ (N.le_s i p0 p1) 
  let max i p0 p1 = T.map2 (fun x y -> Code.ife_lode.ife_  (N.ge_s x y) x y)
  *)
  (* let min i p0 p1 = failwith "Iso_Axis_Ordered_En_Point.min" *)
  (* let max i p0 p1 = failwith "Iso_Axis_Ordered_En_Point.max" *)
  let min i p0 p1 =
    Code.let_ p0 (fun p0 -> Code.let_ p1 (fun p1 ->
      Code.ife (le i p0 p1) p0 p1 Staged.to_code))
  let max i p0 p1 =
    Code.let_ p0 (fun p0 -> Code.let_ p1 (fun p1 ->
      Code.ife (ge i p0 p1) p0 p1 Staged.to_code))
end

(*
  perpendicular bisector of two points
  2 points -> linear equation of the bisector
  bisect a b =
    x = a-b
    y = a+b
    s = -(x.y)/2
    tuple of [x s]
  val bisect :: point -> point -> tuple

let bisect a b =
  let va = P.pos_vec a
  and vb = P.pos_vec b in
  V.sub
  let a = position vector of a
  and b = position vector of b in
  let amb = (V.sub a b)
  and apb = (V.add a b) in
  let s = N.div (V.dot amb apb) N.two in
*)
