open Staged
open Basetypes
open Algebra
open Matrix
open Determinant
open Point

(* n-sphere
   0-, 1-, 2-, 3-, n- spheres are:
   point, line, cicle, sphere, hypersphere *)
module type HSPHERE =
sig
  module N : REALFIELD
  module P : POINT with module N = N
  type 'a sphere_s
  val dim : 'a sphere_s -> int
  val of_points : 'a P.point_s list -> 'a sphere_s
  val of_centre_radius : 'a P.point_s -> 'a N.ns -> 'a sphere_s
  val point : 'a sphere_s -> int -> 'a P.point_s
  val points : 'a sphere_s -> 'a P.point_s list
end

(* Sphere is a generation time concept *)
module Sphere (N : REALFIELD)
    (P : POINT with module N = N) (* : HSPHERE *) =
struct
  module N = N
  module P = P
  type 'a sphere_s = 'a P.point_s array
  let dim s = (Array.length s) - 1
  let of_points = fun l -> Array.of_list l
  let of_centre_radius c r = failwith "of_centre_radius not implemented"
  let point s i = s.(i)
  let points s = Array.to_list s
end

module Sphere_Operations (N : REALFIELD) (H : HSPHERE with module N = N) :
sig
  val centre : 'a H.sphere_s -> 'a H.P.point_s
  val radius : 'a H.sphere_s -> 'a H.N.ns
  val radius2 : 'a H.sphere_s -> 'a H.N.ns
  (* nD content (volume):
     1, length, area, volume, hypervolume for
     point, line, circle, sphere, hypersphere respectively *)
  val content : 'a H.sphere_s -> 'a H.N.ns
  (* (n-1)D content (surface area):
     0, 2, circumference, surface area, hyper-
     surface area for point, line, circle, sphere, hypersphere
     respectively *)
  val surface : 'a H.sphere_s -> 'a H.N.ns

  val c_n : int -> 'a H.N.ns
  val v_n : int -> 'a H.N.ns -> 'a H.N.ns
  val s_n : int -> 'a H.N.ns -> 'a H.N.ns
end
=
struct
  module M = Matrix (H.N)
  module D = Determinant (H.N) (M)

  let centre_rad_2 s =
    let p1 = H.point s 0 and p2 = H.point s 1
    and p3 = H.point s 2 in
    let x1, y1 = H.P.coord p1 0, H.P.coord p1 1
    and x2, y2 = H.P.coord p2 0, H.P.coord p2 1
    and x3, y3 = H.P.coord p3 0, H.P.coord p3 1 in
    let sqr x = H.N.int_pow 2 x in
    let sumsqr x y = H.N.add_s (sqr x) (sqr y) in
    let row_a x y = [| (sumsqr x y); y; Now H.N.one |]
    and row_b x y =  [| x; (sumsqr x y); Now H.N.one |]
    and row_d x y =  [| x; y; Now H.N.one |] in
    let a = [| (row_a x1 y1); (row_a x2 y2); (row_a x3 y3) |]
    and b = [| (row_b x1 y1); (row_b x2 y2); (row_b x3 y3) |]
    and d = [| (row_d x1 y1); (row_d x2 y2); (row_d x3 y3) |] in
    let det_a = (D.eval (M.of_array a))
    and det_b = (D.eval (M.of_array b))
    and det_d = (D.eval (M.of_array d)) in
    let det_d' = (H.N.mul_s (Now H.N.two) det_d) in
    let xc,yc = (H.N.div_s det_a det_d'), (H.N.div_s det_b det_d') in
    let radius2 = H.N.add_s (sqr (H.N.sub_s x1 xc))
	                    (sqr (H.N.sub_s y1 yc)) in
    H.P.of_list [xc; yc], radius2

  let centre_rad_3 s =
    let sqr x = H.N.int_pow 2 x in
    let p1 = H.point s 0 and p2 = H.point s 1
    and p3 = H.point s 2 and p4 = H.point s 3 in
    let x1, y1, z1 = H.P.coord p1 0, H.P.coord p1 1, H.P.coord p1 2
    and x2, y2, z2 = H.P.coord p2 0, H.P.coord p2 1, H.P.coord p2 2
    and x3, y3, z3 = H.P.coord p3 0, H.P.coord p3 1, H.P.coord p3 2
    and x4, y4, z4 = H.P.coord p4 0, H.P.coord p4 1, H.P.coord p4 2 in
    let sum x y z = H.N.add_s (H.N.add_s (sqr x) (sqr y)) (sqr z) in
    let row x y z = [| (sum x y z); x; y; z; Now H.N.one |] in
    let a = [| [| Now H.N.one; Now H.N.one;
		  Now H.N.one; Now H.N.one; Now H.N.one |]; (* dummy row *)
               row x1 y1 z1; row x2 y2 z2; 
               row x3 y3 z3; row x4 y4 z4 |] in
    let m = M.of_array a in
    let m11 = D.eval (M.minor m 0 0) and m12 = D.eval (M.minor m 0 1)
    and m13 = D.eval (M.minor m 0 2) and m14 = D.eval (M.minor m 0 3)
    and m15 = D.eval (M.minor m 0 4) in
    let dx = m12 and dy = H.N.neg_s m13 and dz = m14
    and a2 = H.N.mul_s (Now H.N.two) m11 in
    let xc = H.N.div_s dx a2 and yc = H.N.div_s dy a2
    and zc = H.N.div_s dz a2 in

    (* r2 = [dx^2 + dy^2 + dz^2 - 4ac]/4a^2 = (t0-t1)/t2 *)
    let t0 = H.N.add_s (H.N.add_s (sqr dx) (sqr dy)) (sqr dz)
    and t1 = H.N.mul_s (H.N.mul_s (Now (H.N.of_int 4)) m15) m11
    and t2 = H.N.mul_s (Now (H.N.of_int 4)) (sqr m11) in
    let r2 = H.N.div_s (H.N.sub_s t0 t1) t2
    in
    H.P.of_list [xc; yc; zc], r2

  let centre s =
    if (H.dim s = 2) then fst (centre_rad_2 s)
                     else fst (centre_rad_3 s)
  let radius2 s =
    if (H.dim s = 2) then snd (centre_rad_2 s)
                     else snd (centre_rad_3 s)

  let radius s = H.N.sqrt_s (radius2 s)

  (* util *)
  let even n = (n mod 2) = 0
  let rec pow2 = function 0 -> 1 | n -> 2 * pow2 (n-1)
  let rec fact = function 0 -> 1 | n -> n * fact (n - 1)
  let rec dfact = function 1 -> 1 | n -> n * dfact (n - 2)
  let pi = Now H.N.pi
  (* returns (a, b, c) where C_n = (2^a * pi^b / c) *) 
  let c_n n =
    let (a, b, c) =
      if (even n) then let nh = n/2 in (0, nh, fact (nh))
      else ((n+1)/2, (n-1)/2, dfact (n)) in
    let a = H.N.of_int (pow2 a)
    and b = H.N.of_int b
    and c = H.N.of_int c in
    let a = Now a and b = Now b and c = Now c in
    H.N.div_s (H.N.mul_s a (H.N.pow pi b)) c
  (* V_n = C_n r^n *)
  let v_n n r = H.N.mul_s (c_n n) (H.N.int_pow n r)
  (* S_n = n C_n r^n-1 *)
  let s_n n r =
    H.N.mul_s (Now (H.N.of_int n)) (H.N.mul_s (c_n n) (H.N.int_pow (n-1) r))
  let content c = v_n (H.dim c) (radius c)
  let surface c = s_n (H.dim c) (radius c)
end


(* http://mathforum.org/library/drmath/view/55239.html

circle with radius r, centre (a, b)
(x-a)^2 + (x-b)^2 = r^2

circle passing through (x1, y1) (x2, y2) (x3, y3)

    [ x1^2+y1^2 y1 1 ]
A = [ x2^2+y2^2 y2 1 ]
    [ x3^2+y3^2 y3 1 ]

    [ x1 x1^2+y1^2 1 ]
B = [ x2 x2^2+y2^2 1 ]
    [ x3 x3^2+y3^2 1 ]

    [ x1  y1  1 ]
D = [ x2  y2  1 ]
    [ x3  y3  1 ]

center   = (det(A)/2*det(D), det(B)/2*det(D))
radius^2 = (x1-xc)^2 + (y1-yc)^2
*)


(*
http://local.wasp.uwa.edu.au/~pbourke/geometry/spherefrom4/

Sphere passing through 4 points
(x1,y1,z1) (x2,y2,z2) (x3,y3,z3) (x4,y4,z4)
with centre (xc,yc,zc)
Equation: (x-xc)^2 + (y-yc)^2 + (z-zc)^2 = r^2

M = [ -              -  -  -  - ] <- don't care
    [ x1^2+y1^2+z1^2 x1 y1 z1 1 ]    because it's
    [ x2^2+y2^2+z2^2 x2 y2 z2 1 ]    symbolic. use 0's
    [ x3^2+y3^2+z3^2 x3 y3 z3 1 ]
    [ x4^2+y4^2+z4^2 x4 y4 z4 1 ]

xc = .5 M12 / M11
yc = - .5 M13 / M11
zc = .5 M14 / M11
r^2 = xc^2 + yc^2 + zc^2 - M15 / M11
*)
