open Staged
open Basetypes
open Point

module type VERTEX =
sig
  module P : POINT
  type 'a vertex_s
  type 'a simplex_s
  val dim : 'a vertex_s -> int
  (* equality *)
  val eq : 'a vertex_s -> 'a vertex_s -> 'a Bool.b
  (* isomorphic *)
  val iso : 'a vertex_s -> 'a vertex_s -> 'a Bool.b
  val create : 'a P.point_s -> 'a vertex_s
  val point : 'a vertex_s -> 'a P.point_s
  (* no mutablity requirment *)
  val attach : 'a vertex_s -> 'a simplex_s -> 'a vertex_s
  val incident : 'a vertex_s -> 'a simplex_s option
  val to_string : 'a vertex_s -> ('a, string) Staged.staged
end

(* MetaOCaml is not really friendly with records inside modules *)
type ('p,'s) vertex_record = { point : 'p; simplex : 's option }
let create_vertex_record p s = { point = p; simplex = s }

module Vertex (P : POINT)
    (* S : sig type 'a simplex_s end *) (* : VERTEX *) =
struct
  module P = P
  type 'a simplex_s (* = 'a S.simplex_s *)
  type 'a vertex_s = ('a P.point_s, 'a simplex_s) vertex_record
  let dim v = P.dim
  let eq a b = Now (a = b)
  let iso a b = P.eq a.point a.point 
  let create p = create_vertex_record p None
  let point v = v.point
  let attach v s = create_vertex_record (point v) (Some s)
  let incident v = v.simplex
  let to_string v = P.to_string v.point
end


