# GenNewtonND.pvs
# Parameters coming from the outside:
# N - number of 'dimensions'
# case - a particular test function

N := 3; case := 2;

# The variables are all real.  We could use assume to deal with that, but
# that really mucks things up, so instead we're going to fake it by
# simply forcing Maple to make everything real the 'hard' way.
unprotect('Re'): Re := y -> y;  # hack to get rid of 'Re' without assume

# Some Maple settings
interface(errorbreak=2);  # stop on an error

# Example functions, [dim, case]; will return table and vars
examples := proc(var::symbol, N)
    global y0,y1,y2,y3,y4,y5; local i;
    eval(table([
    (1,1) = exp(y0)+1/2*y0^2-2*y0,
    (1,2) = 2*y0^2+y0^4+sin(y0+1), #bug example
    (1,3) = exp(y0)+2*y0^2+y0^4+sin(y0+1),
    (2,1) = 2*y0^2+y1^4-8*y0*y1,
    (2,2) = 100*(y1-y0^2)^2+(1-y0)^2,
    (2,3) = 2*y0^2+y1^4-8*y0*y1+sin(y0^2+y1^2),
    (3,1) = 1/2*y0^2*((1/6)*y0^2+1)
            -1/2*exp((1-y1))+(y1-10)^4+(y2-4)^2,
    (3,2) = 1/2*y0^2*((1/6)*y0^2+1)
            +1/2*exp(y1-1)+(y1-10)^4+(y2-4)^2,
    (6,1)  = y0*(4*y0^2+2*y1^2+4*y0*y1+2*y1+1)+(y2-2)^2
             +(y3-5)^2+(y4-7)^2+(y5-4)^2 ]), [seq(y||i = var||i, i=0..N-1)]),
    [seq(var||i, i = 0 .. N-1)];
end proc:

# Test vector
Vtest := <-10.0, 10.0, -10.0, 10.0, 10.0, -10.0>;

# Are we somewhere that we can do this?
CheckLocation := proc(d)
    uses FileTools;
    Exists(d) and IsDirectory(d)
end proc;

if not CheckLocation("PVS-code") then
    error "This program is not being run in the proper directory";
end if;

WriteFile := proc(fileName::string, str::string)
    local fd;

    fd := fopen(fileName, WRITE);
    fprintf(fd, "%s", str);
    fclose(fd);
end proc;

WritePVSTheory := proc(theory::string, nam::string, num::posint)
    WriteFile(sprintf("%s/%s%dD.pvs", "PVS-code", nam, num), theory);
end proc;

WriteCCode := proc(ccode::string, nam::string, num::posint)
    WriteFile(sprintf("%s/%s%dD.c", "C-code", nam, num), ccode);
end proc;

getFunc := proc(N::posint, case::posint := 1, var::symbol := 'x')
    local T, argsList;
    (T,argsList) := examples(var,N);
    if assigned(T[N,case]) then
        (T[N,case], argsList, NVec(convert(var,'string'), N, "real"))
    else
        error sprintf("Case %d in dimension %d does not exist", case, N)
    end if;
end proc;

(func, FnOf, AbsFnOf) := getFunc(N,case); 

# Just for sanity's sake, a routine to have Maple check the minimum
# directly -- not used otherwise
sanity := proc(f) print(minimize(evalf(f), 'location')); end proc;

Dfunc := convert(VectorCalculus:-Jacobian(<func>, FnOf),Vector[row]);
DDfunc := VectorCalculus:-Hessian(func, FnOf);
posDef := LinearAlgebra:-IsDefinite(DDfunc);
DDfuncInverse := LinearAlgebra:-MatrixInverse(DDfunc);

# From the above Maple computations, make named, closed versions
g := LetC("func", func, AbsFnOf):
gp := LetC("Dfunc", Dfunc, AbsFnOf):
gpp := LetC("DDfunc", DDfunc, AbsFnOf):
gppinv := LetC("DDfuncInverse", DDfuncInverse, AbsFnOf):

#######################################################################
# Theory structure and its printer

`type/Theory` := 'record( name, dimension, body, kind)':
`type/NVecF` := 'NVec'(string, posint, string):
`type/Let` := 'Let'(string, anything):  # named (ground) value
`type/LetC` := 'LetC'(string, anything, anything): # named, 'closed' value

# kind is used to figure out if the theory is explicitly 'Dimensioned'

# Actually transform a theory data-structure into a string
PrintTheory := module()
    export ModuleApply;
    local b, Name, Value, Deps, line, Write,
          GenSeq, Dispatch, FT1, FT2, FuncTranslation, Translate,
          Expression, Fmt, KnownExpressions, KnownWriters,
          join, unquote, parens, str;

    ModuleApply := proc(t :: Theory, {trans := 'full'})
        local i;

        b := StringTools:-StringBuffer();
        FuncTranslation := `if`(trans='full', FT1, FT2);
        b:-appendf("%s%dD: THEORY", t:-name, t:-dimension); b:-newline();
        line("BEGIN"); b:-newline();
        if member(t:-kind,{'Dimensioned'}) then
            b:-appendf("N: posnat = %d", t:-dimension);
            b:-newline(); b:-newline();
        end if;
        for i in t:-body do Dispatch(i, KnownWriters, 'Write'); b:-newline() end do;
        b:-newline();
        b:-appendf("END %s%dD", t:-name, t:-dimension); b:-newline();
        b:-value();
    end proc;

    FT1 := ['sin'='SIN', 'cos'='COS', 'ln'='LOG', ('exp')='EXP'];
    FT2 := ['ln'='LOG', ('exp')='EXP'];

    Translate := proc(x) eval(x, FuncTranslation) end proc;

    Name := proc(v) option inline; op(1,v) end proc;
    Value := proc(v) option inline; op(2,v) end proc;
    Deps := proc(v) option inline; op(3,v) end proc;

    line := proc(s) b:-append(s); b:-newline() end proc;

    ##################################################################
    # Various printing utilities

    # printer from simple Maple expression to string, but respects PVS 
    # conventions rather than Maple's.  Extra parens, but better than nothing.
    join := proc(l, {sep::string := ","}) op(ListTools:-Join(l, sep)) end proc:
    unquote := proc(s) `if`(s[1]="`", s[2..-2], s) end proc;
    parens := proc() cat("(", args, ")") end proc;
    str := proc(e, {parent::string := ""}) 
    local wrap;
    wrap := `if`(member(parent,{"+","*","^"}),parens,cat);
    if e::symbol then
        unquote(convert(e,'string'));
    elif e::rational then
        if e<0 and (not (parent="*")) then wrap(sprintf("%a",e)) 
        else sprintf("%a",e) end if;
    elif e::function then
        cat(str(op(0,e)),parens(join(map(str,[op(e)]))))
    elif e::`*` then
        wrap(join(map(str,[op(e)], :-parent="*"),:-sep="*"))
    elif e::`+` then
        wrap(join(map(str,[op(e)], :-parent="+"),:-sep="+"))
    elif e::`^` then
        wrap := cat;
        wrap(join(map(str,[op(e)], :-parent="^"),:-sep="^"))
    else
        wrap(sprintf("%a",e))
    end if;
    end proc:

    # Routine to deal with generalized sequences
    GenSeq := proc(pre::string, p, init, last, sep::string, post::string)
        local i;
        cat(pre, join([seq(p(i),i=init..last)],:-sep = sep), post);
    end;

    Write := module ()
        local Body, Header;
        export Element, CMatrix, InnerVector, Lemma, PVSioPredicate, 
        Predicate, Function, DeclType, DeclVar, DefVar, DeclRecord, Import,
        VectorFunction, Tuple, RecordUpdate, Proof, Seq;

    Element := proc(e) Dispatch(e, KnownWriters, 'Write') end proc;
    Import := proc(s) b:-appendf("IMPORTING %s",s); b:-newline() end;

    Body := proc(m) line(cat("= ", str(Translate(m)))); end;
    Header := proc(s1,s2,s3) b:-appendf("%s%s: %s ", s1, s2, s3); end;

    InnerVector := proc(v, esep := ", ", start := "[")
        local p;
        p := j -> sprintf("(%d) := %a",j-1, Translate(v[j]));
        b:-append(GenSeq(start, p, 1, LinearAlgebra:-Dimension(v), esep, "]"))
    end proc;

    CMatrix := proc(l::LetC)
        local i, mname, m;
        uses LinearAlgebra;
        mname := cat(Name(l),Fmt:-TypedArg(Deps(l))); m := Value(l);
        line(cat(mname, " : MatrixNxN = zeroMatrixNxN WITH ["));
        for i from 1 to RowDimension(m)-1 do
            b:-append(sprintf("(%d) := zeroVectorN WITH ", i-1));
            InnerVector(m[i,1..-1]);
            line(",");
        end do; # i correctly defined after loop
        b:-append(sprintf("(%d) := zeroVectorN WITH ", i-1));
        InnerVector(m[i,1..-1]); b:-newline(); 
        line("]");
    end proc;

    VectorFunction := proc(l::LetC, typ, base)
        Header(Name(l), Fmt:-TypedArg(Deps(l)), typ);
        InnerVector(Value(l), ",\n", sprintf("= %s WITH [\n", base));
    end proc;

    DeclVar := proc(x, {var :: string := "VAR "})
        local format; global NVecF;
        format := sprintf("%s: %s%s", "%s", var, "%s");
        if x::NVecF then
            b:-appendf(format, Fmt:-ArgVec(op(x)), op(3,x)); b:-newline();
        elif (nargs=2 or nargs=3) and [x,args[2]]::[string,string] then
            b:-appendf(format, x, args[2]); b:-newline();
        else
            error sprintf("Unknown form of DeclVar declaration %a", [args]);
        end if;
    end proc;

    DefVar := proc(l::Let, typ)
        Header(Name(l), " ", typ); Body(Value(l));
    end proc;

    DeclType := proc(nam::string, typ::string)
        Header(nam, " ", "TYPE");  
        line(cat("= ",typ));
    end proc;

    DeclRecord := proc(nval::Let, typ::string)
        local p;
        p := i->cat(Value(nval)[i],":",typ);
        Header(Name(nval), "", "TYPE");  
        line(GenSeq("= [# ", p, 1, nops(Value(nval)), ","," #]")); 
    end proc;

    RecordUpdate := proc(typ, cv, pv, body)
        b:-appendf("%a: %s = %a  WITH [`%a := %s]", cv, typ, pv, cv, 
            str(Translate(body)));
    end proc;

    Predicate := proc(l::LetC)
        Header(cat(Name(l),"?"), Fmt:-TypedArg(Deps(l)), "bool");
        Body(Value(l));
    end proc;

    Function := proc(l::LetC, typ)
        Header(Name(l), Fmt:-TypedArg(Deps(l)), typ);
        Body(Value(l));
    end proc;

    Tuple := proc(l::Let, typ, n)
        local v;
        v := Value(l);
        b:-append(Name(l)); b:-append(": ");
        line(GenSeq("[",(()->typ),1,n,",","] = (")); 
        line(GenSeq("",(i->str(Translate(v[i]))), 1, n, ",","\n)"));
    end proc;

    Seq := proc(t::table, n::posint)
        local i;
        for i from 1 to n do Element(t[i]); b:-newline() end do;
    end proc;

    Lemma := proc(nam, body, {sep::string := " "})
        b:-appendf("%s: LEMMA%s%s",nam,sep,Expression(body)); 
    end proc;

    PVSioPredicate := proc(nam, body, {sep::string := " "})
        Header(nam,sep,"bool");
        line(cat("= ",Expression(body)));
    end proc;

    Proof := proc(nam, body)
        local pf_line;
        pf_line := proc() line(cat("  %|- ",args)) end proc;
        pf_line(nam, ": PROOF ");
        pf_line(body);
        pf_line("QED");
    end proc;

    end module;
    KnownWriters := [exports(Write)];

    Fmt := module ()
        export Call, Equal, TypedArg, Arg, ArgVec, Let, LetC, Close,
               Deriv, Abs, Tuple, ForAll, DForAll;
        local OneArg, TypedOneArg, PList;

    Let := proc() args[1] end;
    LetC := proc() args[1] end;

    Call := proc(l) cat(Expression(l), Arg([args[2..-1]])); end proc;
    Equal := proc(a,b) cat(Expression(a), " = ", Expression(b)); end proc;
    Deriv := proc(f) cat(" deriv ", parens(Expression(f))) end;
    Abs := proc(f,x) cat("LAMBDA",TypedArg(x),":", Expression(f)) end;
    Tuple := proc(arg::NVecF)
        GenSeq("(",(i->(cat(op(1,arg),i)), 0, op(2,arg)-1, ",",")"));
    end proc;
    Close := proc(a,b,prec)
        sprintf("abs(%s - %s) < %s", Expression(a), Expression(b), Expression(prec))
    end proc;

    ForAll := proc(avec, w)
        sprintf("FORALL%s: %s",Fmt:-TypedArg(avec),Expression(w));
    end proc;

    # discrete, statically known forall
    DForAll := proc(ov::`in`(symbol,anything), e)
        local var, rng, p; global NVecF;
        (var,rng) := op(ov);
        if rng::NVecF then
            p := i -> parens(Expression(eval(e, 
                ['position' = (()->TypedVar(sprintf("%d",i),"integer")),
                 var=TypedVar(op(1,rng), i,op(3,rng))])));
            GenSeq("\n", p, 0, op(2,rng)-1, " AND\n", "");
        elif rng::list(integer) then
            p := i -> parens(Expression(eval(e, 
                 var = TypedVar(sprintf("%d",i),"integer"))));
            GenSeq("\n", p, 0, nops(rng)-1, " AND\n", "");
        else
            error "unrecognized format %1 for arguments of DForAll", [rng];
        end if;
    end proc;

    ArgVec := proc(nam, dim) local i;
        cat(join([seq(sprintf("%s%d",nam,i),i=0..dim-1)], :-sep=", "))
    end;
    
    PList := proc(f, leftp, rightp) proc(arg)
        if arg = 'None' then "" else
            cat(leftp, `if`(arg::list, join(map(f, arg)), f(arg)),rightp)
        end if;
    end proc end proc;

    TypedOneArg := proc(arg) local a, t, i; global NVecF, None;
        if arg::NVecF then
            sprintf("%s: %s", ArgVec(op(arg)), op(3,arg))
        elif typematch(arg,TypedVar(a::string,t::string)) then
            sprintf("%s: %s", a, t);
        elif typematch(arg,TypedVar(a::string,i::nonnegint,t::string)) then
            sprintf("%s%d: %s", a, i, t);
        elif typematch(arg,TypedVar(a::list(string),t::string)) then
            sprintf("%s: %s", cat(join(a)), t);
        elif arg::identical(None) then
            ""
        else
            error sprintf("unrecognized format %a for arguments", [arg]);
        end if;
    end proc;

    OneArg := proc(arg) local a,i;
        if arg::NVecF then
            sprintf("%s", ArgVec(op(arg)))
        elif typematch(arg,TypedVar(a::string,string)) then
            sprintf("%s", a);
        elif typematch(arg,TypedVar(a::string,i::nonnegint,string)) then
            sprintf("%s%d", a, i);
        elif typematch(arg,TypedVar(a::list(string),string)) then
            sprintf("%s", cat(join(a)));
        elif arg::identical(None) then
            ""
        else
            Expression(arg)
        end if;
    end proc;

    TypedArg := PList(TypedOneArg," (",")");
    Arg := PList(OneArg,"(",")");
    end module;
    KnownExpressions := [exports(Fmt)];

    Expression := proc(e) Dispatch(e, KnownExpressions, 'Fmt') end proc;

    Dispatch := proc(e, known, M::name)
        if not member(op(0,e), known) then
            error "Case of element %1 not implemented in %2", op(0,e), M
        else
            M[op(0,e)](op(e));
        end if;
    end proc;
end module:

####################################################################
# Start of definition of theories

#################################
# Inverse -- A * B = Identity

InverseThy := proc(A::LetC,B::LetC,argVec,N)
    local idM, i, j;

    idM := LetC("identityMatrixNxN", LinearAlgebra:-IdentityMatrix(N), 'None');
    i := TypedVar("i", "below(N)");
    j := TypedVar("j", "below(N)");
    [
    Import("MatrixNxN[N]"), # defines multiplyVectors
    CMatrix(idM),
    CMatrix(A),
    CMatrix(B),
    # Should be done via 'expanding' something more basic?
    Lemma("checkInverse",
        ForAll([argVec, TypedVar(["i","j"],"below(N)")], 
        Equal( Call(Let("multiplyVectors",NONE), Call(A,argVec),
                        Call(B,argVec), i, j), 
               Call(Call(idM, i), j)))),
    Proof("checkInverse", "(then (grind) (grind-reals))") ]
end proc:

thy1 := Record('name'="Inverse", 'dimension'=N,
    'body'=InverseThy(gpp, gppinv, AbsFnOf,N), 
    'kind' = 'Dimensioned');

WritePVSTheory(PrintTheory(thy1), "Inverse", N);

##################################
# Derivative
DerivativeThy := proc(f, fp, fpp, argVec,N) 
    local y, i, j;
[ Import("MatrixNxN[N], Derivative_exp_log"), # defines zeroVectorN
  Function(f, "real"),
  VectorFunction(fp, "Vector[N]", "zeroVectorN"),
  CMatrix(fpp),
  DeclVar(argVec),
  Lemma("checkJACOBIAN", 
      DForAll(y in argVec, 
          Equal(Deriv(Abs(Call(f,argVec),y)), 
                Abs(Call(Call(fp,argVec),position(y)),y)))),
  Proof("checkJACOBIAN", "(then (skolem!) (then (repeat (expand* \"func\" \"Dfunc\" \"DDfunc\" \"expt\" \"^\" \"ln\" \"restrict\" \"exp\" \"sq\")) (then (ground) (repeat (deriv)))))"),
# do by linearization of indices
  Lemma("checkHESSIAN", 
      DForAll(j in [i$i=0..N-1], DForAll(y in argVec,
          Equal(Deriv(Abs(Call(Call(fp,argVec),j),y)), 
                Abs(Call(Call(Call(fpp,argVec),j),position(y)),y))))),
  Proof("checkHESSIAN", "(then (skolem!) (then (repeat (expand* \"func\" \"Dfunc\" \"DDfunc\" \"expt\" \"^\" \"ln\" \"restrict\" \"exp\" \"sq\")) (then (ground) (repeat (deriv)))))"),
  Proof("check*_TCC*" , "(then (skolem!) (then (repeat (expand* \"func\" \"Dfunc\" \"DDfunc\" \"expt\" \"^\" \"ln\" \"restrict\" \"exp\" \"sq\")) (then (ground) (then (repeat (diff?)) (assert)))))") ]
end proc;

thy2 := Record('name'="Derivative", 'dimension'=N,
    'body'=DerivativeThy(g, gp, gpp, AbsFnOf, N), kind = 'Dimensioned');

WritePVSTheory(PrintTheory(thy2, trans='partial'), "Derivative", N);

##################################
# GenNewton
stepF := simplify(Dfunc . LinearAlgebra:-MatrixInverse(DDfunc));
stepX := eval(stepF, [seq(FnOf[i+1]=t(i),i=0..N-1)]); # 0-based
stepX1 := unapply(eval(stepF, [seq(FnOf[i]=t[i],i=1..N)]), t); # 1-based

XpreviousArray := Vector[row](N, (i->Xprevious || (i-1)));
XpreviousStep := eval(stepF, [seq(FnOf[i] = XpreviousArray[i],i=1..N)]);
Xnext := eval(XpreviousArray-XpreviousStep);

V0 := Vector[row](N, (i->Vtest[i])); # Truncate Vtest to dimension N
iter := 10;
Newton := proc(V, it::nonnegint, stepF)
    `if`(it=0, V, Newton(V - stepF(V), it-1, stepF))
end proc;
MapleResult := map(convert, Newton(V0, iter, stepX1), 'rational');

# name some important values computed above so PVS can reason with them
mres := LetC("MapleResult", MapleResult, None):
varV := TypedVar("V", "Vector[N]");
pdef := LetC("pos_def", eval(posDef, [seq(FnOf[i+1]=V(i),i=0..N-1)]), varV);
NewtStep := LetC("stepX", stepX, TypedVar("t", "NType"));
testV0 := LetC("V0", V0, None);
numsteps := Let("iter", iter);

InstNewtonThy := proc(step, n, v, testv, pd, mapres, precision)
    local i, pvsres, prec; global V;
    i := TypedVar("i", "below(N)");
    pvsres := LetC("Result", ''NewtonND(V0,iter)'', 'None'); # uneval hack
    prec := Let("precision", precision);
    [
    Import("lnexp@ln_exp, MatrixNxN[N]"),
    Predicate(pd),
    Predicate(LetC("NType", pos_def?(V), v)), # hackish for 2nd arg
    DeclType("NType", "(NType?)"),
    VectorFunction(step, "Vector[N]", "t"),
    Import("NewtonTheoryNDim[N,NType?,stepX]"), # defines NewtonND
    VectorFunction(testv, "NType", "zeroVectorN"),
    DefVar(n, "posnat"),
    Function(pvsres, "Vector[N]"),
    DefVar(prec, "real"),
    VectorFunction(mapres, "Vector[N]", "zeroVectorN"),
    PVSioPredicate("checkNewton",
        ForAll(i, Close(Call(pvsres,i), Call(mapres,i), prec)))]
end proc:

thy3 := Record('name'="InstNewtonTheory", 'dimension'=N, 
    'body'=InstNewtonThy(NewtStep, numsteps, varV, testV0, pdef, mres, 10^(-10)), 
    'kind' = 'Dimensioned');

WritePVSTheory(PrintTheory(thy3), "GenNewton", N);

######################
# computations for taking a step of Newton
# used in both stepND.c and EquivalenceND.pvs

XpreviousArgs := [seq(Xprevious || i, i = 0 .. N-1)];
is_float := x -> x::float:
XpreviousArgsTyped := map( is_float, XpreviousArgs);
resultArgs := [seq(r||i, i = 0 .. N-1)];
Xvec := zip(`=`, resultArgs, convert(Xnext,'list'));
NewtonStep := codegen:-optimize(Xvec):
NewtonStep := [codegen:-optimize([NewtonStep], 'tryhard')]:
localArgs := `minus`(`union`(op(map(indets,NewtonStep,'name'))),
                     convert(XpreviousArgs, set));
localArgs := sort(convert(localArgs, 'list'));  # a bit more stable
nlocs := nops(localArgs): nsteps := nops(NewtonStep):

# Names
XN := Let("Xnext", Xnext):
XP := NVec("Xprevious", N, "real"):
Locs := Let("t_type", localArgs):

######################
# stepND.c
toConvert :=
    CompSeq(params = [op(XpreviousArgsTyped), r],
            locals = map(is_float, localArgs),
        [op(NewtonStep), seq(r[i+1]=r||i, i=0..N-1), ''RETURN''(NULL)]);
NewtonStepND := convert( toConvert, 'procedure');

TransTo := proc(lang) global NewtonStepND;
    CodeGeneration:-Translate(NewtonStepND, 'language'=lang, 'output' = string);
end proc:
CfuncString := TransTo("C"):
WriteCCode(CfuncString, "step-", N);

# But we can just as easily do others
fprintf("JavaEx.java", "%s", TransTo("Java")):
fprintf("FortranEx.f", "%s", TransTo("Fortran")):
fprintf("MatlabEx.m", "%s", TransTo("Matlab")):
fprintf("VisualBasicEx.vb", "%s", TransTo("VisualBasic")):

######################
# EquivalenceND.pvs

# For all occurences of variables from set vars in expr, prepend prefix`
# For reasons of obscure Maple-isms, first use _ then `
translate := proc(expr, prefix, vars)
    evalindets(expr, 'symbol',
        proc(n) if member(n,vars) then cat(prefix,"`",n) else n end if end)
end proc:

# build incrementally via a table
SLPtoRecordUpdates := proc(steps, typ, t, resArgs)
    local trans_vars, pvar, cvar, cexpr, tmpStr, res, i;
    trans_vars := {}; pvar := t: res := table():
    for i from 1 to nops(steps) do
    (cvar,cexpr) := op(steps[i]);
    tmpStr := translate(cexpr, pvar, trans_vars);
        res[i] := RecordUpdate(typ, cvar, pvar, tmpStr);
    pvar := cvar;
    if not member(cvar,resArgs) then 
            trans_vars := trans_vars union {cvar} 
        end if;
    end do;
    res;
end proc;

EquivalenceThy := proc(YN, YP, locs::Let, steps, nam::string, N, resultArgs) 
    local n; 
    n := nops(steps);
    [
    DeclVar(YP, 'var' = ""),
    Tuple(YN, "real", N),
    DeclRecord(locs, "real"),
    DeclVar(nam, op(1,locs), 'var'=""),
    Seq(SLPtoRecordUpdates(steps, op(1,locs), convert(nam,'symbol'), resultArgs), n),
    Lemma("Equiv",
    ForAll(YP, Equal(YN, Tuple(NVec(sprintf("r%d`r",N-1), N, "real"))))),
    Proof("Equiv", "(then (grind) (grind-reals))") ]
end proc;

thy4 := Record('name'="Equivalence", 'dimension'=N, 
    'body'= EquivalenceThy(XN, XP, Locs, NewtonStep, "t", N, resultArgs), 
    'kind' = 'NonDim');

WritePVSTheory(PrintTheory(thy4), "Equivalence", N);

########################################################
#  Generate whole program
#  A bit hacky, but it works.

NewtonGen := proc(N,iter,V,step)
local i, M, mainG, printG, Xnext;
mainG := proc()
    local Xnext, i;
    Xnext := array(1..NN,INIT);

    printVector(Xnext, 0);
    for i from 0 to ITER-1 do
        NewtonStepND(XXXX,Xnext);
        printVector(Xnext, i+1);
    end do;
    printf ("\n\n");
    0;
end proc:
printG := proc(X, iterNum::integer)
    local i;
    printf ("\nX[%d] = [ ", iterNum);
    for i from 0 to NN-1 do
        printf ("%.10lf ", X[i+1]);
    end do;
    printf ("]");
end proc:
M := module () export main, printVector, NewtonStepND; end module;
M:-main := subs({ITER=iter, NN=N, INIT=[seq(V[i],i=1..N)],
    XXXX=(seq(Xnext[i],i=1..N))}, eval(mainG));
M:-printVector := subs({NN=N}, eval(printG));
M:-NewtonStepND := eval(step);
eval(M);
end proc;

Newt3 := NewtonGen(N, 30, Vtest, NewtonStepND):
wholeprog := CodeGeneration[C](Newt3,output=string):

WriteCCode(wholeprog, "whole-", N);
