/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *							Input Parameter Reader								   *
 *																				   *
 *								  Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include "owe_header.h"


void input_parameter_reader_read_input(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  input_parameter_reader_read_input
	*  ---------------------------------
    *
	*	This function reads the input parameter file, which has to be located in the same location
	*	as the program and its name has to be "input". From the information contained in such file
	*	the function initializes the following owe-setting variables:
	*								- owe_setting.incidence_upper_bound
	*								- owe_setting.incidence_lower_bound
	*	The following Abstract data types are initialized from the information provided in the input
	*	parameter file:
	*								- metric_polytope
	*								- orbit_databank
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


	FILE* input_parameter_file;						/* Pointer to the input parameter file. */
	char input_parameter_file_name[]="input.txt";	/* String of name of input parameter file. */
	char buffer[1024];								/* String where each line of the input parameter file will be loaded (one by one). */
	char* pre_processed_buffer_ptr;					/* Pointer to pre-processed buffer (obtained after calling input_parameter_reader_pre_process_buffer_line function. */
	enum input_parameter_file_sections {			/* Set of sections that composes the input parameter file. */
		main_parameters,
		initial_orbit_set
	} current_section;
	unsigned char initialization_flags = 0;			/* Variable used to determine if all required parameters have been initialized.
													   When each of the following parameters is set from the value obtained in the
													   input parameter file, its corresponding bit in the initialization_flags variable
													   is set to 1 by using bit-wise operations.
															- metric polytope:				00000001 (decimal 1)
															- incidence lower bound:		00000010 (decimal 2)
															- incidence upper bound:		00000100 (decimal 4)
															- adjacency lower bound:		00001000 (decimal 8)
															- adjacency upper bound:		00010000 (decimal 16)
															- orbit queue size:				00100000 (decimal 32)
															- double description time limit:01000000 (decimal 64)
														Uppon finalization of input parameter setting, the variable	initialization_flags
														should have the decimal value 15. */

	char error_description[512];					/* If there is an error found while reading the input parameter file, the description of
													   the error is stored in this string so it can be sent to the owe error handler function. */
	/* Open input parameter file. */
	if((input_parameter_file = fopen(input_parameter_file_name,"r")) == NULL) owe_error_handler_raise_error("cannot open input parameter file", 1);
	
	/* The first section of the input parameter file is by default "main parameters". */
	current_section = main_parameters;

	/* Load into buffer the next line in the input parameter file. */
	while (fgets(buffer,1024,input_parameter_file)) {

		/* eliminate ASCII characters from the begining and end of the current buffer */
		pre_processed_buffer_ptr = input_parameter_reader_pre_process_buffer_line(buffer);

		/* If the current line is not empty or is not a comment, analize it. */
		if(*buffer!='*' && *buffer!=10 && *buffer!=13) {

			/* Check if the section "initial orbit set" is begining in the current line and set the current section acordinly.
				Since the begining of this section implies the end of the "main parameters" section, perform the validation of
				main parameters before proceding. */
			if(!strcmp(pre_processed_buffer_ptr,"initial orbit set")) {
				/* Che for the previous section values to be complete. */
				if(initialization_flags!=127) {
					strcpy(error_description, "missing input parameter(s) in input parameter file:\n");
					if(!(initialization_flags&1)) strcat(error_description, " - metric polytope\n");
					if(!(initialization_flags&2)) strcat(error_description, " - incidence lower bound\n");
					if(!(initialization_flags&4)) strcat(error_description, " - incidence upper bound\n");
					if(!(initialization_flags&8)) strcat(error_description, " - adjacency lower bound\n");
					if(!(initialization_flags&16)) strcat(error_description, " - adjacency upper bound\n");
					if(!(initialization_flags&32)) strcat(error_description, " - orbit queue size\n");
					if(!(initialization_flags&64)) strcat(error_description, " - double description time limit\n");
					fclose(input_parameter_file);
					owe_error_handler_raise_error(error_description, 1);
				}
				else
					current_section = initial_orbit_set;
			}
			else
				/* If neither ("initial orbit set" and "precomputed orbit set") sections begin in the current line, then process the line normally. */
				if(current_section==main_parameters)
					/* If the current section is "main parameters", process the current line accordinly. */
					input_parameter_reader_process_main_parameters_section(pre_processed_buffer_ptr, &initialization_flags);
				else
					/* If the current section is not "main parameters", then it is "initial orbit set". */
					input_parameter_reader_process_orbit_set_section(pre_processed_buffer_ptr);
		}

	}
	printf("Input parameters\n----------------\n Metric polytope = %u\n Incidence:\n   Upper bound = %lu\n   Lower bound = %lu\n Adjacency:\n   Upper bound = %lu\n   Lower bound = %lu\n Double Description time limit (minutes) = %lu\n Orbit queue size = %lu\n Initial orbits: (total = %lu)\n   In range (to be computed) = %lu\n   Outside range (wont be computed) = %lu\n Precomputed orbits = %lu\n\n", owe_setting.metric_polytope.n, owe_setting.incidence_upper_bound, owe_setting.incidence_lower_bound, owe_setting.adjacency_upper_bound, owe_setting.adjacency_lower_bound, owe_setting.double_description_time_limit, orbit_databank_get_orbit_queue_size(), orbit_databank_get_number_of_orbits_known(),  orbit_databank_get_number_of_orbits_to_compute(), orbit_databank_get_number_of_orbits_known() - orbit_databank_get_precomputed_orbit_count() - orbit_databank_get_number_of_orbits_to_compute(), orbit_databank_get_precomputed_orbit_count()  );
	fclose(input_parameter_file);


}



char* input_parameter_reader_pre_process_buffer_line(char* buffer) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  input_parameter_reader_pre_process_buffer_line
	*  ----------------------------------------------
	*  Parameters:
	*		- buffer: string to pre-process.
	*
	*  Return value:
	*		- Pointer to the pre-processed string.
    *
	*	This function eliminates all ASCII characters below 33 from the begining and end of the
	*	buffer string and converts its characters to lower case.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	int buffer_length;	/* Variable used traversing the buffer string. */
	char* buffer_ptr;				/* Pointer that will contain the value to return. */

	/* Clear all characters with ASCII code below 33 from the end of the string. */
	if(!(buffer_length = strlen(buffer)-1)) return(buffer);
	while(buffer[buffer_length]<33) buffer[buffer_length--]=0;

	/* Clear all characters with ASCII code below 33 from the begining of the string. */
	buffer_ptr = buffer;
	while(*buffer_ptr<33) buffer_ptr++;
	
	/* Convert all characters to lower case. */
	buffer_length = strlen(buffer_ptr)-1;
	while(buffer_length--) buffer_ptr[buffer_length] = (char)tolower(buffer_ptr[buffer_length]);

	return(buffer_ptr);
}




void input_parameter_reader_process_main_parameters_section(char* buffer,unsigned char *initialization_flags) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  input_parameter_reader_process_main_parameters_section
	*  ------------------------------------------------------
	*  Parameters:
	*		- buffer: pre-processed string that corresponds to a line in the metric_polytope_and_incidence_rage_values
	*				  section of the input parameter file.
    *
	*	This function decomposes the input buffer into two strings: parameter and value. This is
	*	done by splitting the string where the '=' character is. After this, all ASCII characters
	*	below 33 between the parameter and value are cleared. Finally, for each parameter, the function
	*	processes its value by making the required data converstions and initializing the computation.
	*
	*	Parameters:
	*		- "metric polytope": its value is used for initializing owe_setting.metric_polytope information.
	*		- "Incidence lower bound": its value is assigned to owe.setting.incidence_lower_bound.
	*		- "Incidence upper bound": its value is assigned to owe.setting.incidence_upper_bound.
	*		- "orbit heap size": its value is used for initializing the orbit databank.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	char* parameter;			/* String that will contain the name of the parameter. */
	char* value;				/* String that will contain the value of the parameter. */
	char* buffer_end;			/* Variable used for verifying the end of the parameter line. */
	long int parameter_integer_value;	/* Auxiliary variable used to store the converted string to integer value. */
	int length;		/* Variable used for clearing ASCII characters below 33 that may appear around the '=' character in the parameter line. */

	char error_description[512];					/* If there is an error found while reading the input parameter file, the description of
													   the error is stored in this string so it can be sent to the owe error handler function. */
	parameter = buffer;
	value = buffer;
	buffer_end = buffer+strlen(buffer);

	/* Find the '=' character in the buffer. */
	while(*value!='='&& value<=buffer_end) value++;
	
	/* If the character '=' is not found, then the parameter is not properly written (syntax). */
	if(value==buffer_end) {
		sprintf(error_description,"unknown value in input parameter file: %s\n",buffer);
		owe_error_handler_raise_error(error_description, 0);
	}
	else
	{


		/* The buffer is split into parameter and value strings. */
		*value++=0;

		/* Clear all characters with ASCII code below 33 from the end of the parameter. */
		length = strlen(parameter)-1;
		while(parameter[length]<33) parameter[length--]=0;

		/* Clear all characters with ASCII code below 33 from the begining of value. */
		while(*value<33) value++;

		/* Each of the following set of statements corresponds to each of the different
			possible parameters in this section. */

		/* Parameter "metric polytope". */
		if(!strcmp(parameter,"metric polytope")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Initialize canon. */
			init_canon(parameter_integer_value);
			
			/* Initialize metric polytope. */
			metric_polytope_initialize((unsigned char)parameter_integer_value);
			
			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 1);

		}

		/* Parameter "incidence lower bound". */
		if(!strcmp(parameter,"incidence lower bound")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			owe_setting.incidence_lower_bound=parameter_integer_value;
			
			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 2);
		}

		/* Parameter "incidence upper bound". */
		if(!strcmp(parameter,"incidence upper bound")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			owe_setting.incidence_upper_bound=parameter_integer_value;

			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 4);
		}


		/* Parameter "adjacency lower bound". */
		if(!strcmp(parameter,"adjacency lower bound")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			owe_setting.adjacency_lower_bound=parameter_integer_value;
			
			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 8);
		}

		/* Parameter "adjacency upper bound". */
		if(!strcmp(parameter,"adjacency upper bound")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			owe_setting.adjacency_upper_bound=parameter_integer_value;

			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 16);
		}

		/* Parameter "orbit queue size". */
		if(!strcmp(parameter,"orbit queue size")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			orbit_databank_initialize(parameter_integer_value);

			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 32);
		}

		/* Parameter "double description time limit". */
		if(!strcmp(parameter,"double description time limit")) {
			/* Store the value as an integer number. */
			parameter_integer_value = atol(value);	

			/* Set the corresponding value of the owe_setting variable. */
			owe_setting.double_description_time_limit = parameter_integer_value;

			/* Set the corresponding initialization flag. */
			*initialization_flags = (unsigned char) (*initialization_flags | 64);
		}

	}

}




void input_parameter_reader_process_orbit_set_section(char* buffer) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  input_parameter_reader_process_orbit_set_section
	*  ------------------------------------------------
	*  Parameters:
	*		- buffer: pre-processed string that corresponds to a line in the "initial orbit set" or "precomputed orbit set"
	*				  section of the input parameter file.
    *
	*	This function obtains from the buffer string the incidence, coordinates and denominator of the orbit contained within.
	*	It converts them into the apropiate coordinate format and it preloads it into the orbit databank through the
	*	orbit_databank_preload_orbit function. Depending on the value of the "precomputed" variable, the orbit will
	*	be preloaded as precomputed or not (parameter sent to the orbit_databank_preload_orbit function.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	coordinate orbit_coordinate;	/* This variable will contain the preloaded orbit as follows:
										incidence, orbit coordinates, orbit denominator. For this,
										the structure will have owe_setting.metric_polytope.dimension+2
										items. */
	char* buffer_ptr;				/* Auxiliary pointer used for parsing the buffer */
	char* buffer_end;				/* Auxiliary pointer used for parsing the buffer */
	unsigned int i;					/* This variable is used for parsing the elements of each orbit. */
	unsigned int n;					/* This variable is used for parsing the elements of each orbit. */
	unsigned char computed_or_precomputed;		/* This variable will contain 1 if the current orbit is computed or precomputed and 0 otherwise. */

	/* Allocate memory for orbit_coordinate. */
	if ((orbit_coordinate = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+2)))==0) owe_error_handler_raise_error("memory allocation error (orbit_parameter variable : input_parameter_reader_process_orbit_set_section function : input_parameter_reader ADT)",1);

	/* Keep track of the end of buffer in case there is a syntax error in an orbit. */
	buffer_end = buffer+strlen(buffer);
	
	/* Determine if the orbit is "computed/precomputed" or "not-computed" */
	while(!(*buffer == 'c' || *buffer == 'p' || *buffer == 'n') && buffer <= buffer_end) buffer++;
	if(*buffer == 'n')
		computed_or_precomputed = 0;
	else
		computed_or_precomputed = 1;

	/* The denominator is initialized with 0, in case the orbit was not properly loaded, the denominator should remain in 0. */
	orbit_coordinate[owe_setting.metric_polytope.dimension+1] = 0;

	/* Load into the coordinate structure the incidence, followed by the orbit coordinates and finally the orbit denominator. */
	n = owe_setting.metric_polytope.dimension + 2;
	for(i=0;i<n && buffer <= buffer_end;i++) {
		while((*buffer<48 || *buffer>57) && buffer <= buffer_end) buffer++;
		buffer_ptr = buffer+1;
		while(*buffer_ptr>47 && *buffer_ptr<58 && buffer_ptr <= buffer_end) buffer_ptr++;
		*buffer_ptr = 0;
		orbit_coordinate[i] = atoi(buffer);
		buffer = buffer_ptr;
	}

	/* Verify if the last element of the coordinate structure (denominator) is greater than 0, which indicates all elements of the
		orbit were loaded. If this is the case, the orbit is preloaded into the orbit databank. */
	if(orbit_coordinate[owe_setting.metric_polytope.dimension+1])
		orbit_databank_insert_orbit(orbit_coordinate+1, *orbit_coordinate, computed_or_precomputed);
	else
		/* If the value in the dimension position of the coordinate structure is 0, then something must have been wrong in the orbit syntax. */
		owe_error_handler_raise_error("preloaded orbit syntax error in input parameter file", 0);

	free(orbit_coordinate);

}
