/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *	    Orbit tree (binary red-black tree implementation) Abstract Data Type.      *
 *																				   *
 *								Source file										   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include"owe_header.h"

static orbit_tree_node orbit_tree_sentinel = { orbit_tree_NIL, orbit_tree_NIL, 0, orbit_tree_BLACK, 0};

void orbit_tree_initialize(orbit_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_initialize
	*  ---------------------
	*  Parameters:
	*		- Memory address of the red-black orbit tree on which the operation will be performed.
	*		  Pointer to pointer is used to access the information, so in the function call the parameter
	*		  must use the address operator '&' before the name of the variable.
	*
	*		The function initialize the value of the pointer to be the root of an empty
	*		red-black orbit tree.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	tree->root = orbit_tree_NIL;
	tree->size = 0;
}

void orbit_tree_delete(orbit_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbitwise_adjacency_tree_delete
	*  -----------------------------------------
	*  Parameters:
	*		- Memory address of the red-black orbitwise adjacency tree on which the operation will be performed.
	*		  Pointer to pointer is used to access the information, so in the function call the parameter
	*		  must use the address operator '&' before the name of the variable.
	*
	*		The function initialize the value of the pointer to be the root of an empty
	*		red-black orbitwise adjacency tree.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_tree_delete_tree_structure(tree);
	/*orbit_tree_delete_tree_structure_recursive(tree->root);*/
	tree->size = 0;
}

orbit_ptr orbit_tree_insert(orbit_tree_ptr tree, coordinate canonical_representative) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	 *  orbit_tree_insert
	 *  ---------------------------
	 *  Parameters:
	 *		- Coordinate of the canonical representative (inmediatelly followed by the denominator)
	 *		  of the orbit to be inserted into the red-black orbit tree.
	 *  
	 *  Return values:
	 *		- If the orbit was not previously inserted:
	 *           A new node is created (allocated and initialized) in its corredponsing location of
	 *           the red black orbit tree. An orbit structure is allocated in memory by calling the
	 *			 orbit_create_orbit function (its values are initialized). Finally, the value returned
	 *			 is a pointer to the newly allocated orbit structure.
	 *
	 *      - If the orbit was previously inserted:
	 *			  The red black orbit tree structure remains intact, no memory is allocated and
	 *            the function returns the pointer to the already existing orbit.
	 *
	 *  Note: this function does not modify or de-allocate its input parameter.
	 *
	 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    orbit_tree_node *current, *parent, *x;

    /* find future parent */
    current = tree->root;
    parent = 0;
    while (current != orbit_tree_NIL) {
		if (orbit_tree_compEQ(canonical_representative, current->orbit->canonical_representative)) 
			return current->orbit;
        parent = current;
		current = orbit_tree_compLT(canonical_representative, current->orbit->canonical_representative) ?
            current->left : current->right;
    }

	/* Increment the size of the tree. */
	tree->size++;

    /* setup new node */
    if ((x = (orbit_tree_node*) malloc (sizeof(*x))) == 0) owe_error_handler_raise_error("memory allocation error (orbit_tree_node variable : orbit_tree_insert function : orbit_tree ADT)", 1);
    /* set the red-black tree node properties */
	x->parent = parent;
    x->left = orbit_tree_NIL;
    x->right = orbit_tree_NIL;
    x->color = orbit_tree_RED;
	
	/* allocate memory for the vertex coordinate (with denominator) within the information of the orbit */
	/*if (!(x->orbit->canonical_representative = (coordinate) malloc(sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1)))) owe_error_handler_raise_error("memory allocation error (coordinate variable : orbit_create_orbit function: orbit ADT)", 1);*/
	x->orbit = orbit_create_orbit(canonical_representative);
	
	/* set the vertex coordinate value of the newly inserted orbit */
	memcpy((x->orbit->canonical_representative),canonical_representative,sizeof(coordinate_item)*(owe_setting.metric_polytope.dimension+1));

    /* insert node in tree */
    if(parent) {
		if(orbit_tree_compLT(canonical_representative, parent->orbit->canonical_representative))
            parent->left = x;
        else
            parent->right = x;
    } else {
        tree->root = x;
    }

    orbit_tree_insertFixup(&tree->root, x);

	return x->orbit;
}



orbit_ptr orbit_tree_find(orbit_tree_ptr tree, coordinate canonical_representative) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_find
	*  ---------------
	*  Parameters:
	*		- Coordinate of the canonical representative (inmediatelly followed by the denominator)
	*		  of the orbit whose presence in the red-black orbit tree wants to be determined
	*		  (whether it is in the tree or not).
	*
	*  Return values:
	*		- If an orbit with the input coordinate is in the red-black orbit tree:
	*			 The value returned is a pointer to the orbit tree node whose coordinate is that of the input
	*			 parameter.
	*
    *		- If the orbit with the input coordinate is not in the red-black orbit tree:
	*			 The returned value is 0.
	*
	*  Note: this function does not modify or de-allocate its input parameter.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    orbit_tree_node_ptr current = tree->root;
    while(current != orbit_tree_NIL) {
        if(orbit_tree_compEQ(canonical_representative, current->orbit->canonical_representative)) {
			return current->orbit;
        } else {
            current = orbit_tree_compLT (canonical_representative, current->orbit->canonical_representative) ?
                current->left : current->right;
        }
    }
	return 0;
}






void orbit_tree_print_orbits_lexicographically_ordered(orbit_tree_ptr tree, FILE* auxiliary_output) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_print_orbits_lexicographically_ordered
	*  -------------------------------------------------
	*  Parameters:
	*		- tree: pointer to the orbit tree to print
	*		- auxiliary_output: stream (file pointer) to which the orbits will also be printed.
	*							The list header and footer are not printed in the auxiliary output.
	*							If auxiliary_output is null (0), printing will be performed on the
	*							standard output only.
	*
	*	This function prints the contents of the orbit databank, specifying the incidence of each
	*	orbit in it, and whether it is computed (incidence within range), precomputed (preloaded
	*	from the input parameter file), or not computed(incidence outside range).
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_tree_node_ptr node;			/* This variable is used for parsing the tree iteratively. */
	orbit_tree_node_ptr *node_stack;	/* This variable will contain the auxiliary stack used for the itearive parsing of the orbit tree. */
	unsigned long int stack_top = 0;	/* Stack top indicator. */
	unsigned int i;						/* This variable is used for the auxiliary output coordinate printing. */

	/* Generate an auxiliary stack to parse the orbit tree iteratively */
	node_stack = (orbit_tree_node_ptr*) malloc(sizeof(orbit_tree_node_ptr)*(2+2*owe_lib_log_base_2(1+tree->size)));

	/* Print the header of the output. */
	printf("Orbit databank contents (lexycographically ordered):\n\n    Incidence  Orbit (canonical representative)\n ----------------------------------------------");

	/* Iterative parsing of the tree. */
	if(tree->root != orbit_tree_NIL) {
		node = tree->root->left;
		node_stack[stack_top++] = tree->root;
		while(node != orbit_tree_NIL || stack_top) {
			if(node != orbit_tree_NIL) {
				node_stack[stack_top++] = node;
				node = node->left;
			}
			else
			{
				node = node_stack[--stack_top];
				
				/* Print current orbit */
				printf("\n %c %10ld  ", (node->orbit->pre_computed)?'P':((node->orbit->incidence >= owe_setting.incidence_lower_bound && node->orbit->incidence <= owe_setting.incidence_upper_bound) && !node->orbit->skipped)?'C':'N', node->orbit->incidence); orbit_print_coordinate(node->orbit->canonical_representative);
				
				/* If the auxiliary output is not null, print the orbit also in it. */
				if(auxiliary_output) {
					fprintf(auxiliary_output, "\n %c %10ld  ", (node->orbit->pre_computed)?'P':((node->orbit->incidence >= owe_setting.incidence_lower_bound && node->orbit->incidence <= owe_setting.incidence_upper_bound) && !node->orbit->skipped)?'C':'N', node->orbit->incidence);
					for(i=0;i<owe_setting.metric_polytope.dimension;i++) fprintf(auxiliary_output, "%d ", node->orbit->canonical_representative[i]);
					fprintf(auxiliary_output, "| %d",node->orbit->canonical_representative[i]);
				
				}

				node = node->right;
			}

		}

	}
	/* Print the orbit list footer. */
	printf("\n\n P = pre-computed; C = computed; N = not computed");
}




void orbit_tree_load_orbits_into_queue(orbit_tree_ptr tree, incidence_sorted_orbit_priority_queue_ptr queue) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_load_orbits_into_queue
	*  ---------------------------------
	*
	*	Parameters:
	*			- tree: tree that will be loaded into the queue.
	*			- queue: queue in which the orbits of the tree will be loaded.
	*
	*	This function can be auxiliary used for sorting the orbits of a tree by incidence,
	*	Given that once loaded in the queue, by extracting them one by one using extract_min
	*	the result is the complete orbit set ordered by incidence.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	orbit_tree_node_ptr node;			/* This variable is used for parsing the tree iteratively. */
	orbit_tree_node_ptr *node_stack;	/* This variable will contain the auxiliary stack used for the itearive parsing of the orbit tree. */
	unsigned long int stack_top = 0;	/* Stack top indicator. */

	/* Generate an auxiliary stack to parse the orbit tree iteratively */
	node_stack = (orbit_tree_node_ptr*) malloc(sizeof(orbit_tree_node_ptr)*(2+2*owe_lib_log_base_2(1+tree->size)));

	/* Initialy the queue size has to be 0. */
	queue->size=0;

	/* Iterative parsing of the tree. */
	if(tree->root) {
		node = tree->root->left;
		node_stack[stack_top++] = tree->root;
		while(node != orbit_tree_NIL || stack_top) {
			if(node != orbit_tree_NIL) {
				node_stack[stack_top++] = node;
				node = node->left;
			}
			else
			{
				node = node_stack[--stack_top];
				incidence_sorted_orbit_priority_queue_insert(queue,node->orbit);
				node = node->right;
			}

		}

	}

}



void orbit_tree_print_canonical_representatives(orbit_tree_ptr tree) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_tree_print_canonical_representatives
	*  ------------------------------------------
	*  Parameters:
	*		- pointer to the orbit tree to print
	*
	*	This function prints the contents of the orbit databank, but only its canonical representatives.
	*	This is done by calling an auxiliary recursive function orbit_tree_print_canonical_representatives_inorder.
	*	The purpose of this function is for debugging purposes.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	printf("\n Orbit Databank contents (canonical representatives lexicographically sorted)\n ----------------------------------------------------------------------------\n");
	orbit_tree_print_canonical_representatives_inorder(tree->root);
}


void orbit_tree_print_canonical_representatives_inorder(orbit_tree_node *node) {

	/**************************
    *  In order tree parsing  *
    **************************/
	if(node!=orbit_tree_NIL) {
		orbit_tree_print_canonical_representatives_inorder(node->left);
		printf(" "); orbit_print_coordinate(node->orbit->canonical_representative); printf("\n");
		orbit_tree_print_canonical_representatives_inorder(node->right);
	}
}



static void orbit_tree_rotateLeft(orbit_tree_node_ptr *tree, orbit_tree_node *x) {

   /**************************
    *  rotate node x to left *
    **************************/

    orbit_tree_node *y = x->right;

    /* establish x->right link */
    x->right = y->left;
    if (y->left != orbit_tree_NIL) y->left->parent = x;

    /* establish y->parent link */
    if (y != orbit_tree_NIL) y->parent = x->parent;
    if (x->parent) {
        if (x == x->parent->left)
            x->parent->left = y;
        else
            x->parent->right = y;
    } else {
        *tree = y;
    }

    /* link x and y */
    y->left = x;
    if (x != orbit_tree_NIL) x->parent = y;

}



static void orbit_tree_rotateRight(orbit_tree_node_ptr *tree, orbit_tree_node *x) {

   /****************************
    *  rotate node x to right  *
    ****************************/

    orbit_tree_node *y = x->left;

    /* establish x->left link */
    x->left = y->right;
    if (y->right != orbit_tree_NIL) y->right->parent = x;

    /* establish y->parent link */
    if (y != orbit_tree_NIL) y->parent = x->parent;
    if (x->parent) {
        if (x == x->parent->right)
            x->parent->right = y;
        else
            x->parent->left = y;
    } else {
        *tree = y;
    }

    /* link x and y */
    y->right = x;
    if (x != orbit_tree_NIL) x->parent = y;

}



static void orbit_tree_insertFixup(orbit_tree_node_ptr *tree, orbit_tree_node *x) {

   /*************************************
    *  maintain Red-Black tree balance  *
    *  after inserting node x           *
    *************************************/

    /* check Red-Black properties */
    while (x != *tree && x->parent->color == orbit_tree_RED) {
        /* we have a violation */
        if (x->parent == x->parent->parent->left) {
            orbit_tree_node *y = x->parent->parent->right;
            if (y->color == orbit_tree_RED) {

                /* uncle is RED */
                x->parent->color = orbit_tree_BLACK;
                y->color = orbit_tree_BLACK;
                x->parent->parent->color = orbit_tree_RED;
                x = x->parent->parent;
            } else {

                /* uncle is BLACK */
                if (x == x->parent->right) {
                    /* make x a left child */
                    x = x->parent;
                    orbit_tree_rotateLeft(tree, x);
                }

                /* recolor and rotate */
                x->parent->color = orbit_tree_BLACK;
                x->parent->parent->color = orbit_tree_RED;
                orbit_tree_rotateRight(tree, x->parent->parent);
            }
        } else {

            /* mirror image of above code */
            orbit_tree_node *y = x->parent->parent->left;
            if (y->color == orbit_tree_RED) {

                /* uncle is RED */
                x->parent->color = orbit_tree_BLACK;
                y->color = orbit_tree_BLACK;
                x->parent->parent->color = orbit_tree_RED;
                x = x->parent->parent;
            } else {

                /* uncle is BLACK */
                if (x == x->parent->left) {
                    x = x->parent;
                    orbit_tree_rotateRight(tree, x);
                }
                x->parent->color = orbit_tree_BLACK;
                x->parent->parent->color = orbit_tree_RED;
                orbit_tree_rotateLeft(tree, x->parent->parent);
            }
        }
    }
    (**tree).color = orbit_tree_BLACK;
}



void orbit_tree_delete_tree_structure(orbit_tree_ptr tree) {

	/*************************************
    *  Orbit tree structure deletion    *
    *  (iterative tree deletion).       *
    *************************************/

	orbit_tree_node_ptr node, next_node;
	orbit_tree_node_ptr *node_stack;
	unsigned long int stack_top = 0;

	/* Generate an auxiliary stack to parse the orbit tree iteratively */
	node_stack = (orbit_tree_node_ptr*) malloc(sizeof(orbit_tree_node_ptr)*(2+2*owe_lib_log_base_2(1+tree->size)));

	/* Iterative parsing of the tree. */
	if(tree->root != orbit_tree_NIL) {
		node = tree->root->left;
		node_stack[stack_top++] = tree->root;
		while(node != orbit_tree_NIL || stack_top) {
			if(node != orbit_tree_NIL) {
				node_stack[stack_top++] = node;
				node = node->left;
			}
			else
			{
				node = node_stack[--stack_top];
				next_node = node->right;
				
				/* Deletion of orbit. */
				orbit_delete_orbit(node->orbit);
		
				/* Deletion of node in tree. */
				free(node);

				node = next_node;
			}

		}

	}

}