/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *						  OWE General purpose library							   *
 *																				   *
 *								  Source file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "owe_header.h"

/* * * * * * * * * * * * OWE LIB GENERAL FUNCTIONS * * * * * * * * * * * */

int owe_lib_double_to_fraction(double value, int *numerator, int *denominator) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_double_to_fraction
	*  ----------------------
	*  Parameters:
	*		- value: double precision number to convert to fraction.
	*		- numerator:	memory address of the integer where the numerator will be stored.
	*		- denominator:	memory address of the integer where the denominator will be stored.
	*
	*	Return value:
	*		- Integer value that corresponds to the denominator obtained in the fraction.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
	  
    unsigned char isNegative = (unsigned char)(value < 0.0);
    
	unsigned int a;
	double x;

    unsigned int prevPrevN = 0;           
    unsigned int prevPrevD = 1;           
    unsigned int prevN = 1;
    unsigned int prevD = 0;

	value = fabs(value);

    a = (int) floor(value);
    x = value - floor(value);

    *numerator = (a * prevN) + prevPrevN;
    *denominator = (a * prevD) + prevPrevD;

    while (fabs(value - ((double) *numerator / *denominator)) > 1e-6) {
        prevPrevN = prevN;
        prevPrevD = prevD;
        prevD = *denominator;
        prevN = *numerator;
        
        x = 1.0 / x;
        a = (int) floor(x);
        x = x - floor(x);

        *numerator = (a * prevN) + prevPrevN;
        *denominator = (a * prevD) + prevPrevD;
    }

    if (isNegative) *numerator = -*numerator;
	
	return(*denominator);
}



int owe_lib_greatest_common_denominator(int x, int y) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_greatest_common_denominator
	*  -----------------------------------
	*  Parameters:
	*		- Two integer values
	*
	*	Return value:
	*		- An integer value that is the greatest common denominator of the two integer inputs.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
	
	int w;
	while (y != 0) {
		w = x % y;
		x = y;
		y = w;
	}
	return x;
}


int owe_lib_least_common_multiple(int a, int b) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_least_common_multiple
	*  -----------------------------
	*  Parameters:
	*		- Two integer values
	*
	*	Return value:
	*		- An integer value that is the least common multiple of the two integer inputs.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
	
	int x=a, y=b, w;
	while (y != 0) {
		w = x % y;
		x = y;
		y = w;
	}
	return((a*b)/x);
}

int owe_lib_reduce_fraction(int *a, int *b) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_greatest_common_denominator
	*  ---------------------------------
	*  Parameters:
	*		- Memory address of the two integer values that conform the numerator and denominator
	*		  of the fraction.
	*
	*	This function modifies the variables passed as memory address (like reference). It returns
	*	the greatest common denominator used to reduce the fraction.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
	
	int w, x=*a, y=*b;
	while (y != 0) {
		w = x % y;
		x = y;
		y = w;
	}
	*a/=x;
	*b/=x;
	return x;
}



unsigned long int owe_lib_log_base_2(unsigned long int n) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_log_base_2
	*  ------------------
	*  Parameters:
	*		- unsigned integer number from which the log base 2 is desired.
	*
	*	This function computes and return the log base 2 of the number passed as parameter.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned long int log_of_n=0;
	while (n >>= 1) log_of_n++;
	return(log_of_n);

}


/* * * * * * * * * * * * OWE LIB I/O FUNCTIONS * * * * * * * * * * * */




coordinate orbit_print_coordinate(coordinate vertex_coordinate) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  orbit_print_coordinate
	*  ----------------------
	*  Parameters:
	*		- vertex coordinate to be printed
	*
	*	This function displays the coordinates of the vertex/orbit passed as a parameter. Coordinates
	*	are printed on the default standard output stream separated by a single space, followed by "|"
	*	and the vertex denominator.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	unsigned int i;
	for(i=0;i<owe_setting.metric_polytope.dimension;i++) printf("%d ", vertex_coordinate[i]);
	printf("| %d",vertex_coordinate[i]);
	return(vertex_coordinate);
}



void owe_lib_print_time(time_t time) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  owe_lib_print_time
	*  ------------------
	*  Parameters:
	*		- time (seconds) to be displayed.
	*
	*	This function displays the time passed as a parameter in the format:
	*			d day(s) hh:mm:ss   - if the time is greater than 24 hours.
	*			hh:mm:ss			- if the time is less than 24 hours.
	*
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	if(time/86400)
		printf("%lu day(s) %02lu:%02lu:%02lu",time/86400,(time/3600)%24,(time/60)%60,time%60);
	else
			printf("%02lu:%02lu:%02lu",(time/3600)%24,(time/60)%60,time%60);
}
