/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *																			       *
 *							    Processor Queue									   *
 *																				   *
 *								  Header file									   *
 *																				   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include"owe_header.h"

static processor_number* processor_queue_elements;	/* Circular processor queue container. */

static int processor_queue_size;					/* Circular processor queue, size indicator. */

static int processor_queue_first_element;			/* Circular processor queue, first element indicator. */

static int processor_queue_last_element;			/* Circular processor queue, last element indicator. */

static int processor_queue_element_count;

void processor_queue_initialize(unsigned int processors) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  processor_queue_initialize
	*  --------------------------
	*  Parameters:
	*		- processors: number of processors to be used. This number should represent slave processes only.
	*					This is, the master processes should be excluded from this number.
    *
    *	This function initializes the processor queue with all processors (1 to n) in it.
    *
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Set the size of the queue. */
	processor_queue_size = processors;

	/* Set the number of elements in the queue */
	processor_queue_element_count = processors;

	/* The first element of the initialized queue will be in the first position. */
	processor_queue_first_element = 0;

	/* Initially the queue will be filld with all processors 1 to n. */
	processor_queue_last_element = processors - 1;


	/* Allocate memory for the processor queue */
	if(!(processor_queue_elements = (processor_number*) malloc(sizeof(processor_number)*processors))) owe_error_handler_raise_error("memory allocation error (processor_queue_elements variable : processor_queue_initialize function : processor_queue ADT)", 1);

	/* Assign initial values to the queue (processors 1 to n). */
	while(processors) {
		processors--;
		*(processor_queue_elements + processors) = processors + 1;
		}

}


void processor_queue_finalize() {
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  processor_queue_finalize
	*  ------------------------
    *
    *	This function finalizes the processor queue.
    *
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Set the size of the queue to 0 and the first and last element indicators to -1. */
	processor_queue_size = 0;
	processor_queue_first_element = -1;
	processor_queue_last_element = -1;
	processor_queue_element_count = 0;

	/* Free memory taken by the processor queue. */
	free(processor_queue_elements);
}


void processor_queue_insert_processor(processor_number processor) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  processor_queue_insert_processor
	*  --------------------------------
	*  Parameters:
	*		- processor_number: number of the processor that is to be queued.
    *
    *	This function inserts the processor passed in the parameter into the last position of the processor queue.
    *	Note: the number of processors present at any time in the queue should NOT excede the number of processors
	*		  specified in the queue initialization.
    *
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	/* Add the processor to the queue. */
	if(processor_queue_first_element == -1) {
		/* If the queue is empty, add the processor to the first position and set the first and last element indicators accordinly. */
		processor_queue_elements[0] = processor;
		processor_queue_first_element = processor_queue_last_element = 0;
		processor_queue_element_count = 1;
	}
	else
	{
		/* If the queue is not empty, deduce the new position of the processor and insert it. */
		processor_queue_last_element = (processor_queue_last_element + 1) % processor_queue_size;
		processor_queue_elements[processor_queue_last_element] = processor;
		processor_queue_element_count++;
	}


}

processor_number processor_queue_get_next_available_processor(void) {

	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  processor_queue_get_next_available_processor
	*  --------------------------------------------
    *
    *	This function returns the processor number of the next processor in the queue. If the queue is
    *	empty, the value returned is 0. Note that processor numbers have to be between 1 and the total
	*	number of processors, this is, the number 0 is reserved for the master processor which should
	*	not be queued.
    *
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	processor_number next_available_processor;	/* This variable will contain the value of the next available processor. */

	/* If the queue is empty, return 0. */
	if (processor_queue_first_element == -1 && processor_queue_last_element == -1) return 0;

	/* Get the first element in the queue. */
	next_available_processor = processor_queue_elements[processor_queue_first_element];
	
	/* Remove the element from the queue. */
	if(processor_queue_first_element == processor_queue_last_element)
		processor_queue_first_element = processor_queue_last_element = -1;
	else
		processor_queue_first_element = (processor_queue_first_element + 1) % processor_queue_size;

	processor_queue_element_count--;

	return(next_available_processor);

}

unsigned int processor_queue_get_number_of_processors_in_queue(void) {
	
	/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
	*  processor_queue_get_number_of_processors_in_queue
	*  -------------------------------------------------
    *
    *	This function returns the number of processors present in the processor queue.
    *
	* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

	return(processor_queue_element_count);
}
