package interfacing;

import java.awt.BorderLayout;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.Format;
import java.text.SimpleDateFormat;
import java.util.GregorianCalendar;

import javax.swing.JButton;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.JTextField;

import mapping.data.Server;

import adjusting.availability_adjusting.FailureTrace;

/**
 * This is the view class for the FailureTrace class.
 * 
 * @see FailureTrace
 * 
 * @author Majd Kokay
 * 
 */
public class FailuresTraceJPanel extends JPanel implements  ActionListener {

	/**
	 * The serialVersionUID is used to universally identify this version of this
	 * class.
	 */
	private static final long serialVersionUID = 5393004602796008077L;

	/**
	 * The MainFrame container
	 */
	private MainFrame frame;
	/**
	 * The header row labels
	 */
	private String[] headers = { "Start", "End", "Status" };

	/**
	 * A JTable object to view fialureTracesJTable in table form
	 */
	private JTable failureTracesJTable;

	/**
	 * TextField to show the period mean
	 */
	private JTextField periodsMean = new JTextField(15);

	/**
	 * TextField to show the mean time between failures (i.e mean up-time)
	 */
	private JTextField interPeriodsMean = new JTextField(15);

	/**
	 * A formatter to control the way dates are viewed
	 */
	private Format formatter;

	/**
	 * The data failure trace that this class is the view of.
	 */
	private Server server;



	public FailuresTraceJPanel() {
		super();
	}

	/**
	 * A default constructor that takes FailureTrace and build this panel.
	 * 
	 */
	public FailuresTraceJPanel( MainFrame frame, Server server) {
		this.server = server;
		this.frame = frame;
		
		formatter = new SimpleDateFormat("dd/MM/yy '@' HH:mm:ss");
		

		failureTracesJTable = new JTable(server.getFailureTrace().getNumbersOfPeriods(), headers.length);
		failureTracesJTable.setModel(new OurDefaultTableModel(headers, server.getFailureTrace().getNumbersOfPeriods()));
		((OurDefaultTableModel) failureTracesJTable.getModel()).setEditable(false);
		this.refresh();
		JScrollPane scrollPane = new JScrollPane(failureTracesJTable);

		this.setLayout(new BorderLayout());
		this.add(scrollPane, BorderLayout.CENTER);

		JPanel statisticsJPanel = new JPanel();
		statisticsJPanel.setLayout(new BorderLayout());

		JPanel labels = new JPanel();
		labels.setLayout(new GridLayout(2, 2));
		
		labels.add(new JLabel("Mean Failure Period"));
		labels.add(this.periodsMean);

		labels.add(new JLabel("Mean Up-time"));
		labels.add(this.interPeriodsMean);
		
		JButton applyButton = new JButton("Apply");
		applyButton.addActionListener(this);
		
		statisticsJPanel.add(labels, BorderLayout.CENTER);
		statisticsJPanel.add(applyButton, BorderLayout.SOUTH);
		
		this.add(statisticsJPanel, BorderLayout.SOUTH);

	}

	/**
	 * This method refills the JPanel with updates FailureTrace. It should be
	 * invoked every time the FailureTrace object is changed
	 */
	public void refresh() {
		if (server.getFailureTrace() == null) {
			return;
		}
		FailureTrace ft = server.getFailureTrace();

		failureTracesJTable.setModel(new OurDefaultTableModel(headers, server.getFailureTrace().getNumbersOfPeriods()));
		
		for (int i = 0; i < ft.getNumbersOfPeriods(); i++) {
			this.failureTracesJTable.setValueAt(formatter.format(ft.getFailurePeriods().get(i).getBeginningTime()
					.getTime()), i, 0);
			this.failureTracesJTable.setValueAt(formatter.format(ft.getFailurePeriods().get(i).getEndTime().getTime()),
					i, 1);
			this.failureTracesJTable.setValueAt("Failed", i, 2);
		}

		this.periodsMean.setText(ft.getMeanOfPeriodsInMinutes() / frame.getMapper().getTimeUnitInMinutes() +"" );
		this.interPeriodsMean.setText(ft.getMeanIfInterPeriodsInMinutes() / frame.getMapper().getTimeUnitInMinutes()+"");
	}



	public void actionPerformed(ActionEvent e) {
		applyInput();	
	}
	
	
	public void applyInput() {
		if (server.getFailureTrace() == null) {
			System.out.print("NULL");
			return;
		}
		double periodsMean;
		double interPeriodsMean;	
		try {
			 periodsMean = Double.parseDouble( this.periodsMean.getText().trim());
			 interPeriodsMean = Double.parseDouble( this.interPeriodsMean.getText().trim());
		} catch (NumberFormatException ex ) {
			JOptionPane.showMessageDialog(this, "Means should be numbers", "Error",
					JOptionPane.ERROR_MESSAGE);
			return;
		}
		GregorianCalendar startTime = new GregorianCalendar();
		GregorianCalendar endTime = new GregorianCalendar();
		endTime.add(GregorianCalendar.HOUR, 48);
		server.getFailureTrace()
				.fillTrace(startTime, endTime, periodsMean, interPeriodsMean,  this.frame.getMapper().getTimeUnitInMinutes() );
	
	
		refresh();
	}
	
}
