package interfacing;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;

/**
 * This JPanel is responsible for viewing the actual Mue Matrix (Processing
 * Rates) and the Actual Processing Rates in the test.
 * 
 * @author Majd Kokaly
 * 
 */
public class MueJPanel extends JPanel implements MouseListener {
	/**
	 * The serialVersionUID is used to universally identify this version of this
	 * class.
	 */
	private static final long serialVersionUID = -6939332491088413670L;
	
	/**
	 * This is main frame containing all the GUI component. A reference is needed to be
	 * able to access the Mapper object and its data.
	 */
	private MainFrame frame;
	
	/**
	 * The Table containing the Mue Matrix.
	 */
	private JTable jTableForMue;
	
	/**
	 * The Table containing the actual Mue Matrix.
	 */
	private JTable jTableForActualMue;
	
	/**
	 * JScrollPane object for the Mue JTable
	 */
	private JScrollPane scrollPaneForMue;
	
	/**
	 * JScrollPane object for the Actual Mue JTable
	 */
	private JScrollPane scrollPaneForActualMue;

	/**
	 * A SplitPane to seperate the two Tables
	 */
	private JSplitPane splitPane;
	
	/**
	 * This button is used to calculate the acutal Mue
	 */
	private JButton buttonUpdate;

	private JPanel buttonsPanel;

	final String iconsFolder = "Icons";
	
	
	/**
	 * The default constructor. It initializes and lays the GUI objects.
	 * 
	 * @param frame
	 *            this is the MainFrame object that contains this JPanel
	 */
	public MueJPanel(final MainFrame frame) {
		this.frame = frame;

		this.setLayout(new BorderLayout());
		jTableForMue = new JTable();
		jTableForActualMue = new JTable();
		jTableForMue.setSize(1, 1);
		jTableForActualMue.setSize(1, 1);
		scrollPaneForMue = new JScrollPane();
		scrollPaneForActualMue = new JScrollPane();
		scrollPaneForMue.setSize(1, 1);
		scrollPaneForActualMue.setSize(1, 1);
		splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, false, scrollPaneForMue, scrollPaneForActualMue);
		splitPane.setBounds(0, 0, 300, 700);

		buttonsPanel = new JPanel();
		buttonUpdate = new JButton("Update");
		this.buttonUpdate.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				refresh();
			}
		});

		jTableForActualMue.addMouseListener(this);
		jTableForMue.addMouseListener(this);

		buttonsPanel.add(buttonUpdate);
		this.buttonUpdate.setIcon(new ImageIcon(iconsFolder + "/" + "reload_22.png"));

		this.add(buttonsPanel, BorderLayout.PAGE_START);
		this.add(splitPane, BorderLayout.CENTER);
		splitPane.setDividerLocation(0.5);

		jTableForMue.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		jTableForMue.setColumnSelectionAllowed(true);

		jTableForActualMue.getSelectionModel().setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		jTableForActualMue.setColumnSelectionAllowed(true);
	}

	public void repaint() {
		super.repaint();
		if (d != null) {
			scrollPaneForMue.setMaximumSize(d);
			scrollPaneForActualMue.setMaximumSize(d);
		}

	}

	Dimension d;

	/**
	 * This method gets the information about the processing rates and update the tables
	 */
	public void refresh() {
		if (scrollPaneForActualMue == null)
			return;
		d = scrollPaneForActualMue.getSize();
		double mue[][] = this.frame.getMapper().constructMueMatrix();
		jTableForMue.setModel(new NonEditableTableModel(mue.length, mue[0].length));
		

		double actualMue[][] = this.frame.getMapper().constructActualMueMatrix();
		jTableForActualMue.setModel(new NonEditableTableModel(mue.length, mue[0].length));
		for (int i = 0; i < actualMue.length; i++) {
			for (int j = 0; j < actualMue[i].length; j++) {
				this.jTableForActualMue.setValueAt(actualMue[i][j], i, j);
				this.jTableForMue.setValueAt(mue[i][j], i, j);
			}
		}

		for (int j = 0; j <  actualMue[0].length; j++) {
			jTableForActualMue.getColumnModel().getColumn(j).setHeaderValue((j+1)+"");
			jTableForMue.getColumnModel().getColumn(j).setHeaderValue((j+1)+"");
		}
		
		scrollPaneForMue.setViewportView(jTableForMue);
		scrollPaneForActualMue.setViewportView(jTableForActualMue);
		splitPane.setDividerLocation(0.5);
	}
	
	/**
	 * When a cell in a Table is clicked the corresponding cell in the other table is highlighted. This method is responsible for that.
	 */
	public void mouseClicked(MouseEvent e) {
		if (e.getSource() == this.jTableForActualMue) {
			int i = jTableForActualMue.getSelectedRow();
			int j = jTableForActualMue.getSelectedColumn();

			jTableForMue.getSelectionModel().setSelectionInterval(i, i);
			jTableForMue.getColumnModel().getSelectionModel().setSelectionInterval(j, j);
		}

		if (e.getSource() == this.jTableForMue) {
			int i = jTableForMue.getSelectedRow();
			int j = jTableForMue.getSelectedColumn();

			jTableForActualMue.getSelectionModel().setSelectionInterval(i, i);
			jTableForActualMue.getColumnModel().getSelectionModel().setSelectionInterval(j, j);

		}
	}

	public void mouseEntered(MouseEvent e) {

	}

	public void mouseExited(MouseEvent e) {

	}

	public void mousePressed(MouseEvent e) {

	}

	public void mouseReleased(MouseEvent e) {

	}

}
