package logging;

import java.io.Serializable;
import java.text.DateFormat;
import java.util.GregorianCalendar;

/**
 * This class resembles an Event that can be logged (e.g. Job generation, job completion  or error)
 * 
 * @author Majd Kokaly
 *
 */
public class Event implements Serializable {
	/**
	 * The serialVersionUID is used to universally identify this version of this class. 
	 */
	private static final long serialVersionUID = 998748623574865211L;
	
	/**
	 * This string contains the description of the event 
	 */
	private String description;
	
	/**
	 * This calendar or time object stores the timestamp when the event happened
	 */
	private java.util.GregorianCalendar timestamp;
	
	/**
	 * When assigned, the id is the job id for the job related to that event
	 */
	private long id;
	
	/**
	 * Used to initialize id when the job is not known or defined yet
	 */
	private static final long NOT_DEFINED_YET = -5;
	
	/**
	 * Formatter used to format dates
	 */
	private final static DateFormat FORMATTER =  DateFormat.getDateTimeInstance();
	
	public Event(String description, long id) {
		super();
		this.description = description;
		this.setTimestamp(new GregorianCalendar());
		this.setId(id);
	}
	
	public Event(String description, long jobId, GregorianCalendar timeStamp) {
		super();
		this.setDescription(description);
		this.setTimestamp(timeStamp);
		this.setId(jobId);
	}
	
	public Event(String description, GregorianCalendar timeStamp) {
		super();
		this.setTimestamp(timeStamp);
		this.setId(NOT_DEFINED_YET);
		this.setDescription(description);
	}
	
	public Event(String description) {
		super();
		this.setTimestamp(new GregorianCalendar());
		this.setId(NOT_DEFINED_YET);
		this.setDescription(description);
	}
	
	
	public String getDescription() {
		return description;
	}
	public void setDescription(String description) {
		this.description = description;
	}
	public java.util.GregorianCalendar getTimestamp() {
		return timestamp;
	}
	public void setTimestamp(java.util.GregorianCalendar timestamp) {
		this.timestamp = timestamp;
	}
	public long getId() {
		return id;
	}
	public void setId(long id) {
		this.id = id;
	}
	public String toString() {
		if (this.getId() != Event.NOT_DEFINED_YET)
			return "Job\t" + this.getId() + "\t" + this.getDescription() + "\t@\t" + FORMATTER.format(this.getTimestamp().getTime());
		else
			return this.getDescription() + " \t@\t" + FORMATTER.format(this.getTimestamp().getTime());
	}
	
	
	
}
