package mapping;

import mapping.data.Job;

/**
 * To add a mapping scheme, A concrete class that extend this abstract class,
 * thus it must implement the methods of this class.
 * 
 * This class determines what services should the concrete class provides to the
 * Mapper objects.
 * 
 * @author Majd Kokaly
 * 
 */
public abstract class MappingScheme {

	/**
	 * A description of the Mapping Scheme. (e.g. LPAS_DG)
	 */
	private String description;

	/**
	 * The Mapper object of this mapping scheme
	 */
	private Mapper mapper;

	/**
	 * A default constructor
	 * 
	 * @param mapper
	 *            to set the mapper field.
	 */
	public MappingScheme(Mapper mapper) {
		this.setMapper(mapper);
	}

	public String getDescription() {
		return description;
	}

	public void setDescription(String description) {
		this.description = description;
	}

	public Mapper getMapper() {
		return mapper;
	}

	public void setMapper(Mapper mapper) {
		this.mapper = mapper;
	}

	/**
	 * This method is invoked after the servers and the job classes are defined.
	 * It is invoked before the start of the operation. In this method,
	 * initialization operations are defined. For instance, if the mapping
	 * scheme depends on a thread, the thread is initialized and started.
	 * 
	 */
	public abstract void startMappingScheme();

	/**
	 * In this method, the program- mer should define operations that stop the
	 * execution of the system processes (e.g mapping and generation). This can
	 * be useful if the system execution is wished to be restarted after
	 * stopping it.
	 */
	public abstract void stopMappingScheme();

	/**
	 * The implementation of this method determines the mapping scheme. It is
	 * invoked by the mapper object. The job is sent to the active MappingScheme
	 * concrete object (e.g. the LPAS DG object or the Gcu object). This object
	 * then makes the mapping decision according to appropriate policy.
	 * 
	 * @param job
	 *            the job object to be mapped.
	 */
	protected abstract void mapJob(Job job);

	/**
	 * The implementation of this method determines what should happen a job
	 * times out. One way of handling a time-out for instance, is to resubmit
	 * the timed-out job. This way was chosen for the implemented schemes.
	 * 
	 * @param jobID
	 */
	public abstract void handleJobTimeOut(long jobID);
	
	/**
	 * The implementation of this method determines what should happen a server
	 * goes down. For instance, the LPAS_DG policy resolves the LP allocation.
	 * 
	 * @param serverID the serverID of the server that became unavailable
	 */
	public abstract void serverIsDown(int serverID);
	
	/**
	 * The implementation of this method determines what should happen a server
	 * becomes up. For instance, the LPAS_DG policy resolves the LP allocation.
	 * 
	 * @param serverID the serverID of the server that became unavailable
	 */
	public abstract void serverIsUp(int serverID);
}
