package mapping.data;

import java.util.GregorianCalendar;

/**
 * This interface defines the methods that a lister to the job table should
 * implement.
 * 
 * @author Majd Kokaly
 * 
 */
public interface JobsTableListener {
	/**
	 * This method is invoked when a table is added to the table
	 * 
	 * @param jobID
	 *            The job id.
	 * @param jobSecondaryID
	 *            The job secondary id.
	 * @param jobClassID
	 *            The ID of the job class that this job belongs to
	 * @param serverName
	 *            The hostname of the server of this job.
	 * @param timeSubmitted
	 *            The time of submission for this job.
	 * @param timeSent
	 *            The time of sending for this job.
	 * @param timeStarted
	 *            The time of start for this job.
	 * @param timeDone
	 *            The time of completion for this job.
	 * @param status
	 *            The status of this job
	 * @param timedOut
	 *            If this job is timed out or not.
	 */
	public void jobHasBeenAdded(long jobID, long jobSecondaryID, int jobClassID, String serverName,
			GregorianCalendar timeSubmitted, GregorianCalendar timeSent, GregorianCalendar timeStarted,
			GregorianCalendar timeDone, String status, boolean timedOut);

	/**
	 * The method is invoked when the secondary ID of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param jobSecondayID
	 *            The new secondary ID of the job in subject.
	 */
	public void secondaryIDHasBeenChanged(long jobID, long jobSecondayID);

	/**
	 * The method is invoked when the job class ID of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param jobClassId
	 *            the new jobClassID.
	 */
	public void jobClassIDHasBeenChanged(long jobID, int jobClassId);

	/**
	 * The method is invoked when the server of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param serverName
	 *            the new server name of the job is subject
	 */
	public void serverNameHasBeenChanged(long jobID, String serverName);

	/**
	 * The method is invoked when the server of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param ID
	 *            the new server ID of the job is subject.
	 */
	public void serverIDHasBeenChanged(long jobID, int serverID);

	
	/**
	 * The method is invoked when the timeSubmitted of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param timeSubmitted
	 *            the new timeSubmitted of the job in subject.
	 */
	public void timeSubmittedHasBeenChanged(long jobID, GregorianCalendar timeSubmitted);

	/**
	 * The method is invoked when the timeStarted of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param timeSent
	 *            the new timeSubmitted of the job in subject.
	 */
	public void timeSentHasBeenChanged(long jobID, GregorianCalendar timeSent);

	/**
	 * The method is invoked when the timeStarted of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param timeStarted
	 *            the new timeStarted of the job in subject.
	 */
	public void timeStartedHasBeenChanged(long jobID, GregorianCalendar timeStarted);

	/**
	 * The method is invoked when the timeDone of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param timeDone
	 *            the new timeDone of the job in subject.
	 */
	public void timeDoneHasBeenChanged(long jobID, GregorianCalendar timeDone);

	/**
	 * The method is invoked when the status of a job is changed.
	 * 
	 * @param jobID
	 *            The ID of the job in subject.
	 * @param status
	 *            the new status of the job in subject.
	 */
	public void statusHasBeenChanged(long jobID, String status);

	/**
	 * The method is invoked when a job is times-out.
	 * 
	 * @param jobID
	 *            the ID of the job in subject.
	 * @param timedOut
	 *            a boolean value. True if timed-out, false otherwise.
	 */
	public void timedOutStatusChanged(long jobID, boolean timedOut);

}