%%%  src/INet/InetsFile/Abstract.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{|INet.InetsFile.Abstract| --- Abstract Syntax of \texttt{.inet} Files}

The \Inets{} project (source code at \url{https://gna.org/svn/?group=inets})
uses a \texttt{.inet} file format as input.

(Variadic ranges / arrays seem to be almost unused in the \texttt{.inet} files available there.)

The following datatypes have been quickly assembled
from reading |src/inets/syntax/Parser.jjt|.

\begin{ModuleHead}
\begin{code}
module INet.InetsFile.Abstract where

import qualified Data.Char as Char
\end{code}
\end{ModuleHead}

File output of this format will, if needed, be done using a pretty-printing library;
we add ``|deriving (Show)|'' to the data type definitions here
only for testing and debugging purposes.

\begin{code}
type CompilationUnit = Either [Module] [ModuleComponent]
\end{code}


\begin{code}
data Module = Module
  { moduleName :: String
  , moduleComponents :: [ModuleComponent]
  }
  deriving (Show)
\end{code}


\begin{code}
cuComponents :: CompilationUnit -> [ModuleComponent]
cuComponents (Left ms) = concatMap moduleComponents ms
cuComponents (Right mcs) = mcs
\end{code}

\begin{code}
type Name = String
\end{code}

\begin{code}
data ModuleComponent
  = MCImportStmt [Name] -- dot-separated non-empty file path
  | MCRule Rule
  | MCNet Net
  | MCStatements [StatementOrDec]
  deriving (Show)
\end{code}

\begin{code}
mcImports :: ModuleComponent -> [[Name]]
mcImports (MCImportStmt p) = [p]
mcImports _ = []
\end{code}


\begin{code}
mcRules :: ModuleComponent -> [Rule]
mcRules (MCRule rule) = [rule]
mcRules _ = []
\end{code}


\begin{code}
cuRules :: CompilationUnit -> [Rule]
cuRules = concatMap mcRules . cuComponents
\end{code}

\begin{code}
data Rule
  = AgentRule
    { ruleDef :: RuleDef
    }
  | NamedRule
    { ruleName :: Name
    , ruleMoreNames :: [Name] -- \edcomm{WK}{What are these for?}
    , ruleDef :: RuleDef
    }
  deriving (Show)
\end{code}

\begin{code}
cuRuleDefs :: CompilationUnit -> [RuleDef]
cuRuleDefs = map ruleDef . cuRules
\end{code}

\begin{code}
data RuleDef = RuleDef
  { ruleLHS :: Agent
  , ruleMatchStmts :: [StatementOrDec]
  , ruleRHSs :: [RHS]
  }
  deriving (Show)

ruleFuncName :: RuleDef -> Name
ruleFuncName (RuleDef (Agent name _) _ _) = name
\end{code}

In a |RuleDef|,
the principal port of the agent before the |><| is taken to have
negative polarity, while the principal ports of agents after |><| but before |==>|,
which include nested patterns, are taken to be constructors, that is,
have positive polarity.
The agents of the replacing net, that is, after |==>|,
need their polarity derived from the incident connections.

\begin{code}
ruleDefAgents :: RuleDef -> (Agent, [(Agent,[Agent])])
ruleDefAgents (RuleDef lhs matchStmts rhss) = (lhs, map (rhsAgents . ruleRHS) rhss)
  where
    rhsAgents (RuleRHS pat stmts body) = (pat, bodyAgents body)
    bodyAgents (RBeq eqs) = concatMap eqAgents eqs
    bodyAgents (RBif ifstmt) = concatMap eqAgents $ ifEqs ifstmt
    bodyAgents RBskip = []
    ifEqs (IF b t e) = ifBlockEqs t ++ either ifBlockEqs ifEqs e
    ifBlockEqs (IfBlock pre eqs post) = eqs
\end{code}

\begin{code}
data RHS = RhsBlock
  { rhsPre :: [StatementOrDec]
  , ruleRHS :: RuleRHS
  , rhsPost :: [StatementOrDec]
  }
  deriving (Show)
   -- don't emit braces for empty statements both before and after.
\end{code}

\begin{code}
data RuleRHS = RuleRHS
  { rhsPattern :: Agent
  , rhsStmts :: [StatementOrDec]
  , rhsBody :: RuleBody
  }
  deriving (Show)

data RuleBody = RBeq [Equation]  -- non-empty
              | RBif IfStatement
              | RBskip
  deriving (Show)
\end{code}

\begin{code}
data IfStatement = IF Expression IfBlock ElsePart
  deriving (Show)

type ElsePart = Either IfBlock IfStatement

data IfBlock = IfBlock  [StatementOrDec] [Equation] [StatementOrDec]
  deriving (Show)
\end{code}

\begin{code}
data Net
 = UnNamedNet [NetDef]
 | NamedNet Name ParamOrArgs [Either StatementOrDec [NetDef]]
  deriving (Show)

type NetDef = Either Equation NetInst

data NetInst = NetInst Name ParamOrArgs
  deriving (Show)
\end{code}

\begin{code}
data Equation = Equation Term Term
  deriving (Show)

eqAgents :: Equation -> [Agent]
eqAgents (Equation t1 t2) = [termAgent t1, termAgent t2]

data Term
  = TermAgent     {termAgent :: Agent}   -- uppercase |AGENTNAME|
  | TermVar       {termAgent :: Agent}   -- lowercase |VARNAME|
  | TermVariadic  {termAgent :: Agent}   -- |VARIADICNAME| starting with |"'"| (not part of |Agent| name)
  deriving (Show)

data Agent = Agent Name ParamOrArgs
  deriving (Show)
\end{code}

Only uppercase names make ``real'' agents;
lowercase names are connection variables, called ``|varAgent|'' in the \Inets{} source.
\begin{code}
isVarName :: Name -> Bool
isVarName (c : _) = Char.isLower c
isVarName [] = error "isVarName: unexpected empty name"

isVarAgent :: Agent -> Bool
isVarAgent (Agent name _) = isVarName name
\end{code}

\begin{code}
type ParamOrArgs = [ParamOrArg]
data ParamOrArg
  = Param PrimitiveType Agent
  | ParamArray Name Int
  | Arg ExpressionOrTerm
  deriving (Show)
\end{code}

\begin{code}
data ExpressionOrTerm
  = EAgent Agent
  | EVar Agent
  | Evariadic Agent
  | Eexpr Expression
  deriving (Show)
\end{code}

\begin{code}
data StatementOrDec
  = Return (Maybe Term)
  | StatementExpression ETerm Expression -- |LhsExp() "=" RhsExp()|
  | Declaration PrimitiveType [Assignment] -- non-empty
  | Print [ExpOrString] -- non-empty
  | PrintNet [Term] -- non-empty
  deriving (Show)

data Assignment = Assignment Name (Maybe Expression)
  deriving (Show)

type ExpOrString = Either Expression String
\end{code}

\begin{code}
data PrimitiveType = Tint | Tfloat | Tchar | Tbool | TString | TAgent | TAgents
  deriving (Show)
\end{code}

\begin{code}
data Expression
  = Atom ETerm [(Pred , ETerm)]
  | Or Expression Expression
  | And Expression Expression
  deriving (Show)
\end{code}

\begin{code}
data Pred = Peq | Pneq | Plt | Pgt | Pleq | Pgeq
  deriving (Show)

interpPred :: Ord a => Pred -> a -> a -> Bool
interpPred Peq = (==)
interpPred Pneq = (/=)
interpPred Plt = (<)
interpPred Pgt = (>)
interpPred Pleq = (<=)
interpPred Pgeq = (>=)
\end{code}

\begin{code}
data BinOp = Bplus | Bminus | Bmult | Bdiv | Bmod
  deriving (Show)
data UnOp = Uminus | Unot
  deriving (Show)
\end{code}

\begin{code}
data ETerm
  = Bin BinOp ETerm ETerm
  | Un UnOp ETerm
  | Var Name
  | BoolLit Bool
  | IntLit Integer
  | StringLit String
  deriving (Show)
\end{code}

The ``|PrimaryPrefix()|'' |readc| occurs only in |test/examples/hanoi.inet|.



% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
