\section{|INet.InetsFile.Interpret| --- Interpreter for \texttt{.inet} Files}

\begin{ModuleHead}
\begin{code}
module INet.InetsFile.Interpret where

import INet.InetsFile.Abstract
import INet.InetsFile.Parser
import INet.InetsFile.ToDescription

import INet.Description
import INet.Description.Check (checkNetDescription)
import INet.PTerm

import INet.Polar.INet
import INet.Polar.Create
import INet.Polar.Read

import INet.Utils.Usage
import qualified INet.Utils.Vector as V

import Data.Maybe (mapMaybe)
import Data.Map (Map)
import qualified Data.Map as Map

import Control.Monad.State

import System.Environment (getArgs)
import System.IO
import System.Exit
import System.FilePath
import System.Directory
\end{code}
\end{ModuleHead}

\begin{code}
runInets = do
  args <- getArgs
  case args of
    [s] -> runFile s
    _ -> hPutStrLn stderr $ usage "RunInets" "<file>.inet"
\end{code}

\begin{code}
locateImport :: FilePath -> FilePath -> IO (Maybe FilePath)
locateImport baseFile importFile = do
    b <- doesFileExist importFile
    if b
       then return (Just importFile)
       else do
         cwd <- getCurrentDirectory
         findFile [cwd, baseDir, baseDir2] importFile
  where
    baseDir = takeDirectory baseFile
    baseDir2 = takeDirectory baseDir
\end{code}

\begin{code}
readInetsFile :: FilePath -> IO CompilationUnit
readInetsFile path = do
  e <- parseInetsFile path
  case e of
    Left err -> do  hPutStrLn stderr $ "RunInets: Parse error: " ++ show err
                    exitFailure
    Right cu -> return cu
\end{code}

\begin{code}
expandImportsCU :: FilePath -> CompilationUnit -> IO CompilationUnit
expandImportsCU baseFile (Left ms) = liftM Left $ mapM (expandImportsM baseFile) ms
expandImportsCU baseFile (Right mcs) = liftM Right $ expandImportsMCs baseFile mcs

expandImportsMCs :: FilePath ->  [ModuleComponent] -> IO [ModuleComponent]
expandImportsMCs baseFile = liftM concat . mapM (expandImportsMC baseFile)

expandImportsM :: FilePath -> Module -> IO Module
expandImportsM baseFile m = do
  mcs' <- expandImportsMCs baseFile $ moduleComponents m
  return $ m { moduleComponents = mcs' }
\end{code}

We implement na\"{\i}ve and restricted import chasing:
Besides from the current directory,
we also start from the directory containing the importing file,
and its parent directory (if the importing file had a more-than-two-component file path).
This is sufficient for the examples in the \Inets{} repository
as of r65 (2012-03-20, still current on 2015-02-10).
\begin{code}
expandImportsMC :: FilePath -> ModuleComponent -> IO [ModuleComponent]
expandImportsMC baseFile (MCImportStmt ss) = do
  let importPath0 = joinPath ss <.> "inet"
  importPathM <- locateImport baseFile importPath0
  case importPathM of
    Nothing -> do
      hPutStrLn stderr $ unwords[baseFile ++ ": Import", importPath0, "not found"]
      exitFailure
    Just importPath -> do
      cu <- readInetsFile importPath
      case cu of
        Left ms -> do
          ms' <- mapM (expandImportsM baseFile) ms
          return $ concatMap moduleComponents ms -- have to flatten
        Right mcs -> expandImportsMCs baseFile mcs
expandImportsMC baseFile mc = return [mc]
\end{code}

\begin{code}
runFile path = do
  cu <- readInetsFile path
  expandImportsCU path cu >>= runCompilationUnit
\end{code}

\begin{code}
filterCU :: CompilationUnit -> [Either Net [Term]]
filterCU (Left ms) = filterMs ms
filterCU (Right mcs) = filterMCs mcs

filterMCs :: [ModuleComponent] -> [Either Net [Term]]
filterMCs (MCNet net : mcs) = Left net : filterMCs mcs
filterMCs (MCStatements stmts : mcs) = map Right (mapMaybe filterStmt stmts) ++ filterMCs mcs
filterMCs (_ : mcs) = filterMCs mcs
filterMCs [] = []

filterMs = concatMap (filterMCs . moduleComponents)

filterStmt :: StatementOrDec -> Maybe [Term]
-- only |PrintNet| is interesting:
filterStmt (PrintNet ts) = Just ts
filterStmt _ = Nothing
\end{code}

\begin{code}
filterNamedNetBody :: [Either StatementOrDec [NetDef]] -> [Either Net [Term]]
filterNamedNetBody (Left stmt : es) = case filterStmt stmt of
  Nothing -> filterNamedNetBody es
  Just ts -> Right ts : filterNamedNetBody es
filterNamedNetBody (Right nds : es) = Left (UnNamedNet nds) : filterNamedNetBody es
filterNamedNetBody [] = []
\end{code}

\begin{code}
filterNetDefs :: [NetDef] -> Either String [Equation]
filterNetDefs (Left eq : nds) = fmap (eq :) $ filterNetDefs nds
filterNetDefs (Right (NetInst name pas) : nds) = Left $
   "Net instances not handled yet --- found: " ++ name ++ show pas
filterNetDefs [] = Right []
\end{code}

\begin{code}
data InterpState = InterpState
  { iLang :: INetLang (NLab Value)
  , iSplitAttribsMap :: SplitAttribsMap
  , iPortVars :: Map Name (Port (NLab Value))
  , iNamedNets :: Map Name (PTerm (NLab Value) Name)
  }

modifyPortVars f = modify (\ s -> s { iPortVars = f $ iPortVars s})
\end{code}

\begin{code}
evalExpr :: String -> Expression -> Value
evalExpr msg e = case evalExpression val e of
    Left err -> error $ unwords [msg, show e, ":", err]
    Right v -> v
  where
    val :: Map Name Value
    val = Map.empty          -- we are ignoring any global declarations for now.
\end{code}

\begin{code}
interpret :: [Either Net [Term]] -> StateT InterpState IO ()
interpret [] = return ()
interpret (Left (NamedNet name@"main" [] body) : qs) = do
  lift $ putStrLn $ "Entering " ++ name
  interpret $ filterNamedNetBody body
  lift $ putStrLn $ "Leaving " ++ name
  interpret qs
interpret (Left (NamedNet name [] [Left (Return (Just t))]) : qs) = do
  splitAttribsMap <- liftM iSplitAttribsMap get
  let  pt :: PTerm (NLab Value) Name
       pt = mapPTermNLab (fmap (evalExpr $ "interpret: NamedNet " ++ name ++ ":"))
              $ toPTerm ("NamedNet: " ++ name ++ " = " ++ show t) splitAttribsMap t
  modify (\ s -> s {iNamedNets = Map.insert name pt $ iNamedNets s})
  interpret qs
interpret (Left (NamedNet name pas body) : _) = do -- \edcomm{WK}{See \textsf{ArithExpression.inet}.}
  lift $ do  hPutStrLn stderr $ "General named nets not handled yet --- found: " ++ name ++ show pas
             exitFailure
interpret (Left (UnNamedNet nds) : qs) = case filterNetDefs nds of
  Left err -> lift $ hPutStrLn stderr err
  Right eqs -> do
    lang <- liftM iLang get
    splitAttribsMap <- liftM iSplitAttribsMap get
    namedNets <- liftM iNamedNets get
    let  ptEqs0 = map (toPTermEq (show eqs) splitAttribsMap) eqs
         ptEqs1 = map (mapPTermEqNLab (fmap $ evalExpr "interpret: UnNamedNet")) ptEqs0
         ptEqs = closedSubstPTermEqs newStringVar namedNets ptEqs1
    -- |lift $ putStrLn $ unlines $ "UnNamedNet equations:" : map (("  " ++) . show) ptEqs|
    case buildNet ptEqs of
      (sourceVars, ndV) -> case checkNetDescription ndV of
          [] -> do
            -- |lift $ putStrLn "No clashes"|
            -- |lift $ putStrLn $ show sourceVars|
            -- |lift $ putStrLn $ show ndV|
            (src, _trg) <- lift $ createNet lang ndV
            modifyPortVars (foldr ($) `flip` zipWith Map.insert sourceVars (V.toList src))
            interpret qs
          clashes -> lift $ do
            putStrLn $ unlines $
              "\ninterpret: Inconsistent net description:"
              : show sourceVars
              : show ndV : "Clashes:" : map (("  " ++) . show) clashes
            exitFailure
interpret (Right ts : qs) = do -- this is ``|printNet ts|''
    lang <- liftM iLang get
    splitAttribsMap <- liftM iSplitAttribsMap get
    portVars <- liftM iPortVars get
    let  pts = map (toPTerm "printNet" splitAttribsMap) ts
         printPTerm (ConnVar v) = do
           putStr $ "  " ++ v ++ " --> "
           case Map.lookup v portVars of
             Nothing -> putStrLn "undefined!"
             Just p -> do
               r <- getPTerm p
               putStrLn $ showsPTerm shows (error "showsVar") r ""
         printPTerm pt = putStrLn $ "printNet: variable expected: " ++ show pt
    lift $ mapM_ printPTerm pts
    interpret qs
\end{code}

\begin{code}
runCompilationUnit :: CompilationUnit -> IO ()
runCompilationUnit cu = let
    ((symbLangMap, polMap), (splitAttribsMap, lang)) = cuINetLang cu
  in evalStateT  (interpret $ filterCU cu)
                 (InterpState lang splitAttribsMap Map.empty Map.empty)
\end{code}
