%%%  src/INet/PTerm.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{|INet.PTerm| --- ``Principality Terms''}

\begin{ModuleHead}
\begin{code}
{-# LANGUAGE ScopedTypeVariables, FlexibleContexts, PatternGuards #-}
module INet.PTerm where

import INet.Description
import INet.Description.Utils

import INet.Utils.List (removeDuplicates)
import INet.Utils.Vector (Vector)
import qualified INet.Utils.Vector as V

import Data.Map (Map)
import qualified Data.Map as Map

import Data.List (intersperse, sortBy)
import Data.Function (on)

import Control.Monad.State
import Control.Applicative

-- |import Debug.Trace|
\end{code}
\end{ModuleHead}

|PTerm| is the type of Lafont-style terms
denoting parts of interaction nets,
where the orientation of the term constructors
is ``principal port up''.

\begin{code}
data PTerm nlab var
  = ConnVar var
  | PNode nlab [PTerm nlab var]
\end{code}

An ``equation'' |(pt1, pt2)| denotes the net
created by connecting the two trees |pt1| and |pt2|
at the principal ports of their roots.

\begin{code}
type PTermEq nLab var = (PTerm nLab var, PTerm nLab var)
\end{code}

\begin{code}
instance Functor (PTerm nlab) where
  fmap f (ConnVar v) = ConnVar $ f v
  fmap f (PNode nL pts) = PNode nL $ map (fmap f) pts
\end{code}

\begin{code}
mapPTermNLab :: (a -> b) -> PTerm a var -> PTerm b var
mapPTermNLab f (ConnVar v) = ConnVar v
mapPTermNLab f (PNode nL pts) = PNode (f nL) $ map (mapPTermNLab f) pts
\end{code}

\begin{code}
mapPTermEqNLab :: (a -> b) -> PTermEq a var -> PTermEq b var
mapPTermEqNLab f (t1, t2) = (mapPTermNLab f t1, mapPTermNLab f t2)
\end{code}

\begin{code}
showsPTerm :: (nlab -> ShowS) -> (var -> ShowS) -> PTerm nlab var -> ShowS
showsPTerm showsNLab showsVar (ConnVar v) = showsVar v
showsPTerm showsNLab showsVar (PNode nL []) = showsNLab nL
showsPTerm showsNLab showsVar (PNode nL ts) = showsNLab nL . ('(' :) .
  (foldr ($) `flip` intersperse (',' :) (map (showsPTerm showsNLab showsVar) ts))
  . (')' :)

instance (Show var, Show nlab) => Show (PTerm nlab var) where
  showsPrec _ = showsPTerm shows shows
\end{code}

\begin{code}
connVars :: PTerm nlab var -> [var]
connVars t = h t []
  where
    h (ConnVar v) = (v :)
    h (PNode _ pts) = foldr h `flip` pts
\end{code}

\begin{code}
connVarsEq :: PTermEq nlab var -> [var]
connVarsEq (t1, t2) = connVars t1 ++ connVars t2

connVarsEqs :: [PTermEq nlab var] -> [var]
connVarsEqs = concatMap connVarsEq
\end{code}

\edcomm{WK}{Raw substitution is actually never needed --- see |closedSubstPTermEqs| below.
Commented out:
\begin{spec}
substPTerm :: (Ord var) => Map var (PTerm nLab var) -> PTerm nLab var -> PTerm nLab var
substPTerm m t@(ConnVar v) = case Map.lookup v m of
  Nothing -> t
  Just t' -> t'
substPTerm m (PNode nL pts) = PNode nL $ map (substPTerm m) pts

substPTermEq :: (Ord var) => Map var (PTerm nLab var) -> PTermEq nLab var -> PTermEq nLab var
substPTermEq m (t1, t2) = (substPTerm m t1, substPTerm m t2)
\end{spec}
}%edcomm

|mkVarRenaming newVar avoid vs| returns a mapping of all |vs| occurring in |avoid| to different new variables.

\begin{code}
mkVarRenaming :: (Ord var) => ([var] -> var -> var) -> [var] -> [var] -> Map var var
mkVarRenaming newVar avoid = h avoid
  where
    h av [] = Map.empty
    h av (v : vs) = let v' = newVar av v
                    in (if v' == v then id else Map.insert v v') (h (v' : avoid) vs)

varRenaming :: (Ord var) => ([var] -> var -> var) -> [var] -> [var] -> var -> var
varRenaming newVar avoid vs v = Map.findWithDefault v v m
  where
    m = mkVarRenaming newVar avoid vs

newStringVar :: [String] -> String -> String
newStringVar avoid v = if v `notElem` avoid then v
  else head $ filter (`notElem` avoid) $ map ((v ++) . ('_' :) . show) [1..]
\end{code}

When expanding named nets,
their local variables need to be renamed to avoid clashes,
with a different renaming for each occurrence.

\begin{code}
closedSubstPTermEqs  :: forall var nLab . (Ord var)
                     => ([var] -> var -> var)
                     -> Map var (PTerm nLab var)
                     -> [PTermEq nLab var]
                     -> [PTermEq nLab var]
closedSubstPTermEqs newVar m eqs = let
    avoid = connVarsEqs eqs

    substPTerm :: PTerm nLab var -> State [var] (PTerm nLab var)
    substPTerm t@(ConnVar v) = case Map.lookup v m of
        Nothing -> return t
        Just t1 -> do
          avoid <- get
          let  ren = varRenaming newVar avoid $ connVars t1
               t1' = fmap ren t1
          modify (connVars t1' ++)
          return t1'
    substPTerm (PNode nL pts) = fmap (PNode nL) $ mapM substPTerm pts

    substPTermEq :: PTermEq nLab var -> State [var] (PTermEq nLab var)
    substPTermEq (t1, t2) = (,) <$> substPTerm t1 <*> substPTerm t2

    substPTermEqs :: [PTermEq nLab var] -> State [var] [PTermEq nLab var]
    substPTermEqs = mapM substPTermEq

  in evalState (substPTermEqs eqs) avoid
\end{code}

In the generalised |portDescriptions| for each node,
we keep the principal port separate from the auxiliary ports,
since target information for the principal port may become available only much later.
\begin{code}
data BuildState nLab var = BuildState
  { nodeCount :: Int
  , nodeInfo :: Map NI (nLab , (Maybe (Either var PortTargetDescription), [Either var PortTargetDescription]))
  , varMap :: Map var PortTargetDescription
  }
\end{code}


\begin{code}
buildPTerm  :: (MonadState (BuildState nLab var) m, Ord var)
            => PTerm nLab var -> m (Either var PortTargetDescription)
buildPTerm pt = buildPTermInto pt Nothing

buildPTermInto  :: (MonadState (BuildState nLab var) m, Ord var)
                => PTerm nLab var
                -> Maybe (Either var PortTargetDescription)
                -> m (Either var PortTargetDescription)
buildPTermInto (ConnVar v) mvptd = do
        vm <- liftM varMap get
        case mvptd of
          Just (Right ptd) -> modify (\ s -> s {varMap = Map.insert v ptd vm})
          _ -> return ()
        case Map.lookup v vm of -- in state before!
          Nothing -> return $ Left v
          Just ptd' -> return $ Right ptd'
buildPTermInto (PNode nlab succs) mvptd = do
    ni <- liftM nodeCount get
    modify (\ s -> s {nodeCount = succ ni})
    succInfo <- sequence $ zipWith f succs $ map (InternalPort ni) [1..]
    modify (\ s -> s {nodeInfo = Map.insert ni (nlab, (mvptd, succInfo)) (nodeInfo s)})
    let result = Right $ InternalPort ni 0
    return result
  where f pt iptd = buildPTermInto pt (Just $ Right iptd)
\end{code}



\begin{code}
buildEquation  :: (MonadState (BuildState nLab var) m, Ord var, Show var, Show nLab)
               => PTermEq nLab var
               -> m ()
buildEquation (ptL@(PNode _ _), ptR@(ConnVar _))
  = buildEquation (ptR, ptL)  -- \edcomm{WK}{This avoids the current problem with this pattern.}
buildEquation (ptL, ptR) = do
  vptdL <- buildPTerm ptL
  vptdR <- buildPTermInto ptR (Just vptdL)
  case (ptL, vptdL, vptdR) of
    (ConnVar v, _, Right ptdR) -> do
       vm <- liftM varMap get
       modify (\ s -> s {varMap = Map.insert v ptdR vm})
    (_, Right (InternalPort ni' 0), Right ptdR) -> do -- can only be the principal port
       nodeInfo1 <- liftM nodeInfo get
       let (nl', (ppInfo', auxInfo')) = nodeInfo1 Map.! ni'
       modify (\ s -> s {nodeInfo = Map.insert ni' (nl', (Just vptdR, auxInfo')) nodeInfo1})
    _ -> return ()

  -- |trace (unwords ["\nbuildEquation", show ptL, " ~ ", show ptR, " --> ", show [vptdL, vptdR]]) $|
  return ()
\end{code}

\begin{code}
buildOrdinaryRHS  :: (Ord var, Show var, Show nLab)
                  => Map var PortTargetDescription
                  -> [PTermEq nLab var]
                  -> NetDescription nLab
buildOrdinaryRHS ifaceMap eqs = let
  endState = execState (mapM_ buildEquation eqs) $ BuildState
    { nodeCount = 0
    , nodeInfo = Map.empty
    , varMap = ifaceMap
    }
  varMap' = varMap endState
  mkIface msg isIfacePTD = let
         ps = sortBy (compare `on` snd) $ filter (isIfacePTD . snd) $ Map.toList ifaceMap
         f (v, ptd0) = case Map.lookup v varMap' of
           Just ptd' -> if ptd' == ptd0
             then error $ unlines $
               unwords  ["\nbuildOrdinaryRHS: Unconnected", msg, "variable:", show v, show ptd0]
               : show ifaceMap : map show eqs
             else ptd'
           Nothing -> error $ unlines $
               unwords  ["\nbuildOrdinaryRHS: Unknown", msg, "variable:", show v, show ptd0]
               : show ifaceMap : map show eqs
       in V.fromList $ map f ps
  in NetDescription
    { source = mkIface "source" isSourcePTD
    , target = mkIface "target" isTargetPTD
    , nodes = buildStateExtractNodeDescrs endState
    }
\end{code}

\begin{code}
buildStateExtractNodeDescrs  :: (Ord var, Show var, Show nLab)
                             => BuildState nLab var -> Vector (NodeDescription nLab)
buildStateExtractNodeDescrs endState = let
  varMap' = varMap endState
  mkNode ni = case Map.lookup ni $ nodeInfo endState of
    Nothing -> error $ unwords ["buildStateExtractNodeDescrs: Undefined node", show ni]
    Just (nlab, (Nothing, auxInfo)) -> error $
      unwords ["buildStateExtractNodeDescrs: Undefined principal port for node ", show ni, show nlab, show auxInfo]
      -- \edcomm{WK}{This shows up in |ArithExpression1| with equations of shape |Add(s,s2)~s1|.}
    Just (nlab, (Just vptd, auxInfo)) -> let
        mkPortDescr (Left v) = case Map.lookup v varMap' of
          Nothing -> error $
            unwords ["buildOrdinaryRHS: Undefined variable", show v, " in node ", show ni, show nlab, show (vptd : auxInfo)]
          Just ptd -> ptd
        mkPortDescr (Right ptd) = ptd
      in NodeDescription
        { nLab = nlab
        , portDescriptions = V.fromList $ map mkPortDescr (vptd : auxInfo)
        }
  in V.fromList $ map mkNode [0 .. nodeCount endState - 1]
\end{code}

\begin{code}
buildOrdinaryRHS'  :: (Ord var, Show var, Show nLab)
                   => [var] -> [var] -> [PTermEq nLab var] -> NetDescription nLab
buildOrdinaryRHS' sourceVars targetVars eqs = buildOrdinaryRHS vm eqs
  where
    vm = Map.fromList $ f SourcePort sourceVars ++ f TargetPort targetVars
    f mkPTD = zipWith (\ i v -> (v, mkPTD i)) [1..]
\end{code}

\begin{code}
buildNet  :: (Ord var, Show var, Show nLab)
          => [PTermEq nLab var] -> ([var], NetDescription nLab)
buildNet eqs = (,) sourceVars $ buildOrdinaryRHS' sourceVars [] eqs
  where
    sourceVars = removeDuplicates $ foldr (\ (t1, t2) vs -> connVars t1 ++ connVars t2 ++ vs) [] eqs
\end{code}

Lafont-style presentation of interaction net rules
uses a single pair of non-variable |PTerm|s,
has their roots as LHS, and the arguments of the roots
connect to the corresponding interface ports of the redex.

\begin{code}
buildLafontRule  :: (Ord var, Show var, Show nLab)
                 => PTermEq nLab var -> ((nLab, nLab), NetDescription nLab)
buildLafontRule (PNode nlab1 pts1, PNode nlab2 pts2) = (,) (nlab1, nlab2) $ NetDescription
    { source = mkIFace src
    , target = mkIFace trg
    , nodes = buildStateExtractNodeDescrs endState
    }
  where
    f mkPTD = zipWith (\ i t -> buildPTermInto t (Just $ Right $ mkPTD i)) [1..]
    mkIFace = V.fromList . map (either (varMap' Map.!) id)
    varMap' = varMap endState
    ((src, trg), endState) = runState (do
        src0 <- sequence $ f SourcePort pts1
        trg0 <- sequence $ f TargetPort pts2
        return (src0, trg0)
      ) $ BuildState
      { nodeCount = 0
      , nodeInfo = Map.empty
      , varMap = Map.empty
      }
buildLafontRule eq = error $ "buildLafontRule: Illegal argument: " ++ show eq
\end{code}


%{{{ natFromPTerm
For extracting a natural number from a net producing unary natural numbers
created from |S| and |Z| constructors,
returning |Int| should normally be sufficient.
However, it is at least theoretically possible to
be successfully extracting results from nets that can never have been
completely in memory at any one time.
For example, one might have long-running nets on 32-bit platforms
where |natFromPTerm| consumes more than $2^{32}$ top-level |S| constructors
before the |Z| is finally even created.
Therefore we still return |Integer| here.
(We use pattern guards to ensure that the node label is inspected before the successor list.)

\begin{code}
natFromPTerm :: (nLab -> String) -> (nLab -> Bool) -> (nLab -> Bool) -> PTerm nLab var -> Either String Integer
natFromPTerm showNLab isZ isS = h 0 where
  err i s = Left $ "natFromPTerm " ++ shows i (": " ++ s)
  h i (PNode nL ts) | isZ nL , [] <- ts   =  Right i
  h i (PNode nL ts) | isS nL , [t] <- ts  =  h (succ i) t
  h i (PNode nL ts) = err i $  "encountered " ++ showNLab nL ++
                               " with " ++ shows (length ts) " successors"
  h i (ConnVar _) = err i $ "encountered ConnVar!"
\end{code}
%}}}


% Local Variables:
% folded-file: t
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% fold-internal-margins: 0
% end:
