%%%  src/INet/Rule.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Rules}

For the actual execution of interaction net reduction,
the rules are ``contained'' in the function constituents of an |INetLang| record
(defined in |INet.Description|, \sectref{INet.Description}).

Here we present a datatype for rules together with a number of utility functions
for checking rules, and for converting rule sets into the |ruleRHS| function
required for |INetLang|.

\begin{ModuleHead}
\begin{code}
module INet.Rule where

import INet.Description
import INet.Description.Check
import INet.Description.Flip
import INet.Utils.List (groupByOnFst)
import INet.Utils.Map (mkLookup2)
import qualified INet.Utils.Vector as V

import Data.List
import Data.Function (on)

import Data.Map.Strict (Map)
import qualified  Data.Map.Strict as Map

import Control.Arrow (second)
import Control.Monad (when)

-- |import Debug.Trace|
\end{code}
\end{ModuleHead}

\begin{code}
data Rule nLab = Rule
  { lhs :: {-# UNPACK #-} ! (nLab, nLab)
  , rhs :: {-# UNPACK #-} ! (NetDescription nLab)
  }
\end{code}

\begin{code}
showLHS :: (Show nLab) => Rule nLab -> String
showLHS r = unwords [show . fst $ lhs r, show . snd $ lhs r]
\end{code}

\edcomm{WK}{No way to check that |source| and |target| are the tails?}

\begin{code}
checkRule :: (Show nLab) => (nLab -> (PI, PI)) -> Rule nLab -> IO ()
checkRule arity r = let
    header = showLHS r
    ptdMismatches = checkNetDescription $ rhs r
  in do
    putStrLn header
    let  srcArity = second pred . arity . fst $ lhs r
         srcBounds = V.bounds . source $ rhs r
      in when (srcArity /= srcBounds)
            $ putStrLn $ "source arity: " ++ shows srcArity "\t    source bounds: " ++ show srcBounds
    let  trgArity = second pred . arity . snd $ lhs r
         trgBounds = V.bounds . target $ rhs r
      in when (trgArity /= trgBounds)
            $ putStrLn $ "target arity: " ++ shows trgArity "\t    target bounds: " ++ show trgBounds
    case ptdMismatches of
      [] -> return ()
      _ -> putStrLn . unlines $ map show ptdMismatches
\end{code}

\begin{code}
fromRule :: Rule nLab -> ((nLab, nLab), NetDescription nLab)
fromRule r = (lhs r, rhs r)

fromRule' :: Rule nLab -> (nLab, (nLab, NetDescription nLab))
fromRule' r = case lhs r of
  (f,c) -> (f, (c, rhs r))

flipRule :: Rule nLab -> Rule nLab
flipRule (Rule (lSrc, lTrg) r) = Rule (lTrg, lSrc) $ flipNetDescription r

type RuleMap nLab = Map (nLab, nLab) (NetDescription nLab)

type Rules nLab = (nLab, nLab) -> NetDescription nLab
\end{code}

\begin{code}
ruleLHS :: Rule nLab -> NetDescription nLab
ruleLHS (Rule (labL, labR) (NetDescription src trg nodes)) = NetDescription
  { source  = V.generate (V.length src) (InternalPort 0 . succ)
  , target  = V.generate (V.length trg) (InternalPort 1 . succ)
  , nodes   = V.fromList
      [NodeDescription labL $ V.cons (InternalPort 1 0)
                            $ V.generate (V.length src) (SourcePort . succ)
      ,NodeDescription labR $ V.cons (InternalPort 0 0)
                            $ V.generate (V.length trg) (TargetPort . succ)
      ]
  }
\end{code}

\begin{code}
mkRulesFC :: (Ord nLab, Show nLab) => [Rule nLab] -> nLab -> nLab -> NetDescription nLab
mkRulesFC rs = mkLookup2 (map fromRule rs)
\end{code}



\begin{code}
mkRulesFC' :: (Ord nLab, Show nLab) => [Rule nLab] -> nLab -> nLab -> NetDescription nLab
mkRulesFC' rs = let
    rs' = groupByOnFst (==) . sortBy (compare `on` second fst) $ map fromRule' rs
    h (f, ps) =  ( f
                 , let m = Map.fromList ps
                   in (Map.!) m   -- |findRule m (\ c -> show (f, c))|
                 )
    m = Map.fromList $ map h rs'
  in (Map.!) m   -- |findRule m show|
  where
    findRule m descr label =  case Map.lookup label m of
                                Nothing -> error $ "No rule for " ++ descr label
                                Just d -> d
\end{code}


\begin{code}
mkRulesM :: (Ord nLab, Show nLab) => [Rule nLab] -> (nLab, nLab) -> Maybe (NetDescription nLab)
mkRulesM rs = let
    rs' = map flipRule rs
    ruleMap = Map.fromList $ map fromRule (rs' ++ rs)
  in \ p -> Map.lookup p ruleMap

mkRules :: (Ord nLab, Show nLab) => [Rule nLab] -> Rules nLab
mkRules rs p = case mkRulesM rs p of
        Nothing -> error $ "No rule for " ++ show p
        Just rhs -> rhs
\end{code}



\begin{code}


\end{code}
