%%%  src/INet/Simple/Rules.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
\section{Example Rules for Simple Nets}

\begin{code}
{-# LANGUAGE TypeSynonymInstances, FlexibleInstances #-}
module INet.Simple.Rules where

import INet.Polarity (Polarity(..))
import INet.Description
import INet.Description.Dot
import INet.Rule
import INet.Simple

import INet.Utils.Vector  (Vector, (!))
import qualified INet.Utils.Vector as V

import Data.Map (Map)
import qualified Data.Map as Map
-- |import Data.Set (Set)|
import qualified Data.Set as Set
import Control.Arrow (first)

-- |import Debug.Trace|
\end{code}

%{{{ proj1_pair_redex
\begin{code}
proj1_pair_redex :: NetDescription NLab
proj1_pair_redex = NetDescription
    { source = V.fromList [InternalPort 0 1]
    , target = V.fromList [InternalPort 1 1, InternalPort 1 2]
    , nodes = V.fromList
        [NodeDescription (NLab "proj₁") (V.fromList [InternalPort 1 0, SourcePort 0])
        ,NodeDescription (NLab "pair") (V.fromList [InternalPort 0 0, TargetPort 0, TargetPort 1])
        ]
    }
\end{code}
%}}}

%{{{ lastPorts, lastPortMap, lastPort, arity, nLabs
\begin{code}
lastPorts :: [(NLab, PI)]
lastPorts = map (first NLab)
  [("!", 0)  -- needs to be first for |∇|-rules
  ,("∇", 2)
  ,("pair", 2)
  ,("proj₁", 1)
  ,("proj₂", 1)
  ,("Z", 0)
  ,("S", 1)
  ,("+", 2)
  ,("*", 2)
  ,("fact", 1)
  ,("fib", 1)
  ,("fib'", 1)
  ,("ack", 2)
  ,("ack'", 2)
  ,("λc", 2)
  ,("λ", 3)
  ,("λb", 3)
  ,("@", 2)
  ,("b", 3)
  ,("δb", 3)
  ,("v", 0)
  ,("d", 0)
  ,("▵", 2)
  ]

lastPortMap :: Map NLab PI
lastPortMap = Map.fromList lastPorts

lastPort :: NLab -> PI
lastPort = (lastPortMap Map.!)

arity nLab = (0, lastPort nLab)

nLabs = map fst lastPorts
\end{code}
%}}}

%{{{ data PortLayout, pl*, plPolarity, plPolV
\begin{code}
data PortLayout = PortLayout
  { resPorts :: ! [PI]
  , argPorts :: ! [PI]
  }

plLength (PortLayout xs ys) = length xs + length ys

plC0 = PortLayout [0] []
plC1 = PortLayout [0] [1]
plC2 = PortLayout [0] [1,2]
plC n = PortLayout [0] [1 .. n]
plF1 = PortLayout [1] [0]
plF2a = PortLayout [2] [0,1]
plF2b = PortLayout [2] [1,0]

plPolarity pl i = if  i `elem` resPorts pl then Pos
  else if i `elem` argPorts pl then Neg
  else error $ "plPolarity " ++ show i

plPolV :: PortLayout -> Vector Polarity
plPolV pl = V.generate (plLength pl) (plPolarity pl)
\end{code}
%}}}

%{{{ plNLabMap, plNLab, portPolarity
\begin{code}
plNLabMap :: Map NLab PortLayout
plNLabMap = Map.fromList $ map (first NLab)
  [("!", PortLayout [] [0])
  ,("∇", PortLayout [1,2] [0])
  ,("pair", plC2)
  ,("proj₁",plF1)
  ,("proj₂",plF1)
  ,("S", plC1)
  ,("Z", plC0)
  ,("fact", plF1)
  ,("fib",  plF1)
  ,("fib'", plF1)
  ,("+", plF2a)
  ,("*", plF2a)
  ,("ack",  plF2a)
  ,("ack'", plF2b)
  ,("λc",  PortLayout [2,0] [1])  -- 2:BV is Pos
  ,("λ",  PortLayout [2,0] [3,1])  -- 3:fv-link is Neg, 2:BV is Pos
  ,("λb",  PortLayout [2,3] [0,1])  -- 0:fv-link is Neg, 2:BV is Pos, 3:result
  ,("@", plF2a)
  ,("b", PortLayout [2,1] [0,3])  -- fv passes from right (3:Neg) to left (2:Pos)
  ,("δb",  PortLayout [2,1] [0,3])  -- like b
  ,("v", plC0)
  ,("d", PortLayout [] [0])
  ,("▵", PortLayout [0] [1,2])
  ]
plNLab :: NLab -> PortLayout
plNLab = (plNLabMap Map.!)

polMap :: Map NLab (Vector Polarity)
polMap = Map.map plPolV plNLabMap

portPolarity :: NLab -> Vector Polarity
portPolarity = (polMap Map.!)
\end{code}
%}}}

%{{{ portName, defaultDotLabel'
\begin{code}
portName i = 'p' : show i

defaultDotLabel' nl = defaultDotLabel portName nl (0, lastPort nl')
  where nl' = NLab nl
\end{code}
%}}}

\edcomm{WK}{|dotNLabMap| duplicates |plNLabMap| information!}

%{{{ dotNLabMap, dotNLab
\begin{code}
dotNLabMap :: Map NLab String
dotNLabMap = Map.fromList $ map (first NLab)
  [("!", defaultDotLabel2 portName "!" [] [0])
  ,("∇", defaultDotLabel2 portName "dup" [1,2] [0])
  ,("pair", defaultDotLabel' "pair")
  ,("proj₁",defaultDotLabel1 portName "proj1" [1,0])
  ,("proj₂",defaultDotLabel1 portName "proj2" [1,0])
  ,("S", defaultDotLabel' "S")
  ,("Z", defaultDotLabel' "Z")
  ,("fact",defaultDotLabel1 portName "fact" [1,0])
  ,("fib",defaultDotLabel1 portName "fib" [1,0])
  ,("fib'",defaultDotLabel1 portName "fib'" [1,0])
  ,("+",defaultDotLabel1 portName  "+" [2,0,1])
  ,("*",defaultDotLabel1 portName "*" [2,0,1])
  ,("ack",defaultDotLabel1 portName "ack" [2,0,1])
  ,("ack'",defaultDotLabel1 portName "ack'" [2,1,0])
  ,("λc", defaultDotLabel2 portName "lambdaC" [2,0] [1])
  ,("λ", defaultDotLabel2 portName "lambda" [2,0] [3,1])
  ,("λb", defaultDotLabel2 portName "lambdaB" [2,3] [0,1])
  ,("@", defaultDotLabel1 portName  "@" [2,0,1])
  ,("b", defaultDotLabel2 portName "b" [2,1] [0,3])
  ,("δb", defaultDotLabel2 portName "b" [2,1] [0,3])
  ,("v", defaultDotLabel' "v")
  ,("d", defaultDotLabel2 portName "d" [] [0])
  ,("▵", defaultDotLabel2 portName "join" [0] [1,2])
  ]

dotNLab :: NLab -> Maybe String
dotNLab = flip Map.lookup dotNLabMap
\end{code}
%}}}

\begin{code}
-- |instance HasDot (NetDescription NLab) where|
-- |  dotGraph = dotNetDescription show dotNLab|
\end{code}

%{{{ mkNodeDescr :: NLab -> [PortTargetDescription] -> NodeDescription NLab
\begin{code}
mkNodeDescr :: NLab -> [PortTargetDescription] -> NodeDescription NLab
mkNodeDescr nLab ptds = NodeDescription nLab $ V.fromList ptds
\end{code}
%}}}

%{{{ mkIFace :: [PortTargetDescription] -> Vector (PortTargetDescription)
\begin{code}
mkIFace :: [PortTargetDescription] -> Vector (PortTargetDescription)
mkIFace ptds = V.fromList ptds -- previously started from 1!
\end{code}
%}}}

%{{{ mkNodes :: [NodeDescription NLab] -> Vector (NodeDescription NLab)
\begin{code}
mkNodes :: [NodeDescription NLab] -> Vector (NodeDescription NLab)
mkNodes nds = V.fromList nds
\end{code}
%}}}

%{{{ mkNat :: PortTargetDescription -> Int -> Int -> [NodeDescription NLab]
\begin{code}
mkNat :: PortTargetDescription -> Int -> Int -> [NodeDescription NLab]
mkNat caller start 0 = [mkNodeDescr (NLab "Z") [caller]]
mkNat caller start k = mkNodeDescr (NLab "S") [caller, InternalPort start' 0] : mkNat (InternalPort start 1) start' k'
  where  k' = pred k
         start' = succ start
\end{code}
%}}}

%{{{ fact :: Int -> NetDescription NLab
\begin{code}
fact :: Int -> NetDescription NLab
fact n = NetDescription
  { source = mkIFace [InternalPort 0 1]
  , target = mkIFace []
  , nodes = mkNodes $ mkNodeDescr (NLab "fact") [InternalPort 1 0, SourcePort 1] : mkNat (InternalPort 0 0) 1 n
  }
\end{code}
%}}}

%{{{ fibND :: Int -> NetDescription NLab
\begin{code}
fibND :: Int -> NetDescription NLab
fibND n = NetDescription
  { source = mkIFace [InternalPort 0 1]
  , target = mkIFace []
  , nodes = mkNodes $ mkNodeDescr (NLab "fib") [InternalPort 1 0, SourcePort 1] : mkNat (InternalPort 0 0) 1 n
  }
\end{code}
%}}}

%{{{ ackND :: Int -> Int -> NetDescription NLab
\begin{code}
ackND :: Int -> Int -> NetDescription NLab
ackND m n = NetDescription
  { source = mkIFace [InternalPort 0 2]
  , target = mkIFace []
  , nodes = mkNodes $ mkNodeDescr (NLab "ack") [InternalPort 1 0, InternalPort (2 + m) 0, SourcePort 1] : mkNat (InternalPort 0 0) 1 m ++ mkNat (InternalPort 0 1) (2 + m) n
  }
\end{code}
%}}}

%{{{ rules :: [Rule NLab]
\begin{code}
rules :: [Rule NLab]
rules =
  [ proj1_pair, proj2_pair
  , plus_Z, plus_S
  , mult_Z , mult_S
  , fact_Z, fact_S
  , fib_Z, fib_S
  , fib'_Z, fib'_S
  , ack_Z, ack_S, ack'_Z, ack'_S
  , apply_lambdaC, apply_lambda, d_v, b_lambdaC, b_lambda
  , dup_join, dup_lambdaC, dup_lambda, deltaB_v, lambdaB_v
  ]
  ++ map bang ({- drop 2 -} nLabs)
  ++ map nabla (filter ((`notElem` ["λc", "λ", "▵"]) . unNLab) $ drop 2 nLabs)
\end{code}
\edcomm{WK}{|nabla| needs to be careful with polarity!}
%}}}

%{{{ simpleLang
\begin{code}
simpleLang = INetLang
  { polarity = portPolarity
  , ruleRHS = mkRulesFC rules
  }
\end{code}
%}}}

%{{{ bang :: NLab -> Rule NLab
|() = ! p0|

\begin{code}
bang :: NLab -> Rule NLab
bang nl = Rule
  { lhs = (NLab "!", nl)
  , rhs = NetDescription
    { source = mkIFace []
    , target = mkIFace $ map (\ n -> InternalPort n 0) is
    , nodes = V.fromList -- previously started from 1!
        $ map (\ i -> mkNodeDescr (NLab "!") [TargetPort $ succ i]) is
    }
  }
  where
    k = pred $ lastPort nl
    is = [0 .. k] -- previously started from 1!
\end{code}
%}}}

%{{{ nabla :: NLab -> Rule NLab
|p1 p2 = ∇ p0|

\begin{code}
nabla :: NLab -> Rule NLab
nabla nl = Rule
  { lhs = (NLab "∇", nl)
  , rhs = NetDescription
    { source = mkIFace $ [InternalPort k1 0, InternalPort k2 0]
    , target = mkIFace $ map (\ n -> InternalPort (pred n) 0) is
    , nodes = V.fromList -- previously started from 1!
        $ map (\ i -> mkNodeDescr (NLab "∇") [TargetPort i, InternalPort k1 i, InternalPort k2 i]) is
        ++ [mkNodeDescr nl $ SourcePort 1 : map (\ n -> InternalPort (pred n) 1) is
           ,mkNodeDescr nl $ SourcePort 2 : map (\ n -> InternalPort (pred n) 2) is
           ]
    }
  }
  where
    k = lastPort nl
    is = [1 .. k] -- previously started from 1!
    k1 = k
    k2 = succ k
\end{code}
%}}}

|p0 = pair p1 p2|

%{{{ proj1_pair :: Rule NLab
|p1 = proj₁ p0|

\begin{code}
proj1_pair :: Rule NLab
proj1_pair = Rule
  { lhs = (NLab "proj₁", NLab "pair")
  , rhs = NetDescription
    { source = mkIFace [TargetPort 1]
    , target = mkIFace [SourcePort 1, InternalPort 0 0]
    , nodes  = mkNodes [mkNodeDescr (NLab "!") [TargetPort 2]]
    }
  }
\end{code}
%}}}

%{{{ proj2_pair
|p1 = proj₂ p0|

\begin{code}
proj2_pair :: Rule NLab
proj2_pair = Rule
  { lhs = (NLab "proj₂", NLab "pair")
  , rhs = NetDescription
    { source = mkIFace [TargetPort 2]
    , target = mkIFace [InternalPort 0 0, SourcePort 1]
    , nodes = mkNodes [mkNodeDescr (NLab "!") [TargetPort 1]]
    }
  }
\end{code}
%}}}

|p2 = p0 + p1|

%{{{ plus_Z :: Rule NLab
\begin{code}
plus_Z :: Rule NLab
plus_Z = Rule
  { lhs = (NLab "+", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace [SourcePort 2, SourcePort 1]
    , target = mkIFace []
    , nodes = mkNodes []
    }
  }
\end{code}
%}}}

|p0 = S p1|

%{{{ plus_S :: Rule NLab
\begin{code}
plus_S :: Rule NLab
plus_S = Rule
  { lhs = (NLab "+", NLab "S")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 1 1, InternalPort 0 0]
    , target = mkIFace [InternalPort 1 0]
    , nodes = mkNodes  [mkNodeDescr (NLab "S") [SourcePort 2, InternalPort 1 2]
                       ,mkNodeDescr (NLab "+") [TargetPort 1, SourcePort 1, InternalPort 0 1]
                       ]
    }
  }
\end{code}
%}}}

|p2 = p0 * p1|

%{{{ mult_Z :: Rule NLab
\begin{code}
mult_Z :: Rule NLab
mult_Z = Rule
  { lhs = (NLab "*", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 1 0, InternalPort 0 0]
    , target = mkIFace []
    , nodes = mkNodes  [mkNodeDescr (NLab "Z") [SourcePort 2]
                       ,mkNodeDescr (NLab "!") [SourcePort 1]
                       ]
    }
  }
\end{code}
%}}}

%{{{ mult_S :: Rule NLab
\begin{code}
mult_S :: Rule NLab
mult_S = Rule
  { lhs = (NLab "*", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 2 0, InternalPort 0 2]
    , target = mkIFace  [InternalPort 1 0]
    , nodes  = mkNodes  [mkNodeDescr (NLab "+") [InternalPort 2 1, InternalPort 1 2,SourcePort 2]
                        ,mkNodeDescr (NLab "*") [TargetPort 1, InternalPort 2 2,InternalPort 0 1]
                        ,mkNodeDescr (NLab "∇") [SourcePort 1, InternalPort 0 0,InternalPort 1 1]
                        ]
    }
  }
\end{code}
%}}}

|p1 = fact p0|

%{{{ fact_Z :: Rule NLab
\begin{code}
fact_Z :: Rule NLab
fact_Z = Rule
  { lhs = (NLab "fact", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 0]
    , target = mkIFace  []
    , nodes  = mkNodes  [mkNodeDescr (NLab "S") [SourcePort 1, InternalPort 1 0]
                        ,mkNodeDescr (NLab "Z") [InternalPort 0 1]
                        ]
    }
  }
\end{code}
%}}}

%{{{ fact_S :: Rule NLab
\begin{code}
fact_S :: Rule NLab
fact_S = Rule
  { lhs = (NLab "fact", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 2]
    , target = mkIFace  [InternalPort 3 0]
    , nodes  = mkNodes  [mkNodeDescr (NLab"*") [InternalPort 1 0, InternalPort 2 1,SourcePort 1]
                        ,mkNodeDescr (NLab "S") [InternalPort 0 0, InternalPort 3 1]
                        ,mkNodeDescr (NLab "fact") [InternalPort 3 2,InternalPort 0 1]
                        ,mkNodeDescr (NLab "∇") [TargetPort 1, InternalPort 1 1,InternalPort 2 0]
                        ]
    }
  }
\end{code}
%}}}

%{{{ fib, ffib :: Integer -> Integer
\begin{code}
fib 0 = 0
fib m = fib' n
  where n = pred m
fib' 0 = 1
fib' m = fib n + fib (n + 1) -- = fib n + fib' n
  where n = pred m
fibs = 0 : 1 : zipWith (+) fibs (tail fibs)

ffib 0 = 0
ffib m = ffib' n
  where n = pred m
ffib' 0 = 1
ffib' m = ffib'' n 0 1
  where n = pred m
ffib'' 0 k l = k + l
ffib'' m k l = ffib'' n l (k + l)
  where n = pred m
\end{code}
%}}}

%{{{ fib_Z :: Rule NLab
\begin{code}
fib_Z :: Rule NLab
fib_Z = Rule
  { lhs = (NLab "fib", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 0]
    , target = mkIFace  []
    , nodes  = mkNodes  [mkNodeDescr (NLab "Z") [SourcePort 1]
                        ]
    }
  }
\end{code}
%}}}

%{{{ fib_S :: Rule NLab
\begin{code}
fib_S :: Rule NLab
fib_S = Rule
  { lhs = (NLab "fib", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 1]
    , target = mkIFace  [InternalPort 0 0]
    , nodes  = mkNodes  [mkNodeDescr (NLab"fib'") [TargetPort 1,SourcePort 1]]
    }
  }
\end{code}
%}}}

%{{{ fib'_Z :: Rule NLab
\begin{code}
fib'_Z :: Rule NLab
fib'_Z = Rule
  { lhs = (NLab "fib'", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 0]
    , target = mkIFace  []
    , nodes  = mkNodes  [mkNodeDescr (NLab "S") [SourcePort 1, InternalPort 1 0]
                        ,mkNodeDescr (NLab "Z") [InternalPort 0 1]
                        ]
    }
  }
\end{code}
%}}}

%{{{ fib'_S :: Rule NLab
\begin{code}
fib'_S :: Rule NLab
fib'_S = Rule
  { lhs = (NLab "fib'", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 2]
    , target = mkIFace  [InternalPort 3 0]
    , nodes  = mkNodes  [mkNodeDescr (NLab"+") [InternalPort 1 1, InternalPort 2 1,SourcePort 1]
                        ,mkNodeDescr (NLab "fib") [InternalPort 3 1, InternalPort 0 0]
                        ,mkNodeDescr (NLab "fib'") [InternalPort 3 2,InternalPort 0 1]
                        ,mkNodeDescr (NLab "∇") [TargetPort 1, InternalPort 1 0,InternalPort 2 0]
                        ]
    }
  }
\end{code}
%}}}

|p2 = ack p0 p1|

%{{{ ack :: Integer -> Integer -> Integer
\begin{code}
ack 0 n = succ n
ack m n = ack' (pred m) n                   -- |ack (S m) n = ack' m n|
ack' m 0 = ack m 1
ack' m n = ack m (ack (succ m) (pred n))    -- |ack' m (S n) = ack m (ack (S m) n)|
\end{code}
%}}}

%{{{ ack_Z :: Rule NLab
\begin{code}
ack_Z :: Rule NLab
ack_Z = Rule
  { lhs = (NLab "ack", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 1, InternalPort 0 0]
    , target = mkIFace  []
    , nodes  = mkNodes  [mkNodeDescr (NLab "S") [SourcePort 2, SourcePort 1]]
    }
  }
\end{code}
%}}}

%{{{ ack_S :: Rule NLab
\begin{code}
ack_S :: Rule NLab
ack_S = Rule
  { lhs = (NLab "ack", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 0, InternalPort 0 2]
    , target = mkIFace  [InternalPort 0 1]
    , nodes  = mkNodes  [mkNodeDescr (NLab "ack'") [SourcePort 1, TargetPort 1, SourcePort 2]]
    }
  }
\end{code}
%}}}

|p2 = ack' p1 p0|

%{{{ ack'_Z :: Rule NLab
\begin{code}
ack'_Z :: Rule NLab
ack'_Z = Rule
  { lhs = (NLab "ack'", NLab "Z")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 0 0, InternalPort 0 2]
    , target = mkIFace  []
    , nodes  = mkNodes  $ mkNodeDescr (NLab "ack") [SourcePort 1, InternalPort 1 0, SourcePort 2]
                        : mkNat (InternalPort 0 1) 1 1
    }
  }
\end{code}
%}}}

%{{{ ack'_S :: Rule NLab
\begin{code}
ack'_S :: Rule NLab
ack'_S = Rule
  { lhs = (NLab "ack'", NLab "S")
  , rhs = NetDescription
    { source = mkIFace  [InternalPort 3 0, InternalPort 0 2]                                  -- |m|, result
    , target = mkIFace  [InternalPort 1 1]                                                    -- |n|
    , nodes  = mkNodes  [mkNodeDescr (NLab "ack") [InternalPort 3 1, InternalPort 1 2,SourcePort 2]
                        ,mkNodeDescr (NLab "ack") [InternalPort 2 0, TargetPort 1,InternalPort 0 1]
                        ,mkNodeDescr (NLab "S") [InternalPort 1 0, InternalPort 3 2]
                        ,mkNodeDescr (NLab "∇") [SourcePort 1, InternalPort 0 0,InternalPort 2 1]
                        ]
    }
  }
\end{code}
%}}}

\subsection{Result Extraction}

%{{{ checkNat :: NetDescription NLab -> Either String Int
\begin{code}
checkNat :: NetDescription NLab -> Either String Int
checkNat nd = case V.toList $ source nd of
  [InternalPort n 0] -> let
         h vis m = if Set.member m vis
            then Left $ "Cycle at " ++ show m
            else case nodes nd ! m of
                  NodeDescription (NLab "Z") _ -> Right 0
                  NodeDescription (NLab "S") sna -> case V.toList sna of
                    [_, InternalPort m' 0] -> fmap succ $ h (Set.insert m vis) m'
                    ptds -> Left $ "node " ++ shows m " interface: " ++ show ptds
                  NodeDescription nl _ -> Left $ "node " ++ shows m ": " ++ show nl
    in h Set.empty n
  ptds -> Left $ "source interface: " ++ show ptds
\end{code}
%}}}


\subsection{KCLE according to \citet{Mackie-2004}}

%{{{ apply_lambdaC :: Rule NLab
\begin{code}
apply_lambdaC :: Rule NLab
apply_lambdaC = Rule
  { lhs = (NLab "@", NLab "λc")
  , rhs = NetDescription
    { source = mkIFace [TargetPort 2, TargetPort 1]
    , target = mkIFace [SourcePort 2, SourcePort 1]
    , nodes = mkNodes []
    }
  }
\end{code}
%}}}

%{{{ apply_lambda :: Rule NLab
\begin{code}
apply_lambda :: Rule NLab
apply_lambda = Rule
  { lhs = (NLab "@", NLab "λ")
  , rhs = NetDescription
    { source = mkIFace [TargetPort 2, TargetPort 1]
    , target = mkIFace [SourcePort 2, SourcePort 1, InternalPort 0 0]
    , nodes = mkNodes [mkNodeDescr (NLab "d") [TargetPort 3]]
    }
  }
\end{code}
%}}}

%{{{ d_v :: Rule NLab
\begin{code}
d_v :: Rule NLab
d_v = Rule
  { lhs = (NLab "d", NLab "v")
  , rhs = NetDescription
    { source = mkIFace []
    , target = mkIFace []
    , nodes = mkNodes []
    }
  }
\end{code}
%}}}

%{{{ b_lambdaC :: Rule NLab
\begin{code}
b_lambdaC :: Rule NLab
b_lambdaC = Rule
  { lhs = (NLab "b", NLab "λc")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 0, SourcePort 3, SourcePort 2]
    , target = mkIFace [InternalPort 0 1, InternalPort 0 2]
    , nodes = mkNodes [mkNodeDescr (NLab "λc") [SourcePort 1, TargetPort 1, TargetPort 2]]
    }
  }
\end{code}
%}}}

%{{{ b_lambda :: Rule NLab
\begin{code}
b_lambda :: Rule NLab
b_lambda = Rule
  { lhs = (NLab "b", NLab "λ")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 0, InternalPort 1 2, InternalPort 1 3]
    , target = mkIFace [InternalPort 0 2, InternalPort 0 1, InternalPort 1 0]
    , nodes = mkNodes [mkNodeDescr (NLab "λ") [SourcePort 1, TargetPort 2, TargetPort 1, InternalPort 1 1]
                      ,mkNodeDescr (NLab "δb") [TargetPort 3, InternalPort 0 3, SourcePort 2, SourcePort 3]
                      ]
    }
  }
\end{code}
%}}}

%{{{ dup_join :: Rule NLab
\begin{code}
dup_join :: Rule NLab
dup_join = Rule
  { lhs = (NLab "∇", NLab "▵")
  , rhs = NetDescription
    { source = mkIFace [TargetPort 1, TargetPort 2]
    , target = mkIFace [SourcePort 1, SourcePort 2]
    , nodes = mkNodes []
    }
  }
\end{code}
%}}}

%{{{ dup_lambdaC :: Rule NLab
\begin{code}
dup_lambdaC :: Rule NLab
dup_lambdaC = Rule
  { lhs = (NLab "∇", NLab "λc")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 0, InternalPort 1 0]
    , target = mkIFace [InternalPort 3 0, InternalPort 2 0]
    , nodes = mkNodes [mkNodeDescr (NLab "λc") [SourcePort 1, InternalPort 3 1, InternalPort 2 1]
                      ,mkNodeDescr (NLab "λc") [SourcePort 2, InternalPort 3 2, InternalPort 2 2]
                      ,mkNodeDescr (NLab "▵") [TargetPort 2, InternalPort 0 2, InternalPort 1 2]
                      ,mkNodeDescr (NLab "∇") [TargetPort 1, InternalPort 0 1, InternalPort 1 1]
                      ]
    }
  }
\end{code}
%}}}

%{{{ dup_lambda :: Rule NLab
\begin{code}
dup_lambda :: Rule NLab
dup_lambda = Rule
  { lhs = (NLab "∇", NLab "λ")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 1, InternalPort 0 2]
    , target = mkIFace [InternalPort 1 1, InternalPort 1 2, InternalPort 1 0]
    , nodes = mkNodes [mkNodeDescr (NLab "∇") [InternalPort 1 3, SourcePort 1, SourcePort 2]
                      ,mkNodeDescr (NLab "λb") [TargetPort 3, TargetPort 1, TargetPort 2, InternalPort 0 0]
                      ]
    }
  }
\end{code}
%}}}

%{{{ deltaB_v :: Rule NLab
\begin{code}
deltaB_v :: Rule NLab
deltaB_v = Rule
  { lhs = (NLab "δb", NLab "v")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 0, SourcePort 3, SourcePort 2]
    , target = mkIFace []
    , nodes = mkNodes [mkNodeDescr (NLab "v") [SourcePort 1]]
    }
  }
\end{code}
%}}}

%{{{ lambdaB_v :: Rule NLab
\begin{code}
lambdaB_v :: Rule NLab
lambdaB_v = Rule
  { lhs = (NLab "λb", NLab "v")
  , rhs = NetDescription
    { source = mkIFace [InternalPort 0 1, InternalPort 0 2, InternalPort 0 0]
    , target = mkIFace []
    , nodes = mkNodes [mkNodeDescr (NLab "λc") [SourcePort 3, SourcePort 1, SourcePort 2]
                      ]
    }
  }
\end{code}
%}}}


%{{{ EMACS lv
% Local Variables:
% folded-file: t
% fold-internal-margins: 0
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% end:
%}}}
