%%%  src/INet/Utils/Dot.lhs
%%%
%%%  Copyright ©  2015 Wolfram Kahl
%%%
%%%  This file is part of HINet.
%%%
%%%  HINet is free software: you can redistribute it and/or modify
%%%  it under the terms of the GNU General Public License as published by
%%%  the Free Software Foundation, in version 3 of the License.
%%%
%%%  HINet is distributed in the hope that it will be useful,
%%%  but WITHOUT ANY WARRANTY; without even the implied warranty of
%%%  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%%%  GNU General Public License version 3 for more details.
%%%
%%%  You should have received a copy of the GNU General Public License
%%%  along with HINet.  If not, see <http://www.gnu.org/licenses/>.
% Copied 2014-04-30 from svn/RATH/trunk/RATH-Utils/Data/Rel/Utils/Dot.lhs
%
\section{Dot Graphs}

\begin{code}
module INet.Utils.Dot where


import INet.Utils.GhostView (ghostViewFile)
import INet.Utils.TmpFile
import System.Process (system)
import System.FilePath (addExtension, splitExtension)
\end{code}

%{{{ Attrs
\begin{code}
type Attr = (String,String)

escape :: Char -> ShowS
escape '\n' = ('\\' :) . ('n' :)
escape '\t' = ('\\' :) . ('t' :)
escape '\r' = ('\\' :) . ('r' :)
escape '\b' = ('\\' :) . ('b' :)
escape '\f' = ('\\' :) . ('f' :)
escape '\\' = ('\\' :) . ('\\' :)
escape '"'  = ('\\' :) . ('"' :)
escape c    = (c :)

showsString s = ('"' :) . flip (foldr escape) s . ('"' :)

showsAttr (name,value) = (name ++) . ('=' :) . showsString value

showsAttrs [] = id
showsAttrs attrs = (' ' :) . ('[' :) .
                   sepShowsList (',' :) showsAttr attrs .
                   (']' :) . (';' :)

type Attrs = [Attr]
\end{code}
%}}}

%{{{ data Stmt, DotGraph
\begin{code}
data Stmt = Setting Attr
          | NodeSettings Attrs
          | EdgeSettings Attrs
          | Node String Attrs
          | Edge String String Attrs
          | UndirEdge String String Attrs
          | Edges [String] Attrs
          | Subgraph String [Stmt]

data DotGraphKind = Graph | Digraph deriving (Eq, Ord)
instance Show DotGraphKind where
  show Graph = "graph"
  show Digraph = "digraph"

data DotGraph = DotGraph DotGraphKind String [Stmt]
\end{code}
%}}}


%{{{ instance Show Stmt
\begin{code}
instance Show Stmt where
  showsPrec _ = showsStmt (' ' :)
  showList = showsStmts (' ' :)

showsStmt = showsStmt0 False

showsNeatoStmt = showsStmt0 True

showsStmt0 undir indent = f where
  f (Setting attr) = showsAttr attr . (';' :)
  f (NodeSettings attrs) = ("node" ++) . showsAttrs attrs
  f (EdgeSettings attrs) = ("edge" ++) . showsAttrs attrs
  f (Node name attrs) = showsString name . showsAttrs attrs
  f (UndirEdge src trg attrs) =
     if undir then showsEdge undir src trg attrs else id
  f (Edge src trg attrs) = showsEdge undir src trg attrs
  f (Edges ns attrs) = sepShowsList (arrow undir) showsString ns . showsAttrs attrs
  f (Subgraph name stmts) =
      ("subgraph " ++) . (name ++) . (" {\n" ++) .
      showsStmts0 undir ((' ':) . indent) stmts .
      indent . ('}':)

noconstraint = ("constraint","false")

showsEdge undir src trg attrs =
   ((++) src) . arrow undir . ((++) trg) . showsAttrs attrs
--   showsString src . arrow undir . showsString trg . showsAttrs attrs

arrow False = (" -> " ++)
arrow True  = (" -- " ++)

showsStmts = showsStmts0 False
showsNeatoStmts = showsStmts0 True

showsStmts0 undir indent [] = id
showsStmts0 undir indent (stmt:stmts) =
    indent . showsStmt0 undir indent stmt . ('\n' :) .
    showsStmts0 undir indent stmts
\end{code}
%}}}

%{{{ instance Show DotGraph
\begin{code}
showsDotGraph (DotGraph kind name stmts) =
    shows kind . (' ' :) . (name ++) . (" {\n" ++) .
    (case kind of
       Graph -> showsNeatoStmts
       Digraph -> showsStmts
     ) (' ':) stmts . ('}':) . ('\n':)

instance Show DotGraph where
  showsPrec _ = showsDotGraph

  showList [] = id
  showList (g:gs) = shows g . showList gs
\end{code}
%}}}

\begin{code}
sepShowsList sep shows [] = id
sepShowsList sep shows [x] = shows x
sepShowsList sep shows (x:xs) = shows x . sep . sepShowsList sep shows xs
\end{code}

%{{{ \subsection{Class |HasDot|}
\subsection{Class |HasDot|}

\begin{code}
class HasDot a where
  dotGraph :: String -> a -> DotGraph

dotGraphAddSettings :: [Attr] -> DotGraph -> DotGraph
dotGraphAddSettings ss (DotGraph k n ss') = DotGraph k n (map Setting ss ++ ss')

dotGraphWithSetting :: HasDot a => [Attr] -> String -> a -> DotGraph
dotGraphWithSetting ss name = dotGraphAddSettings ss . dotGraph name

dotString :: HasDot a => String -> a -> String
dotString name = show . dotGraph name

dotToFile :: HasDot a => FilePath -> String -> a -> IO ()
dotToFile = dotToFileWithSetting []

dotToFileWithSetting :: HasDot a => [Attr] -> FilePath -> String -> a -> IO ()
dotToFileWithSetting ss f name = writeFile f . show . dotGraphWithSetting ss name

dot :: HasDot a => a -> IO ()
dot = showDot . dotString "RATHRel"
\end{code}

\begin{code}
dotFileFormat :: String -> String -> FilePath -> IO ()
dotFileFormat format suffixOut fileIn = let
    (basename, suffixIn) = splitExtension fileIn
    fileOut =  (if suffixIn == ".dot" then basename else fileIn)
             `addExtension` suffixOut
  in do
    system $ unwords ["dot -T" ++ format, "-o", fileOut, fileIn]
    return ()

dotFilePS :: FilePath -> IO ()
dotFilePS = dotFileFormat "ps" ".eps"

dotGraphPS :: [Attr] -> DotGraph -> IO ()
dotGraphPS ss dg@(DotGraph kind name ss') = let fileDot = name ++ ".dot"
  in do
    writeFile fileDot . show $ dotGraphAddSettings ss dg
    dotFilePS fileDot
\end{code}

\begin{code}
showDot :: String -> IO ()
showDot s = do
  tmpFile <- writeTmp ".dot" s
  let (basename, suffix) = splitExtension tmpFile
      psfile =  (if suffix == ".dot" then basename else tmpFile)
                `addExtension` ".eps"
  system $ unwords ["dot -Tps ", tmpFile, ">", psfile]
  ghostViewFile psfile
\end{code}
%}}}

%{{{ \subsection{Generating Dot Graphs from Relations}
\subsection{Generating Dot Graphs from Relations}

For simple generation of dot graphs from relations,
we assume the node labels to be given as a list of strings,
and the edges as index pairs.
We also insert some useful default settings.

\begin{code}
dotOfSepPairs :: DotGraphKind -> String -> [String] -> [Int] -> [(Int,Int)] -> [(Int,Int)] -> DotGraph
dotOfSepPairs kind name labels loops syms arrows =
    DotGraph kind name . (settings ++) . (nodes ++) $ edges
  where
    mkSymEdge (x,y) = Edge (show x) (show y) [("dir","both")]
    symEdges = map mkSymEdge syms
    mkLoop x = Edge (show x) (show x) [("dir","none"),("tailclip","false"),("headclip","false ")]
    loopEdges = map mkLoop loops
    mkEdge (x,y) = Edge (show x) (show y) []
    mkNode i n = Node (show i) [("label",n)]
    nodes = zipWith mkNode [0..] labels
    edges = loopEdges ++ symEdges ++ map mkEdge arrows

settings =
  NodeSettings
    [("shape","plaintext"), ("height","0"), ("width","0")
    ,("fontsize","20")
    ]
  : map Setting [("nodesep","0.1")
                ,("nslimit","100"), ("mclimit","100")] -- make dot work harder
\end{code}

The first three attributes produce outline-free nodes
with as little free space around them as possible.
The choice of font size,
with otherwise standard settings,
makes arrows reasonably thin and short (relative to the nodes).

Since the generated dot file can be edited and dot run again,
and dot settings can also be supplied on the dot command-line,
the lack of possibility to influence the settings chosen here
should not be a big problem.
%}}}

%{{{ EMACS lv
% Local Variables:
% folded-file: t
% fold-internal-margins: 0
% eval: (fold-set-marks "%{{{ " "%}}}")
% eval: (fold-whole-buffer)
% end:
%}}}
